#!/usr/bin/perl

use Direct::Modern;

use Test::More tests => 15;

#use my_inc '../../../../', for => 'api/t';

use API::Authorization;
use API::Authorization::User;
use API::Service::Campaigns;

use Settings;
use Units::TestDummy;

use Yandex::DBUnitTest qw/ :all /;
use Yandex::Test::UTF8Builder;

{
    no strict qw/ refs /;
    no warnings qw/ once redefine /;

    *API::Authorization::User::get_info_by_uid_passport = sub ($)   { +{} };
    *API::Authorization::User::create_update_user       = sub ($;$) { };

    *API::Service::Base::rbac                         = sub { bless {} => 'RBAC2::Extended' };
    *API::Service::Base::operator_role                = sub { 'client' };
    *API::Service::Base::subclient_role               = sub { 'client' };
    *API::Service::Base::not_owned_campaign_ids_hash  = sub { () };
    *API::Service::Base::check_campaigns_write_detail = sub { () };
    *API::Service::Base::check_campaigns_delete       = sub { () };
    *API::Service::Base::units = sub { state $dummy_units = Units::TestDummy->new() };

    *Direct::Campaigns::rbac_delete_campaign          = sub {  };

    $Direct::Campaigns::SEND_ALERT_MAILS = 0;
}

my $TEST_DATA = {
    api_special_user_options => {
        original_db => PPC( shard => 'all' ),
    },
    bs_export_candidates => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 5, },
            ],
        },
    },
    bs_export_queue => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 6, },
            ],
        },
    },
    campaigns => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 2,  uid => 1, type => 'text',   statusEmpty => 'Yes', sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 3,  uid => 1, type => 'wallet', statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 4,  uid => 2, type => 'geo',    statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 5,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 6,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 7,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'Yes', currency => 'YND_FIXED', },
                { cid => 8,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '3.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 9,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '3.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 10, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '3.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 11, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 1, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 12, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 13, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 14, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                { cid => 15,  uid => 2, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
            ],
        },
    },
    camp_operations_queue => {
        original_db => PPC( shard => 'all' ),
    },
    camp_options => {
        original_db => PPC( shard => 'all' ),
        # TODO: как тестировать изменения в этой табличке?
        rows => {
            1 => [
                { cid => 1,  },
                { cid => 2,  },
                { cid => 3,  },
                { cid => 4,  },
                { cid => 5,  },
                { cid => 6,  },
                { cid => 7,  },
                { cid => 8,  },
                { cid => 9,  },
                { cid => 10, },
                { cid => 11, },
                { cid => 12, },
                { cid => 13, },
            ],
        },
    },
    campaigns_performance => {
        original_db => PPC( shard => 'all' ),
    },
    campaigns_cpm_yndx_frontpage => {
        original_db => PPC( shard => 'all' ),
    },
    subcampaigns => {
        original_db => PPC(shard => 'all'),
    },
    clients_api_options => {
        original_db => PPC( shard => 'all' ),
    },
    clients_options => {
        original_db => PPC( shard => 'all' ),
    },
    metrika_counters => {
        original_db => PPC( shard => 'all' ),
    },
    shard_client_id => {
        original_db => PPCDICT(),
        rows => [
            { ClientID => 1, shard => 1, },
            { ClientID => 2, shard => 1, }
        ],
    },
    shard_inc_cid => {
        original_db => PPCDICT(),
        rows => [
            { cid => 1,  ClientID => 1, },
            { cid => 2,  ClientID => 1, },
            { cid => 3,  ClientID => 1, },
            { cid => 4,  ClientID => 2, },
            { cid => 5,  ClientID => 1, },
            { cid => 6,  ClientID => 1, },
            { cid => 7,  ClientID => 1, },
            { cid => 8,  ClientID => 1, },
            { cid => 9,  ClientID => 1, },
            { cid => 10, ClientID => 1, },
            { cid => 11, ClientID => 1, },
            { cid => 12, ClientID => 1, },
            { cid => 13, ClientID => 1, },
        ],
    },
    shard_uid => {
        original_db => PPCDICT(),
        rows => [
            { uid => 1, ClientID => 1, },
            { uid => 2, ClientID => 2, },
        ],
    },
    ppc_properties => {
        original_db => PPCDICT()
    },
    users => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, login => 'unit-tester', statusBlocked => 'no', ClientID => 1, },
                { uid => 2, login => 'not-allowed-to-edit-geo-camp', statusBlocked => 'no', ClientID => 2, }
            ],
        },
    },
    users_api_options => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, api_geo_allowed => 'Yes', },
                { uid => 2, api_geo_allowed => 'No', }
            ],
        },
    },
    users_options => {
        original_db => PPC( shard => 'all' ),
    },
};

init_test_dataset( $TEST_DATA );


my $login = 'unit-tester';
my $uid   = 1;
my $LIMIT = $API::Service::Campaigns::DELETE_IDS_LIMIT;

my $user = API::Authorization::User->new(
    uid      => $uid,
    login    => $login,
    ClientID => 1,
);

my $service = API::Service::Campaigns->new('Campaigns');
$service->set_current_operation('delete');
$service->set_subclient( $user );
$service->set_authorization( API::Authorization->fake_new( $uid, $login, { chief_rep_user => $user, karma => 0 } ) );


my %RESPONSES = (
    invalid_id => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
        ],
    },
    not_owned_campaign => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            }
        ],
    },
    not_exists_camps => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    readonly_campaign => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 54,
                        Message => 'Нет прав',
                        Details => 'Нет прав на запись',
                    },
                ],
            }
        ],
    },
    undeletable_campaign => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 54,
                        Message => 'Нет прав',
                        Details => 'Нет прав на удаление',
                    },
                ],
            }
        ],
    },
    duplicates => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 9800,
                        Message => 'Объект не может присутствовать в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 9800,
                        Message => 'Объект не может присутствовать в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                ],
            },
        ],
    },
    bad_status_empty => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    bad_kind_campaign => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 3500,
                        Message => 'Не поддерживается',
                        Details => 'Тип кампании не поддерживается',
                    },
                ],
            },
        ],
    },
    bad_operator => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    bs_export => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8301,
                        Message => 'Невозможно удалить объект',
                        Details => 'Невозможно удалить кампанию',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 8301,
                        Message => 'Невозможно удалить объект',
                        Details => 'Невозможно удалить кампанию',
                    },
                ],
            },
        ],
    },
    currency_converted => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8301,
                        Message => 'Невозможно удалить объект',
                        Details => 'Невозможно удалить кампанию',
                    },
                ],
            },
        ],
    },
    with_money => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8301,
                        Message => 'Невозможно удалить объект',
                        Details => 'Запрещено удалять кампанию с деньгами, либо на которую выставлен счет на оплату',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 8301,
                        Message => 'Невозможно удалить объект',
                        Details => 'Запрещено удалять кампанию с деньгами, либо на которую выставлен счет на оплату',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 8301,
                        Message => 'Невозможно удалить объект',
                        Details => 'Запрещено удалять кампанию с деньгами, либо на которую выставлен счет на оплату',
                    },
                ],
            },
        ],
    },
    with_statistics => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 8301,
                        Message => 'Невозможно удалить объект',
                        Details => 'Запрещено удалять кампанию с деньгами, либо на которую выставлен счет на оплату',
                    },
                ],
            },
        ],
    },
    rbac_die => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 1002,
                        Message => 'Ошибка операции',
                    },
                ],
            },
        ],
    },
    rbac_error_code => {
        DeleteResults => [
            {
                Errors => [
                    {
                        Code    => 1002,
                        Message => 'Ошибка операции',
                    },
                ],
            },
        ],
    },
    delete_text_camp => {
        DeleteResults => [
            { Id => 12, },
        ],
    },
    delete_geo_camp => {
        DeleteResults => [
            { Id => 13, },
        ],
    },
);

my %SAVED_DATA = (
    not_changed => {
        campaigns => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 2,  uid => 1, type => 'text',   statusEmpty => 'Yes', sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 3,  uid => 1, type => 'wallet', statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 4,  uid => 2, type => 'geo',    statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 5,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 6,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 7,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'Yes', currency => 'YND_FIXED', },
                    { cid => 8,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '3.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 9,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '3.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 10, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '3.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 11, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 1, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 12, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 13, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 14, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 15,  uid => 2, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                ],
            },
        },
        camp_operations_queue => {
            original_db => PPC( shard => 'all' ),
        },
    },
    delete_text_camp => {
        campaigns => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 2,  uid => 1, type => 'text',   statusEmpty => 'Yes', sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 3,  uid => 1, type => 'wallet', statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 4,  uid => 2, type => 'geo',    statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 5,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 6,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 7,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'Yes', currency => 'YND_FIXED', },
                    { cid => 8,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '3.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 9,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '3.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 10, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '3.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 11, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 1, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 12, uid => 1, type => 'text',   statusEmpty => 'Yes', sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 13, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 14, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 15,  uid => 2, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },

                ],
            },
        },
        camp_operations_queue => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 12, operation => 'del', },
                ],
            },
        },
    },
    delete_geo_camp => {
        campaigns => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 2,  uid => 1, type => 'text',   statusEmpty => 'Yes', sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 3,  uid => 1, type => 'wallet', statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 4,  uid => 2, type => 'geo',    statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 5,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 6,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 7,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'Yes', currency => 'YND_FIXED', },
                    { cid => 8,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '3.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 9,  uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '3.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 10, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '3.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 11, uid => 1, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 1, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 12, uid => 1, type => 'text',   statusEmpty => 'Yes', sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 13, uid => 1, type => 'text',   statusEmpty => 'Yes', sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 14, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                    { cid => 15,  uid => 2, type => 'text',   statusEmpty => 'No',  sum => '0.000000', sum_to_pay => '0.000000', sum_last => '0.000000', OrderID => 0, currencyConverted => 'No',  currency => 'RUB', },
                ],
            },
        },
        camp_operations_queue => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 12, operation => 'del', },
                    { cid => 13, operation => 'del', },
                ],
            },
        },
    },
);

# use feature qw/say/;
# use DDP;
# use Test::Deep qw/ superhashof cmp_details deep_diag/;
# use Yandex::DBTools;
# my $expected = $SAVED_DATA{not_changed}->{campaigns}{rows}{1};
# my $rows = get_all_sql(SHUT_1, 'SELECT cid, uid, type, statusEmpty, sum, sum_to_pay, sum_last, OrderID, currencyConverted, currency from campaigns ORDER BY cid');
# p $rows;
# for ( my $i = 0; $i <= $#$rows; $i++ ) {
#     my ($ok, $stack) = cmp_details( $rows->[ $i ], superhashof( $expected->[ $i ] ) );
#     next if ok($ok);
#     say STDERR $i;
#     p deep_diag($stack);
# }
# die;

subtest 'Campaigns.Delete - request objects limit' => sub {
    plan tests => 5;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ map { $_ } 0 .. $LIMIT ], },
    });
    isa_ok($resp, 'Direct::Defect');
    is( $resp->code, 4001, 'error code' );
    is( $resp->text, 'Неверно заданы параметры фильтрации SelectionCriteria', 'error message' );
    is( $resp->description, "Массив Ids не может содержать более $LIMIT элементов", 'error description' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - errors for invalid ids' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 'abc', '-1' ], },
    });
    is_deeply( $resp, $RESPONSES{invalid_id}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - error for not owned campaign' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 15 ], },
    });
    is_deeply( $resp, $RESPONSES{not_owned_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - errors for not exist or hidden campaigns' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'n' } @_ } };

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 1 ], },
    });
    is_deeply( $resp, $RESPONSES{not_exists_camps}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    #done_testing();
};

subtest 'Campaigns.Delete - error for read-only campaign' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'r' } @_ } };

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 5 ], },
    });
    is_deeply( $resp, $RESPONSES{readonly_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    #done_testing();
};

subtest 'Campaigns.Delete - error for campaign without rights for delete' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin_check_campaigns_delete = \&API::Service::Base::check_campaigns_delete;
    *API::Service::Base::check_campaigns_delete = sub { shift; map { $_ => undef } @_ };

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 5 ], },
    });
    is_deeply( $resp, $RESPONSES{undeletable_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_delete = $origin_check_campaigns_delete;

    #done_testing();
};

subtest 'Campaigns.Delete - error for duplicates' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 1, 1, ], },
    });
    is_deeply( $resp, $RESPONSES{duplicates}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - error for draft campaign' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 2, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_status_empty}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - error for campaign with not "api5_edit" kind' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 3, 14 ], },
    });
    is_deeply( $resp, $RESPONSES{bad_kind_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

=for Включить после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Delete - error for geo campaign and operator, not allowed to edit geo camp with api' => sub {
    plan tests => 2;

    my $srv = API::Service::Campaigns->new('Campaigns');
    $srv->set_current_operation('delete');
    my $user = API::Authorization::User->new( uid => 2, login => 'not-allowed-to-edit-geo-camp', ClientID => 2 );
    $srv->set_subclient( $user );
    $srv->set_authorization( API::Authorization->fake_new( 2, 'not-allowed-to-edit-geo-camp', { chief_rep_user => $user, karma => 0 } ) );

    my $resp = $srv->delete({
        SelectionCriteria => { Ids => [ 4, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_operator}, 'expected response' );

    # TODO: check for opposite case

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};
=cut

subtest 'Campaigns.Delete - error for campaign in bs export' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 5, 6, ], },
    });
    is_deeply( $resp, $RESPONSES{bs_export}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - error for currency converted campaign' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 7, ], },
    });
    is_deeply( $resp, $RESPONSES{currency_converted}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - error for campaign with money' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 8, 9, 10, ], },
    });
    is_deeply( $resp, $RESPONSES{with_money}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - error for campaign with statistics' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 11, ], },
    });
    is_deeply( $resp, $RESPONSES{with_statistics}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Delete - rbac die or return error code' => sub {
    plan tests => 4;

    no warnings qw/ once redefine /;

    my $origin = \&Direct::Campaigns::rbac_delete_campaign;


    *Direct::Campaigns::rbac_delete_campaign = sub { die '`cause!'; };

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 1, ], },
    });
    is_deeply( $resp, $RESPONSES{rbac_die}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );


    *Direct::Campaigns::rbac_delete_campaign = sub { return 1; };

    $resp = $service->delete({
        SelectionCriteria => { Ids => [ 1, ], },
    });
    is_deeply( $resp, $RESPONSES{rbac_error_code}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );


    *Direct::Campaigns::rbac_delete_campaign = $origin;

    #done_testing();
};

subtest 'Campaigns.Delete - delete text and geo campaigns' => sub {
    plan tests => 4;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 12, ], },
    });
    is_deeply( $resp, $RESPONSES{delete_text_camp}, 'expected response' );

    check_test_dataset( $SAVED_DATA{delete_text_camp}, 'expected db state' );

    $resp = $service->delete({
        SelectionCriteria => { Ids => [ 13, ], },
    });
    is_deeply( $resp, $RESPONSES{delete_geo_camp}, 'expected response' );

    check_test_dataset( $SAVED_DATA{delete_geo_camp}, 'expected db state' );

    #done_testing();
};

=for TODO: дописать после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Resume - delete geo campaign' => sub {
    plan tests => 2;

    my $resp = $service->delete({
        SelectionCriteria => { Ids => [ 12, ], },
    });
    is_deeply( $resp, $RESPONSES{delete_geo}, 'expected response' );

    check_test_dataset( $SAVED_DATA{delete_geo}, 'expected db state' );

    #done_testing();
};
=cut

#done_testing();
