#!/usr/bin/perl

use Direct::Modern;

use Test::More tests => 11;

#use my_inc '../../../../', for => 'api/t';

use API::Authorization;
use API::Authorization::User;
use API::Service::Campaigns;
use Settings;
use Units::TestDummy;
use Yandex::DBUnitTest qw/ :all /;
use Yandex::Test::UTF8Builder;

{
    no strict qw/ refs /;
    no warnings qw/ once redefine /;

    *API::Authorization::User::get_info_by_uid_passport = sub ($)   { +{} };
    *API::Authorization::User::create_update_user       = sub ($;$) { };

    *API::Service::Base::rbac                         = sub { bless {} => 'RBAC2::Extended' };
    *API::Service::Base::operator_role                = sub { 'client' };
    *API::Service::Base::subclient_role               = sub { 'client' };
    *API::Service::Base::not_owned_campaign_ids_hash  = sub { () };
    *API::Service::Base::check_campaigns_write_detail = sub { () };
    *API::Service::Base::units = sub { state $dummy_units = Units::TestDummy->new() };

    *Direct::Campaigns::mass_mail_notification        = sub {};
    *Direct::Campaigns::log_cmd                       = sub {};
    *Client::ClientFeatures::has_access_to_new_feature_from_java = sub { return 0 };
}

my $TEST_DATA = {
    api_special_user_options => {
        original_db => PPC( shard => 'all' ),
    },
    campaigns => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 10, uid => 1, type => 'mcb',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 11, uid => 2, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 12, uid => 1, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 13, uid => 1, type => 'wallet', statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 14, uid => 1, type => 'text',   statusEmpty => 'Yes', statusShow => 'No',  archived => 'No',  statusBsSynced => 'No',  },
                { cid => 15, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'Yes', statusBsSynced => 'No',  },
                { cid => 16, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                { cid => 17, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 18, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 19, uid => 2, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                { cid => 20, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
            ],
        },
    },
    camp_options => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 1,  },
                { cid => 10, },
                { cid => 11, },
                { cid => 12, },
                { cid => 13, },
                { cid => 14, },
                { cid => 15, },
                { cid => 16, },
                { cid => 17, },
                { cid => 18, },
            ],
        },
    },
    campaigns_performance => {
        original_db => PPC( shard => 'all' ),
    },
    campaigns_cpm_yndx_frontpage => {
        original_db => PPC( shard => 'all' ),
    },
    subcampaigns => {
        original_db => PPC(shard => 'all'),
    },
    clients_api_options => {
        original_db => PPC( shard => 'all' ),
    },
    clients_options => {
        original_db => PPC( shard => 'all' ),
    },
    metrika_counters => {
        original_db => PPC( shard => 'all' ),
    },
    shard_client_id => {
        original_db => PPCDICT(),
        rows => [
            { ClientID => 1, shard => 1, },
            { ClientID => 2, shard => 1, }
        ],
    },
    shard_inc_cid => {
        original_db => PPCDICT(),
        rows => [
            { cid => 1,  ClientID => 1, },
            { cid => 10, ClientID => 1, },
            { cid => 11, ClientID => 2, },
            { cid => 12, ClientID => 1, },
            { cid => 13, ClientID => 1, },
            { cid => 14, ClientID => 1, },
            { cid => 15, ClientID => 1, },
            { cid => 16, ClientID => 1, },
            { cid => 17, ClientID => 1, },
            { cid => 18, ClientID => 1, },
            { cid => 19, ClientID => 2, },
            { cid => 20, ClientID => 1, },
        ],
    },
    shard_uid => {
        original_db => PPCDICT(),
        rows => [
            { uid => 1, ClientID => 1, },
            { uid => 2, ClientID => 2, },
        ],
    },
    ppc_properties => {
        original_db => PPCDICT()
    },
    users => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, login => 'unit-tester', statusBlocked => 'no', ClientID => 1, },
                { uid => 2, login => 'not-allowed-to-edit-geo-camp', statusBlocked => 'no', ClientID => 2, }
            ],
        },
    },
    users_api_options => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, api_geo_allowed => 'Yes', },
                { uid => 2, api_geo_allowed => 'No', }
            ],
        },
    },
    users_options => {
        original_db => PPC( shard => 'all' ),
    },
};

init_test_dataset( $TEST_DATA );


my $login = 'unit-tester';
my $uid   = 1;
my $LIMIT = $API::Service::Campaigns::RESUME_IDS_LIMIT;

my $user = API::Authorization::User->new(
    uid      => $uid,
    login    => $login,
    ClientID => 1,
);

my $service = API::Service::Campaigns->new('Campaigns');
$service->set_current_operation('resume');
$service->set_subclient( $user );
$service->set_authorization( API::Authorization->fake_new( $uid, $login, { chief_rep_user => $user, karma => 0 } ) );


my %RESPONSES = (
    invalid_id => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
        ],
    },
    not_owned_campaign => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            }
        ],
    },
    not_exists_camps => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    readonly_campaign => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 54,
                        Message => 'Нет прав',
                        Details => 'Нет прав на запись',
                    },
                ],
            }
        ],
    },
    bad_kind_campaign => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 3500,
                        Message => 'Не поддерживается',
                        Details => 'Тип кампании не поддерживается',
                    },
                ],
            },
        ],
    },
    bad_operator => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    bad_status_empty => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    bad_status_archived => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 8300,
                        Message => 'Неверный статус у объекта',
                        Details => 'Запрещено изменять заархивированную кампанию',
                    },
                ],
            },
        ],
    },
    draft_campaign => {
        ResumeResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    not_suspended => {
        ResumeResults => [
            {
                Id => 16,
                Warnings => [
                    {
                        Code    => 10021,
                        Message => 'Объект не остановлен',
                        Details => 'Кампания не остановлена',
                    },
                ],
            },
        ],
    },
    resume => {
        ResumeResults => [
            { Id => 17, },
        ],
    },
    duplicates => {
        ResumeResults => [
            {
                Id => 18,
                Warnings => [
                    {
                        Code    => 10000,
                        Message => 'Объект присутствует в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                ],
            },
            {
                Id => 18,
                Warnings => [
                    {
                        Code    => 10000,
                        Message => 'Объект присутствует в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                ],
            },
        ],
    },
    resume_geo => {
        ResumeResults => [
            { Id => 12, },
        ],
    },
);

my %SAVED_DATA = (
    not_changed => {
        campaigns => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 10, uid => 1, type => 'mcb',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 11, uid => 2, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 12, uid => 1, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 13, uid => 1, type => 'wallet', statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 14, uid => 1, type => 'text',   statusEmpty => 'Yes', statusShow => 'No',  archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 15, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'Yes', statusBsSynced => 'No',  },
                    { cid => 16, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 17, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 18, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 19, uid => 2, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 20, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                ],
            },
        },
    },
    resume => {
        campaigns => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 10, uid => 1, type => 'mcb',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 11, uid => 2, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 12, uid => 1, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 13, uid => 1, type => 'wallet', statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 14, uid => 1, type => 'text',   statusEmpty => 'Yes', statusShow => 'No',  archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 15, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'Yes', statusBsSynced => 'No',  },
                    { cid => 16, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 17, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 18, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 19, uid => 2, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 20, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                ],
            },
        },
    },
    duplicates => {
        campaigns => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 10, uid => 1, type => 'mcb',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 11, uid => 2, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 12, uid => 1, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 13, uid => 1, type => 'wallet', statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 14, uid => 1, type => 'text',   statusEmpty => 'Yes', statusShow => 'No',  archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 15, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'Yes', statusBsSynced => 'No',  },
                    { cid => 16, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 17, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 18, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 19, uid => 2, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 20, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                ],
            },
        },
    },
    resume_geo => {
        campaigns => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 1,  uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 10, uid => 1, type => 'mcb',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 11, uid => 2, type => 'geo',    statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 12, uid => 1, type => 'geo',    statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 13, uid => 1, type => 'wallet', statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 14, uid => 1, type => 'text',   statusEmpty => 'Yes', statusShow => 'No',  archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 15, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'Yes', statusBsSynced => 'No',  },
                    { cid => 16, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 17, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 18, uid => 1, type => 'text',   statusEmpty => 'No',  statusShow => 'Yes', archived => 'No',  statusBsSynced => 'No',  },
                    { cid => 19, uid => 2, type => 'text',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                    { cid => 20, uid => 1, type => 'mcbanner',   statusEmpty => 'No',  statusShow => 'No',  archived => 'No',  statusBsSynced => 'Yes', },
                ],
            },
        },
    },
);

subtest 'Campaigns.Resume - request objects limit' => sub {
    plan tests => 5;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ map { $_ } 0 .. $LIMIT ], },
    });
    isa_ok($resp, 'Direct::Defect');
    is( $resp->code, 4001, 'error code' );
    is( $resp->text, 'Неверно заданы параметры фильтрации SelectionCriteria', 'error message' );
    is( $resp->description, "Массив Ids не может содержать более $LIMIT элементов", 'error description' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Resume - errors for invalid ids' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 'abc', '-1' ], },
    });
    is_deeply( $resp, $RESPONSES{invalid_id}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Resume - error for not owned campaign' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 19 ], },
    });
    is_deeply( $resp, $RESPONSES{not_owned_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Resume - errors for not exist or hidden campaigns' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'n' } @_ } };

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 1 ], },
    });
    is_deeply( $resp, $RESPONSES{not_exists_camps}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    #done_testing();
};

subtest 'Campaigns.Resume - error for read-only campaign' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'r' } @_ } };

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 1 ], },
    });
    is_deeply( $resp, $RESPONSES{readonly_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    #done_testing();
};

subtest 'Campaigns.Resume - error for campaign with not "api5_edit" kind' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 20, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_kind_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

=for Включить после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Resume - error for geo campaign and operator, not allowed to edit geo camp with api' => sub {
    plan tests => 2;

    my $srv = API::Service::Campaigns->new('Campaigns');
    $srv->set_current_operation('suspend');
    my $user = API::Authorization::User->new( uid => 2, login => 'not-allowed-to-edit-geo-camp', ClientID => 2 );
    $srv->set_subclient( $user );
    $srv->set_authorization( API::Authorization->fake_new( 2, 'not-allowed-to-edit-geo-camp', { chief_rep_user => $user, karma => 0 } ) );

    my $resp = $srv->resume({
        SelectionCriteria => { Ids => [ 11, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_operator}, 'expected response' );

    # my $resp = $service->resume({
    #     SelectionCriteria => { ids => [ 11, ], },
    # });
    # is_deeply( $resp, $RESPONSES{bad_operator}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};
=cut

subtest 'Campaigns.Resume - error for draft campaign' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 14, ], },
    });
    is_deeply( $resp, $RESPONSES{draft_campaign}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Resume - error for archived campaign' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 15, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_status_archived}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Resume - warning for not suspended campaign' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 16, ], },
    });
    is_deeply( $resp, $RESPONSES{not_suspended}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Resume - resume campaign' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 17, ], },
    });
    is_deeply( $resp, $RESPONSES{resume}, 'expected response' );

    check_test_dataset( $SAVED_DATA{resume}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Resume - warnings for duplicates' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 18, 18, ], },
    });
    is_deeply( $resp, $RESPONSES{duplicates}, 'expected response' );

    check_test_dataset( $SAVED_DATA{duplicates}, 'expected db state' );

    #done_testing();
};

=for Включить после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Resume - resume geo campaign' => sub {
    plan tests => 2;

    my $resp = $service->resume({
        SelectionCriteria => { Ids => [ 12, ], },
    });
    is_deeply( $resp, $RESPONSES{resume_geo}, 'expected response' );

    check_test_dataset( $SAVED_DATA{resume_geo}, 'expected db state' );

    #done_testing();
};
=cut

#done_testing();
