#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Yandex::DateTime qw/now/;
use Yandex::TimeCommon qw/unix2human/;
use DateTime::Duration;

use Test::More tests => 30;

BEGIN {
    use_ok('API::Service::Campaigns::Statuses');
}
my $state_from_model = \&API::Service::Campaigns::Statuses::state_from_model;
my $status_from_model = \&API::Service::Campaigns::Statuses::status_from_model;
my $status_payment_from_model = \&API::Service::Campaigns::Statuses::status_payment_from_model;


my $finish_time = now() + DateTime::Duration->new( days => 1 );
my $ended_finish_time = now() - DateTime::Duration->new( days => 1 );

my @state_tests = (
    [
        {
            currency => 'YND_FIXED',
            currencyConverted => "Yes",
            archived => 'Yes',
        },
        'CONVERTED'
    ], [
        {
            currency => 'YND_FIXED',
            currencyConverted => "Yes",
            archived => 'No',
        },
        'UNKNOWN'
    ], [
        {
            currency => 'RUB',
            currencyConverted => "Yes",
            archived => 'No',
        },
        'UNKNOWN'
    ], [
        {
            archived => 'Yes',
            currencyConverted => "No"
        },
        'ARCHIVED'
    ], [
        {
            archived => 'Yes',
            currencyConverted => "No",
            statusShow => 'No',
        },
        'ARCHIVED'
    ], [
        {
            finish_time => $ended_finish_time->ymd,
            archived => 'No',
            currencyConverted => "No"
        },
        'ENDED'
    ], [
        {
            finish_time => now()->ymd,
            archived => 'No',
            currencyConverted => "No"
        },
        'UNKNOWN'
    ], [
        {
            finish_time => undef,
            archived => 'No',
            currencyConverted => "No"
        },
        'UNKNOWN'
    ], [
        {
            statusShow => 'No',
            archived => 'No',
            currencyConverted => "No",
            finish_time => undef,
        },
        'SUSPENDED'
    ], [
        {
            statusShow => 'No',
            archived => 'No',
            currencyConverted => "No",
            finish_time => now()->ymd,
        },
        'SUSPENDED'
    ], [
        {
            statusShow => 'No',
            archived => 'No',
            currencyConverted => "No",
            finish_time => unix2human($finish_time->epoch),
        },
        'SUSPENDED'
    ], [
        {
            statusShow => 'Yes',
            archived => 'No',
            currencyConverted => "No",
            finish_time => now()->ymd,
        },
        'UNKNOWN'
    ], [
        {
            statusActive => 'No',
            statusShow => 'Yes',
            archived => 'No',
            currencyConverted => 'No',
            finish_time => undef,
        },
        'OFF'
    ], [
        {
            statusActive => 'Yes',
            statusShow => 'Yes',
            archived => 'No',
            currencyConverted => 'No',
            finish_time => undef,
        },
        'ON'
    ]

);

foreach my $test ( @state_tests ) {
    is_deeply(
        $state_from_model->( $test->[0] ), $test->[1], "state $test->[1]"
    );
}


my @status_tests = (
    [
        {
            statusModerate => 'New'
        },
        'DRAFT'
    ], [
        {
            statusModerate => 'Sent',
        },
        'MODERATION'
    ], [
        {
            statusModerate => 'Ready',
        },
        'MODERATION'
    ], [
        {
            statusModerate => 'No'
        },
        'REJECTED'
    ], [
        {
            statusModerate => 'Yes',
        },
        'ACCEPTED'
    ], [
        {
            statusModerate => 'qwe',
        },
        'UNKNOWN'
    ]
);

foreach my $test ( @status_tests ) {
    is_deeply(
        $status_from_model->( $test->[0] ), $test->[1], "status $test->[1]"
    );
}

my @status_payment_tests = (
    [
        {
            statusModerate => 'New',
            wallet_cid => 0
        },
        'DISALLOWED'
    ], [
        {
            statusModerate => 'Sent',
            wallet_cid => 0
        },
        'DISALLOWED'
    ], [
        {
            statusModerate => 'No',
            wallet_cid => 0
        },
        'DISALLOWED'
    ], [
        {
            statusModerate => 'Ready',
            wallet_cid => 892
        },
        'ALLOWED'
    ], [
        {
            statusModerate => 'Yes',
            wallet_cid => 0
        },
        'ALLOWED'
    ], [
        {
            statusModerate => 'Yes',
            wallet_cid => 892
        },
        'ALLOWED'
    ], [
        {
            statusModerate => 'New',
            wallet_cid => 892
        },
        'ALLOWED'
    ], [
        {
            statusModerate => 'No',
            wallet_cid => 892
        },
        'ALLOWED'
    ], [
        {
            statusModerate => 'qwe',
        },
        'UNKNOWN'
    ]
);


foreach my $test ( @status_payment_tests ) {
    is_deeply(
        $status_payment_from_model->( $test->[0] ), $test->[1], "status_payment $test->[1]"
    );
}


