#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Test::More tests => 34;

BEGIN {
    use_ok('API::Service::Campaigns::Strategies');
}

my $to_internal_func = \&API::Service::Campaigns::Strategies::strategy_sets_to_internal;
my $to_external_func = \&API::Service::Campaigns::Strategies::strategy_sets_to_external;

subtest 'Search -> HIGHEST_POSITION Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "HIGHEST_POSITION",
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 0,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "default",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "default"
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> IMPRESSIONS_BELOW_SEARCH Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "IMPRESSIONS_BELOW_SEARCH",
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 0,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "strategy_no_premium",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "no_premium",
            place => "highest_place"
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> WEEKLY_CLICK_PACKAGE Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "WEEKLY_CLICK_PACKAGE",
            "WeeklyClickPackage" => {
                    "ClicksPerWeek" => 500,
                    "AverageCpc" => 12_000_000
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_week_bundle",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget_week_bundle",
            avg_bid => 12,
            limit_clicks => 500
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> WB_MAXIMUM_CONVERSION_RATE Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "WB_MAXIMUM_CONVERSION_RATE",
            "WbMaximumConversionRate" => {
                "WeeklySpendLimit" => 320_000_000,
                "BidCeiling" => 15_000_000,
                "GoalId" => 0
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "cpa_optimizer",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget",
            goal_id => 0,
            sum => 320,
            bid => 15
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> WB_MAXIMUM_CLICKS Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "WB_MAXIMUM_CLICKS",
            "WbMaximumClicks" => {
                "WeeklySpendLimit" => 320_000_000,
                "BidCeiling" => 15_000_000,
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget",
            sum => 320,
            bid => 15
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> AVERAGE_CPC Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "AVERAGE_CPC",
            "AverageCpc" => {
                "AverageCpc" => 9_000_000,
                "WeeklySpendLimit" => 650_000_000,
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_avg_click",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget_avg_click",
            sum => 650,
            avg_bid => 9
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> AVERAGE_CPA Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "AVERAGE_CPA",
            "AverageCpa" => {
                "AverageCpa" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "GoalId" => 0,
                "BidCeiling" => 11_000_000
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_avg_cpa",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget_avg_cpa",
            sum => 650,
            bid => 11,
            pay_for_conversion => 0,
            goal_id => 0,
            avg_cpa => 20
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> PAY_FOR_CONVERSION Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "PAY_FOR_CONVERSION",
            "PayForConversion" => {
                "Cpa" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "GoalId" => 123
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_avg_cpa",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name               => "autobudget_avg_cpa",
            sum                => 650,
            goal_id            => 123,
            avg_cpa            => 20,
            pay_for_conversion => 1
        }
    };

    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> PAY_FOR_CONVERSION_PER_CAMPAIGN  Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "PAY_FOR_CONVERSION_PER_CAMPAIGN",
            "PayForConversionPerCampaign" => {
                "Cpa" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "GoalId" => 123,
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_avg_cpa_per_camp",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name               => "autobudget_avg_cpa_per_camp",
            sum                => 650,
            goal_id            => 123,
            avg_cpa            => 20,
            pay_for_conversion => 1
        }
    };

    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> AVERAGE_CPA_PER_CAMPAIGN  Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "AVERAGE_CPA_PER_CAMPAIGN",
            "AverageCpaPerCampaign" => {
                "AverageCpa" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "GoalId" => 123,
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_avg_cpa_per_camp",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name               => "autobudget_avg_cpa_per_camp",
            sum                => 650,
            goal_id            => 123,
            avg_cpa            => 20,
            pay_for_conversion => 0
        }
    };

    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> AVERAGE_CRR Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "AVERAGE_CRR",
            "AverageCrr" => {
                "GoalId" => 0,
                "WeeklySpendLimit" => 650_000_000,
                "Crr" => 100
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_crr",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget_crr",
            sum => 650,
            goal_id => 0,
            crr => 100,
            pay_for_conversion => 0
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> AVERAGE_ROI Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "AVERAGE_ROI",
            "AverageRoi" => {
                "ReserveReturn" => 20,
                "RoiCoef" => 2_300_000,
                "GoalId" => 0,
                "BidCeiling" => 11_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "Profitability" => 80_000_000
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_roi",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget_roi",
            sum => 650,
            bid => 11,
            goal_id => 0,
            roi_coef => 2.3,
            reserve_return => 20,
            profitability => 80,
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};


subtest 'Search -> AVERAGE_CPI Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "AVERAGE_CPI",
            "AverageCpi" => {
                "AverageCpi" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "BidCeiling" => 11_000_000
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_avg_cpi",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget_avg_cpi",
            sum => 650,
            bid => 11,
            avg_cpi => 20,
            pay_for_conversion => 0
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};


subtest 'Search -> PAY_FOR_INSTALL Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "PAY_FOR_INSTALL",
            "PayForInstall" => {
                "AverageCpi" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_avg_cpi",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget_avg_cpi",
            sum => 650,
            avg_cpi => 20,
            pay_for_conversion => 1
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> WB_MAXIMUM_APP_INSTALLS Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "WB_MAXIMUM_APP_INSTALLS",
            "WbMaximumAppInstalls" => {
                "WeeklySpendLimit" => 320_000_000,
                "BidCeiling" => 15_000_000,
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "cpa_optimizer",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget",
            goal_id => 4,
            sum => 320,
            bid => 15
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> WB_MAXIMUM_APP_INSTALLS Network -> NETWORK_DEFAULT2' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "WB_MAXIMUM_APP_INSTALLS",
            "WbMaximumAppInstalls" => {
                "WeeklySpendLimit" => 320_000_000,
                "BidCeiling" => 15_000_000,
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "cpa_optimizer",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name => "autobudget",
            goal_id => 38402972,
            sum => 320,
            bid => 15
        }
    };
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> HIGHEST_POSITION Network -> MAXIMUM_COVERAGE' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "HIGHEST_POSITION",
        },
        "Network" => {
            "BiddingStrategyType" => "MAXIMUM_COVERAGE",
        }
    };
    my $internal_strategy = {
        is_autobudget => 0,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "different_places",
        net => {
            name => "maximum_coverage"
        },
        search => {
            name => "default"
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};
subtest 'Search -> IMPRESSIONS_BELOW_SEARCH Network -> MAXIMUM_COVERAGE' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "IMPRESSIONS_BELOW_SEARCH",
        },
        "Network" => {
            "BiddingStrategyType" => "MAXIMUM_COVERAGE",
        }
    };
    my $internal_strategy = {
        is_autobudget => 0,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "different_places",
        net => {
            name => "maximum_coverage"
        },
        search => {
            name => "no_premium",
            place => "highest_place"
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> MAXIMUM_COVERAGE' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "MAXIMUM_COVERAGE",
        }
    };
    my $internal_strategy = {
        is_autobudget => 0,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "maximum_coverage"
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> HIGHEST_POSITION Network -> SERVING_OFF' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "HIGHEST_POSITION",
        },
        "Network" => {
            "BiddingStrategyType" => "SERVING_OFF",
        }
    };
    my $internal_strategy = {
        is_autobudget => 0,
        is_net_stop => 1,
        is_search_stop => 0,
        name => "default",
        net => {
            name => "stop"
        },
        search => {
            name => "default"
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> WEEKLY_CLICK_PACKAGE' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "WEEKLY_CLICK_PACKAGE",
            "WeeklyClickPackage" => {
                "ClicksPerWeek" => 500,
                "AverageCpc" => 12_000_000
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_week_bundle",
            avg_bid => 12,
            limit_clicks => 500
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> WB_MAXIMUM_CONVERSION_RATE' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "WB_MAXIMUM_CONVERSION_RATE",
            "WbMaximumConversionRate" => {
                "WeeklySpendLimit" => 320_000_000,
                "BidCeiling" => 15_000_000,
                "GoalId" => 0
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget",
            goal_id => 0,
            sum => 320,
            bid => 15
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> WB_MAXIMUM_CLICKS' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "WB_MAXIMUM_CLICKS",
            "WbMaximumClicks" => {
                "WeeklySpendLimit" => 320_000_000,
                "BidCeiling" => 15_000_000,
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget",
            sum => 320,
            bid => 15
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> AVERAGE_CPC' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "AVERAGE_CPC",
            "AverageCpc" => {
                "AverageCpc" => 9_000_000,
                "WeeklySpendLimit" => 650_000_000,
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_avg_click",
            sum => 650,
            avg_bid => 9
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> AVERAGE_CPA' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "AVERAGE_CPA",
            "AverageCpa" => {
                "AverageCpa" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "GoalId" => 0,
                "BidCeiling" => 11_000_000
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_avg_cpa",
            sum => 650,
            bid => 11,
            goal_id => 0,
            avg_cpa => 20,
            pay_for_conversion => 0
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> AVERAGE_CRR' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "AVERAGE_CRR",
            "AverageCrr" => {
                "GoalId" => 0,
                "WeeklySpendLimit" => 650_000_000,
                "Crr" => 100
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_crr",
            sum => 650,
            goal_id => 0,
            crr => 100,
            pay_for_conversion => 0
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> WB_AVERAGE_CPV' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "WB_AVERAGE_CPV",
            "WbAverageCpv" => {
                "AverageCpv" => 1_000_000,
                "SpendLimit" => 650_000_000,
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_avg_cpv",
            sum => 650,
            avg_cpv => 1,
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> CP_AVERAGE_CPV' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "CP_AVERAGE_CPV",
            "CpAverageCpv" => {
                "AverageCpv" => 1_000_000,
                "SpendLimit" => 650_000_000,
                "StartDate" => "2020-09-21",
                "EndDate" => "2020-09-24",
                "AutoContinue" => "NO",
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name              => "autobudget_avg_cpv_custom_period",
            budget            => 650,
            avg_cpv           => 1,
            start             => "2020-09-21",
            finish            => "2020-09-24",
            auto_prolongation => 0,
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> AVERAGE_ROI' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "AVERAGE_ROI",
            "AverageRoi" => {
                "ReserveReturn" => 20,
                "RoiCoef" => 2_300_000,
                "GoalId" => 0,
                "BidCeiling" => 11_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "Profitability" => 80_000_000
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_roi",
            sum => 650,
            bid => 11,
            goal_id => 0,
            roi_coef => 2.3,
            reserve_return => 20,
            profitability => 80,
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> AVERAGE_CPI' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "AVERAGE_CPI",
            "AverageCpi" => {
                "AverageCpi" => 20_000_000,
                "WeeklySpendLimit" => 650_000_000,
                "BidCeiling" => 11_000_000
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_avg_cpi",
            sum => 650,
            bid => 11,
            avg_cpi => 20,
            pay_for_conversion => 0
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> WB_MAXIMUM_APP_INSTALLS' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "WB_MAXIMUM_APP_INSTALLS",
            "WbMaximumAppInstalls" => {
                "WeeklySpendLimit" => 320_000_000,
                "BidCeiling" => 15_000_000,
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget",
            goal_id => 4,
            sum => 320,
            bid => 15
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> PAY_FOR_CONVERSION_CRR Network -> NETWORK_DEFAULT' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "PAY_FOR_CONVERSION_CRR",
            "PayForConversionCrr" => {
                "GoalId" => 0,
                "WeeklySpendLimit" => 650_000_000,
                "Crr" => 100
            },
        },
        "Network" => {
            "BiddingStrategyType" => "NETWORK_DEFAULT",
            "NetworkDefault" => {
                "LimitPercent" => 50,
            }
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 0,
        name => "autobudget_crr",
        net => {
            ContextLimit => 50,
            name => "default"
        },
        search => {
            name               => "autobudget_crr",
            sum                => 650,
            goal_id            => 0,
            crr                => 100,
            pay_for_conversion => 1
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};

subtest 'Search -> SERVING_OFF Network -> PAY_FOR_CONVERSION_CRR' => sub {
    my $external_strategy = {
        "Search" => {
            "BiddingStrategyType" => "SERVING_OFF",
        },
        "Network" => {
            "BiddingStrategyType" => "PAY_FOR_CONVERSION_CRR",
            "PayForConversionCrr" => {
                "GoalId" => 0,
                "WeeklySpendLimit" => 650_000_000,
                "Crr" => 100
            },
        }
    };
    my $internal_strategy = {
        is_autobudget => 1,
        is_net_stop => 0,
        is_search_stop => 1,
        name => "different_places",
        net => {
            name => "autobudget_crr",
            sum => 650,
            goal_id => 0,
            crr => 100,
            pay_for_conversion => 1
        },
        search => {
            name => "stop",
        }
    };
    is_deeply($to_internal_func->($external_strategy), $internal_strategy, 'to_internal');
    is_deeply($to_external_func->($internal_strategy), $external_strategy, 'to_external');
};
