#!/usr/bin/perl

use Direct::Modern;

use Test::More tests => 11;

#use my_inc '../../../../', for => 'api/t';

use API::Authorization;
use API::Authorization::User;
use API::Service::Campaigns;
use Mediaplan qw//;
use Settings;
use Units::TestDummy;
use Yandex::DBUnitTest qw/ :all /;
use Yandex::Test::UTF8Builder;
use Client;

{
    no strict qw/ refs /;
    no warnings qw/ once redefine /;

    *API::Authorization::User::get_info_by_uid_passport = sub ($)   { +{} };
    *API::Authorization::User::create_update_user       = sub ($;$) { };

    *API::Service::Base::rbac                         = sub { bless {} => 'RBAC2::Extended' };
    *API::Service::Base::operator_role                = sub { 'client' };
    *API::Service::Base::subclient_role               = sub { 'client' };
    *API::Service::Base::not_owned_campaign_ids_hash  = sub { () };
    *API::Service::Base::check_campaigns_write_detail = sub { () };
    *API::Service::Base::units = sub { state $dummy_units = Units::TestDummy->new() };

    *API::Service::Base::send_alert                   = sub {};
    *Mediaplan::close_request_first_aid               = sub {};

    *Yandex::Log::new                                 = sub { return bless {}, $_[0] };
    *Yandex::Log::out                                 = sub {};

    *Client::ClientFeatures::has_access_to_new_feature_from_java = sub { return 1 };
}

my $TEST_DATA = {
    adgroups_dynamic => {
        original_db => PPC( shard => 'all' ),
    },
    adgroups_mobile_content => {
        original_db => PPC( shard => 'all' ),
    },
    api_special_user_options => {
        original_db => PPC( shard => 'all' ),
    },
    banner_images => {
        original_db => PPC( shard => 'all' ),
    },
    images => {
        original_db => PPC( shard => 'all' ),
    },
    banners_performance => {
        original_db => PPC( shard => 'all' ),
    },
    banner_display_hrefs => {
        original_db => PPC( shard => 'all' ),
    },
    banner_turbolandings => {
        original_db => PPC( shard => 'all' ),
    },
    banners => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { bid => 1, pid => 1, statusActive => 'No', statusModerate => 'No', LastChange => '2015-08-03 13:00:00', },
            ],
        },
    },
    bids => {
        original_db => PPC( shard => 'all' ),
    },
    bids_arc => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { id => 1, cid => 4, pid => 1, },
            ],
        },
    },
    campaigns => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { cid => 1, uid => 1, type => 'wallet', statusEmpty => 'No', currency => 'YND_FIXED', },
                { cid => 2, uid => 2, type => 'geo',    statusEmpty => 'No', currency => 'YND_FIXED', },
                { cid => 3, uid => 3, type => 'text',   statusEmpty => 'No', archived => 'Yes', currency => 'YND_FIXED', },
                { cid => 4, uid => 1, type => 'text',   statusEmpty => 'No', archived => 'Yes', start_time => '2015-08-03', currency => 'YND_FIXED', },
                { cid => 5, uid => 1, type => 'text',   statusEmpty => 'No', archived => 'Yes', currency => 'YND_FIXED', },
                { cid => 6, uid => 1, type => 'text',   statusEmpty => 'No', archived => 'Yes', currency => 'YND_FIXED', },
                { cid => 7, uid => 1, type => 'mcbanner',   statusEmpty => 'No', archived => 'Yes', currency => 'YND_FIXED', },
            ],
        },
    },
    campaigns_performance => {
        original_db => PPC( shard => 1 ),
        rows => {},
    },
    camp_operations_queue => {
        original_db => PPC( shard => 'all' ),
    },
    camp_options => {
        original_db => PPC( shard => 'all' ),
    },
    subcampaigns => {
        original_db => PPC(shard => 'all'),
    },
    clients => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { ClientID => 1, work_currency => undef, },
                { ClientID => 2, work_currency => undef, },
                { ClientID => 3, work_currency => 'RUB', },
            ],
        },
    },
    client_limits => {
        original_db => PPC( shard => 'all' ),
    },
    clients_api_options => {
        original_db => PPC( shard => 'all' ),
    },
    clients_options => {
        original_db => PPC( shard => 'all' ),
    },
    domains => {
        original_db => PPC( shard => 'all' ),
    },
    mod_export_candidates => {
        original_db => PPC( shard => 'all' ),
    },
    minus_words => {
        original_db => PPC( shard => 'all' ),
    },
    phrases => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { pid => 1, cid => 4, LastChange => '2015-08-03 13:00:00', statusShowsForecast => 'Archived', },
            ],
        },
    },
    aggr_statuses_campaigns => {
        original_db => PPC(shard => 'all'),
        rows => [],
    },
    shard_inc_bid => {
        original_db => PPCDICT(),
        rows => [
            { bid => 1, ClientID => 1, },
        ],
    },
    shard_client_id => {
        original_db => PPCDICT(),
        rows => [
            { ClientID => 1, shard => 1, },
            { ClientID => 2, shard => 1, },
            { ClientID => 3, shard => 1, },
        ],
    },
    shard_inc_cid => {
        original_db => PPCDICT(),
        rows => [
            { cid => 1, ClientID => 1, },
            { cid => 2, ClientID => 2, },
            { cid => 3, ClientID => 3, },
            { cid => 4, ClientID => 1, },
            { cid => 5, ClientID => 1, },
        ],
    },
    shard_inc_pid => {
        original_db => PPCDICT(),
        rows => [
            { pid => 1, ClientID => 1, },
        ],
    },
    shard_uid => {
        original_db => PPCDICT(),
        rows => [
            { uid => 1, ClientID => 1, },
            { uid => 2, ClientID => 2, },
            { uid => 3, ClientID => 3, },
        ],
    },
    strategies => {
        original_db => PPC( shard => 'all' ),
    },
    users => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, login => 'unit-tester',                  statusBlocked => 'no', ClientID => 1, },
                { uid => 2, login => 'not-allowed-to-edit-geo-camp', statusBlocked => 'no', ClientID => 2, },
                { uid => 3, login => 'multicurrency-converted',      statusBlocked => 'no', ClientID => 3, },
            ],
        },
    },
    users_api_options => {
        original_db => PPC( shard => 'all' ),
        rows => {
            1 => [
                { uid => 1, api_geo_allowed => 'Yes', },
                { uid => 2, api_geo_allowed => 'No',  },
                { uid => 3, api_geo_allowed => 'No',  },
            ],
        },
    },
    users_options => {
        original_db => PPC( shard => 'all' ),
    },
    vcards => {
        original_db => PPC( shard => 'all' ),
    },
};

init_test_dataset( $TEST_DATA );


my $login = 'unit-tester';
my $uid   = 1;
my $LIMIT = $API::Service::Campaigns::UNARCHIVE_IDS_LIMIT;

my $user = API::Authorization::User->new(
    uid      => $uid,
    login    => $login,
    ClientID => 1,
);

my $service = API::Service::Campaigns->new('Campaigns');
$service->set_current_operation('unarchive');
$service->set_subclient( $user );
$service->set_authorization( API::Authorization->fake_new( $uid, $login, { chief_rep_user => $user, karma => 0 } ) );


my %RESPONSES = (
    invalid_id => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 5005,
                        Message => 'Поле задано неверно',
                        Details => 'Значение поля Ids должно быть целым положительным числом',
                    },
                ],
            },
        ],
    },
    not_exists_camps => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    not_owned_campaign => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            }
        ],
    },
    readonly_campaign => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 54,
                        Message => 'Нет прав',
                        Details => 'Нет прав на запись',
                    },
                ],
            }
        ],
    },
    bad_kind_campaign => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
            {
                Errors => [
                    {
                        Code    => 3500,
                        Message => 'Не поддерживается',
                        Details => 'Тип кампании не поддерживается',
                    },
                ],
            },
        ],
    },
    bad_operator => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8800,
                        Message => 'Объект не найден',
                        Details => 'Кампания не найдена',
                    },
                ],
            },
        ],
    },
    multicurrency_converted => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8304,
                        Message => 'Невозможно разархивировать объект',
                        Details => 'Кампания 3 не может быть разархивирована, так как находится в специальном архиве.',
                    },
                ],
            },
        ],
    },
    camps_count_limit => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8304,
                        Message => 'Невозможно разархивировать объект',
                        Details => 'Превышено максимальное количество кампаний — 0',
                    },
                ],
            },
        ],
    },
    unarc_camps_count_limit => {
        UnarchiveResults => [
            {
                Errors => [
                    {
                        Code    => 8304,
                        Message => 'Невозможно разархивировать объект',
                        Details => 'Превышено максимальное количество незаархивированных кампаний — 0',
                    },
                ],
            },
        ],
    },
    unarchive => {
        UnarchiveResults => [
            { Id => 4, },
        ],
    },
    duplicates_and_not_archived => {
        UnarchiveResults => [
            {
                Id       => 4,
                Warnings => [
                    {
                        Code    => 10000,
                        Message => 'Объект присутствует в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                    {
                        Code    => 10023,
                        Message => 'Объект не заархивирован',
                        Details => 'Кампания не заархивирована',
                    },
                ],
            },
            {
                Id       => 4,
                Warnings => [
                    {
                        Code    => 10000,
                        Message => 'Объект присутствует в запросе более одного раза',
                        Details => 'Id кампании присутствует в запросе более одного раза',
                    },
                    {
                        Code    => 10023,
                        Message => 'Объект не заархивирован',
                        Details => 'Кампания не заархивирована',
                    },
                ],
            },
        ],
    },
    enqueue_for_unarc => {
        UnarchiveResults => [
            { Id => 5, },
        ],
    },
);

my %SAVED_DATA = (
    unarchive => {
        banners => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { bid => 1, pid => 1, statusBsSynced => 'No', },
                ],
            },
        },
        bids => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { id => 1, cid => 4, pid => 1, },
                ],
            },
        },
        bids_arc => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [],
            },
        },
        campaigns => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { cid => 1, uid => 1, type => 'wallet', statusEmpty => 'No', },
                    { cid => 2, uid => 2, type => 'geo',    statusEmpty => 'No', },
                    { cid => 3, uid => 3, type => 'text',   statusEmpty => 'No', archived => 'Yes', },
                    { cid => 4, uid => 1, type => 'text',   statusEmpty => 'No', archived => 'No', statusModerate => 'Ready', statusBsSynced => 'No', start_time => '2015-08-03'},
                    { cid => 5, uid => 1, type => 'text',   statusEmpty => 'No', archived => 'Yes', },
                    { cid => 6, uid => 1, type => 'text',   statusEmpty => 'No', archived => 'Yes', },
                    { cid => 7, uid => 1, type => 'mcbanner',   statusEmpty => 'No', archived => 'Yes', },
                ],
            },
        },
        mod_export_candidates => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { cid => 4, },
                ],
            },
        },
        phrases => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { pid => 1, cid => 4, LastChange => '2015-08-03 13:00:00', statusBsSynced => 'No', statusShowsForecast => 'New', },
                ],
            },
        },
    },
    enqueue_for_unarc => {
        camp_operations_queue => {
            original_db => PPC( shard => 'all' ),
            rows => {
                1 => [
                    { cid => 5, operation => 'unarc', },
                ],
            },
        },
    },
);

###########################
# TODO:
#   * checks for db stat
#   * succes for geo with proper operator
#

=for
use feature qw/say/;
use DDP;
use Test::Deep qw/ superhashof cmp_details deep_diag/;
use Yandex::DBTools;
my $expected = $SAVED_DATA{unarchive}->{campaigns}{rows}{1};
my $rows = get_all_sql(SHUT_1, 'SELECT cid, uid, type, statusEmpty, archived, statusModerate, statusBsSynced, start_time from campaigns ORDER BY cid');
p $rows;
for ( my $i = 0; $i <= $#$rows; $i++ ) {
    my ($ok, $stack) = cmp_details( $rows->[ $i ], superhashof( $expected->[ $i ] ) );
    next if ok($ok);
    say STDERR $i;
    p deep_diag($stack);
}
=cut

subtest 'Campaigns.Unarchive - request objects limit' => sub {
    #plan tests => 5;

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ map { $_ } 0 .. $LIMIT ], },
    });
    isa_ok($resp, 'Direct::Defect');
    is( $resp->code, 4001, 'error code' );
    is( $resp->text, 'Неверно заданы параметры фильтрации SelectionCriteria', 'error message' );
    is( $resp->description, "Массив Ids не может содержать более $LIMIT элементов", 'error description' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};

subtest 'Campaigns.Unarchive - errors for invalid ids' => sub {
    plan tests => 2;

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 'abc', '-1' ], },
    });
    is_deeply( $resp, $RESPONSES{invalid_id}, 'expected response' );

    check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Unarchive - error for not owned campaign' => sub {
    #plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::not_owned_campaign_ids_hash;
    *API::Service::Base::not_owned_campaign_ids_hash = sub {shift; { map { $_ => undef } @_ } };

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 3 ], },
    });
    is_deeply( $resp, $RESPONSES{not_owned_campaign}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::not_owned_campaign_ids_hash = $origin;

    done_testing();
};

subtest 'Campaigns.Unarchive - errors for not exist campaigns or hidden campaigns' => sub {
    #plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'n' } @_ } };

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 1000 ], },
    });
    is_deeply( $resp, $RESPONSES{not_exists_camps}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    done_testing();
};

subtest 'Campaigns.Unarchive - error for read-only campaign' => sub {
    #plan tests => 2;

    no warnings qw/ once redefine /;
    my $origin = \&API::Service::Base::check_campaigns_write_detail;
    *API::Service::Base::check_campaigns_write_detail = sub { shift; { map { $_ => 'r' } @_ } };

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 6 ], },
    });
    is_deeply( $resp, $RESPONSES{readonly_campaign}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    *API::Service::Base::check_campaigns_write_detail = $origin;

    done_testing();
};

subtest 'Campaigns.Unarchive - error for campaign with not "api_edit" kind' => sub {
    #plan tests => 2;

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 1, 7 ], },
    });
    is_deeply( $resp, $RESPONSES{bad_kind_campaign}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};

=for Включить после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Unarchive - error for geo campaign and operator, not allowed to edit geo camp with api' => sub {
    #plan tests => 2;

    my $srv = API::Service::Campaigns->new('Campaigns');
    $srv->set_current_operation('archive');
    my $user = API::Authorization::User->new( uid => 2, login => 'not-allowed-to-edit-geo-camp', ClientID => 2 );
    $srv->set_subclient( $user );
    $srv->set_authorization( API::Authorization->fake_new( 2, 'not-allowed-to-edit-geo-camp', { chief_rep_user => $user, karma => 0 } ) );

    my $resp = $srv->unarchive({
        SelectionCriteria => { Ids => [ 2, ], },
    });
    is_deeply( $resp, $RESPONSES{bad_operator}, 'expected response' );

    # TODO: check for opposite case

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};
=cut

subtest 'Campaigns.Unarchive - error for client, converted to multicurrency' => sub {
    #plan tests => 2;

    my $srv = API::Service::Campaigns->new('Campaigns');
    $srv->set_current_operation('archive');
    my $user = API::Authorization::User->new( uid => 3, login => 'multicurrency-converted', ClientID => 3 );
    $srv->set_subclient( $user );
    $srv->set_authorization( API::Authorization->fake_new( 3, 'multicurrency-converted', { chief_rep_user => $user, karma => 0 } ) );

    my $resp = $srv->unarchive({
        SelectionCriteria => { Ids => [ 3, ], },
    });
    is_deeply( $resp, $RESPONSES{multicurrency_converted}, 'expected response' );

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};

subtest 'Campaigns.Unarchive - error for client, breaking the camps count limit' => sub {
    #plan tests => 2;

    no warnings qw/ once redefine /;

    my $orig_unarc_camp_count_limit = $Settings::DEFAULT_UNARC_CAMP_COUNT_LIMIT;
    
    $Settings::DEFAULT_UNARC_CAMP_COUNT_LIMIT = 0;
    # надо сбросить кеш
    delete_client_limits(1);

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 4, ], },
    });
    is_deeply( $resp, $RESPONSES{unarc_camps_count_limit}, 'expected response' );

    $Settings::DEFAULT_UNARC_CAMP_COUNT_LIMIT = $orig_unarc_camp_count_limit;
    # надо сбросить кеш
    delete_client_limits(1);

    #check_test_dataset( $SAVED_DATA{not_changed}, 'expected db state' );

    done_testing();
};

subtest 'Campaigns.Unarchive - succes case' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;

    my $orig_check_moderate = \&Common::check_domain_phone_on_moderate;
    my $orig_send_moderate  = \&Common::send_banners_to_moderate;

    *Common::check_domain_phone_on_moderate = sub ($$$) { 0 };
    *Common::send_banners_to_moderate       = sub { 1 };

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 4, ], },
    });
    is_deeply( $resp, $RESPONSES{unarchive}, 'expected response' );

    *Common::check_domain_phone_on_moderate = $orig_check_moderate;
    *Common::send_banners_to_moderate       = $orig_send_moderate;

    check_test_dataset( $SAVED_DATA{unarchive}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Unarchive - warnings for not archived and duplicates' => sub {
    plan tests => 2;

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 4, 4 ], },
    });
    is_deeply( $resp, $RESPONSES{duplicates_and_not_archived}, 'expected response' );

    check_test_dataset( $SAVED_DATA{duplicates_and_not_archived}, 'expected db state' );

    #done_testing();
};

subtest 'Campaigns.Unarchive - enqueue heavy campaign for unarchiving' => sub {
    plan tests => 2;

    no warnings qw/ once redefine /;

    my $orig = \&Common::is_campaign_heavy;

    *Common::is_campaign_heavy = sub ($) { 1 };

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 5 ], },
    });
    is_deeply( $resp, $RESPONSES{enqueue_for_unarc}, 'expected response' );

    check_test_dataset( $SAVED_DATA{enqueue_for_unarc}, 'expected db state' );

    *Common::is_campaign_heavy = $orig;

    #done_testing();
};

=for TODO: дописать после того как разрешим работу с гео кампаниями в API5
subtest 'Campaigns.Resume - unarchive geo campaign' => sub {
    plan tests => 2;

    my $resp = $service->unarchive({
        SelectionCriteria => { Ids => [ 12, ], },
    });
    is_deeply( $resp, $RESPONSES{unarchive_geo}, 'expected response' );

    check_test_dataset( $SAVED_DATA{unarchive_geo}, 'expected db state' );

    #done_testing();
};
=cut

#done_testing();
