#!/usr/bin/perl
use Direct::Modern;
use my_inc '../../../..', for => 'api/t';

use Encode 'encode_utf8';
use Guard;
use Hash::Util 'lock_hash_recurse';
use List::MoreUtils qw/natatime uniq/;
use Scalar::Util;
use Test::Deep;
use Test::MockTime ':all';
use Test::More tests => 8;

use Yandex::ListUtils 'xflatten';
use Yandex::HashUtils;

use Yandex::Clone 'yclone';

use API::Reports::DataRules qw();
use API::Test::Reports::FakeService::Reports;
use API::Test::Reports::FakeUser;
use API::Test::Reports::FakeRBAC;
use API::Test::Reports::FakeCampaignIdLookup;
use API::Test::Reports::FakePageIdMap;
use API::Test::Reports::Fields qw(
    all_fields
    all_date_range_types
    @REPORT_TYPES
    ADGROUP_PERFORMANCE_REPORT_TYPE
    AD_PERFORMANCE_REPORT_TYPE
    ACCOUNT_PERFORMANCE_REPORT_TYPE
    CAMPAIGN_PERFORMANCE_REPORT_TYPE
    CRITERIA_PERFORMANCE_REPORT_TYPE
    CUSTOM_REPORT_TYPE
    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE
    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE
    STRATEGY_PERFORMANCE_REPORT_TYPE
);
use API::Test::MockHelper qw( mock_subs restore_mocked_subs );
cmp_deeply(\@API::Reports::DataRules::REPORT_TYPES, \@REPORT_TYPES, 'lists of report types are not match');

no warnings 'redefine';
local *Client::ClientFeatures::has_access_to_new_feature_from_java = sub { return 0 };
local *Property::get = sub { return undef };

my $SAMPLE_REQUEST = {
    'Currency' => 'RUB',
    'DateRangeType' => 'CUSTOM_DATE',
    'FieldNames' => [
        'Date',
        'Impressions',
        'Clicks',
        'Cost',
        'Ctr',
        'AvgCpc',
    ],
    'Format' => 'TSV',
    'IncludeDiscount' => 'YES',
    'IncludeVAT' => 'YES',
    'OrderBy' => [ { 'Field' => 'Impressions', 'SortOrder' => 'DESCENDING' } ],
    'Page' => { 'Limit' => 5 },
    'ReportName' => "Report 1",
    'ReportType' => CUSTOM_REPORT_TYPE,
    'SelectionCriteria' => {
        'DateFrom' => '20151020',
        'DateTo' => '20160406',
        'Filter' => [
            { 'Field' => 'Cost', 'Operator' => 'GREATER_THAN', 'Values' => [ '7000000000' ] },
        ],
    },
};

## какие параметры у теста
my $user_login = 'geoscientists';
my $request_data = $SAMPLE_REQUEST;
my $unavailable_cids = [];
my $request_http_headers = {};
my $all_cids_with_statistics = [ map { $_ + 200 } qw( 1 2 4 18 19 22 33 43 57 ) ];
my $subclient_currency = 'RUB';
my $user_ids = { $user_login => { uid => 823, ClientID => 283 } };
my $rbac_allow_show_stat_camps = { map { $_ => 1 } @$all_cids_with_statistics };
my $cid_to_orderid_map = { map { $_ => $_ - 200 } @$all_cids_with_statistics };
my $page_name_to_page_ids_map = { foo => [ 123, 456 ], bar => [ 789 ], 'Яндекс' => [1,2,3] };

subtest FieldNames => sub {
    my %mandatory_group_by = (
        ADGROUP_PERFORMANCE_REPORT_TYPE()  => [ 'adgroup' ],
        AD_PERFORMANCE_REPORT_TYPE()       => [ 'banner' ],
        CAMPAIGN_PERFORMANCE_REPORT_TYPE() => [ 'campaign' ],
        STRATEGY_PERFORMANCE_REPORT_TYPE() => ['strategy_id'],
        CRITERIA_PERFORMANCE_REPORT_TYPE() => sub {
            my ($field_name) = @_;
            my $criterion_type_field = $field_name eq 'CriterionType' ? 'criterion_type' : 'contexttype_orig';
            my $criteria_field = ($field_name eq 'Criteria' || $field_name eq 'CriteriaId') ? 'contextcond_mod' : 'contextcond_orig_mod';
            return ['adgroup', $criterion_type_field, $criteria_field];
        },
        CUSTOM_REPORT_TYPE() => [],
        SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => ['adgroup', 'search_query'],
        ACCOUNT_PERFORMANCE_REPORT_TYPE() => [],
        REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE() => [],
    );
    foreach my $report_type (@REPORT_TYPES) {
        for my $field_name ( sort ( all_fields() ) ) {
            next if exists $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_FIELD_NAMES{$report_type}{$field_name};

            my $mandatory_group_by = $mandatory_group_by{$report_type};
            if (ref $mandatory_group_by eq 'CODE') {
                $mandatory_group_by = $mandatory_group_by->($field_name);
            }

            my $field_name_in_provider_response = {
                AdFormat => 'banner_image_type',
                AdGroupId => 'adgroup_id',
                AdGroupName => 'adgroup_name',
                AdId => 'bid',
                AdNetworkType => 'targettype',
                Age => 'age',
                AvgCpc => 'av_sum',
                AvgCpm => 'avg_cpm',
                AvgClickPosition => 'fp_clicks_avg_pos',
                AvgImpressionPosition => 'fp_shows_avg_pos',
                AvgImpressionFrequency => 'avg_view_freq',
                AvgPageviews => 'adepth',
                BounceRate => 'bounce_ratio',
                Bounces => 'bounces',
                CampaignId => 'cid',
                StrategyId => 'strategy_id',
                CampaignName => 'camp_name',
                CampaignUrlPath => 'camp_url_path',
                CampaignType => 'campaign_type',
                CarrierType => 'connection_type',
                ClickType => 'click_place',
                TargetingCategory => 'targeting_category',
                IncomeGrade => 'prisma_income_grade',
                Clicks => 'clicks',
                Conversions => 'agoalnum',
                ConversionRate => 'aconv',
                Cost => 'sum',
                CostPerConversion => 'agoalcost',
                Criteria => 'phrase',
                CriteriaId => 'bs_criterion_id',
                CriteriaType => 'contexttype',
                Criterion => 'phrase',
                CriterionId => 'bs_criterion_id',
                CriterionType => 'criterion_type',
                Ctr => 'ctr',
                Date => 'stat_date',
                Device => 'device_type',
                ExternalNetworkName => 'ssp',
                Gender => 'gender',
                GoalId => 'goal_id',
                GoalsRoi => 'agoalroi',
                ImpressionShare => 'winrate',
                Impressions => 'shows',
                LocationOfPresenceId => 'physical_region',
                LocationOfPresenceName => 'physical_region_name',
                MatchType => 'match_type',
                MatchedKeyword => 'matched_phrase',
                MobilePlatform => 'detailed_device_type',
                Month => 'stat_date',
                Placement => 'page_name',
                Query => 'search_query',
                Quarter => 'stat_date',
                ImpressionReach => 'uniq_viewers',
                Revenue => 'agoalincome',
                Sessions => 'asesnum',
                Slot => 'position',
                TargetingLocationId => 'region',
                TargetingLocationName => 'region_name',
                Week => 'stat_date',
                Year => 'stat_date',
                RlAdjustmentId => 'coef_ret_cond_id',
                WeightedCtr => 'ectr',
                WeightedImpressions => 'eshows',
                AvgTrafficVolume => 'avg_x',
                AvgTimeToConversion => 'avg_time_to_conv',
                Profit => 'agoals_profit',
                AvgEffectiveBid => 'avg_bid',
                ClientLogin => 'client_login',
            }->{$field_name};

            unless ($field_name_in_provider_response) {
                fail("$field_name is not covered");
                next;
            }

            $field_name_in_provider_response = $field_name_in_provider_response->{$report_type} if ref $field_name_in_provider_response eq 'HASH';

            my $expected_group_by = {
                AdFormat => ['banner_image_type'],
                AdGroupId => ['adgroup'],
                AdGroupName => ['adgroup'],
                AdId => ['banner'],
                AdNetworkType => ['targettype'],
                Age => ['age'],
                CampaignId => ['campaign'],
                CampaignName => ['campaign'],
                StrategyId => ['strategy_id'],
                CampaignUrlPath => ['campaign'],
                CampaignType => ['campaign_type'],
                CarrierType => ['connection_type'],
                ClickType => ['click_place'],
                TargetingCategory => ['targeting_category'],
                IncomeGrade => ['prisma_income_grade'],
                Criteria => {
                    CUSTOM_REPORT_TYPE() => ['contextcond_mod'],
                    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => ['contextcond_ext_mod'],
                },
                CriteriaId => {
                    CUSTOM_REPORT_TYPE() => ['contextcond_mod','adgroup'],
                    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => ['contextcond_ext_mod','adgroup'],
                },
                CriteriaType => ['contexttype_orig'],
                Criterion => {
                    CUSTOM_REPORT_TYPE() => ['contextcond_orig_mod'],
                    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => ['contextcond_orig_mod'],
                },
                CriterionId => {
                    CUSTOM_REPORT_TYPE() => ['contextcond_orig_mod','adgroup'],
                    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => ['contextcond_orig_mod','adgroup'],
                },
                CriterionType => ['criterion_type'],
                Date => ['date'],
                Device => ['device_type'],
                ExternalNetworkName => ['ssp'],
                Gender => ['gender'],
                ImpressionShare => ['campaign'],
                LocationOfPresenceId => ['physical_region'],
                LocationOfPresenceName => ['physical_region'],
                MatchType => ['match_type'],
                MatchedKeyword => ['matched_phrase'],
                MobilePlatform => ['detailed_device_type'],
                Month => ['date'],
                Placement => ['page_group'],
                Query => ['search_query'],
                Quarter => ['date'],
                Slot => ['position'],
                TargetingLocationId => ['region'],
                TargetingLocationName => ['region'],
                Week => ['date'],
                Year => ['date'],
                RlAdjustmentId => ['retargeting_coef'],
                ClientLogin => ['client_login'],
            }->{$field_name} // [];

            $expected_group_by = $expected_group_by->{$report_type} // $expected_group_by->{+CUSTOM_REPORT_TYPE} if ref $expected_group_by eq 'HASH';

            my $expected_date_aggregation_by = {
                Date => 'day',
                Month => 'month',
                Quarter => 'quarter',
                Week => 'week',
                Year => 'year',
            }->{$field_name} // 'none';

            my $guards = get_guards();
            my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

            my $request = $service->convert_client_request_to_internal_representation(
                build_request_data( {
                    Currency => 'RUB',
                    FieldNames => [$field_name],
                    ReportType => $report_type,
                } ),
                lang => 'en'
            );

            subtest $field_name => sub {
                check_request_class($request);
                ok( $request->need_to_request_report_from_provider, "must request something from provider" );

                $expected_group_by = bag(uniq xflatten($expected_group_by, $mandatory_group_by));

                cmp_deeply( $request->provider_request_params->{group_by}, $expected_group_by, 'group_by in provider request' );
                cmp_deeply( $request->provider_request_params->{date_aggregation_by}, $expected_date_aggregation_by, 'date_aggregation_by in provider request' );
                cmp_deeply( $request->displayed_field_names, [$field_name], 'displayed_field_names' );
                cmp_deeply( $request->field_names_in_provider_response, [$field_name_in_provider_response], 'field_names_in_provider_response' );

                my $with_winrate = $field_name_in_provider_response eq 'winrate' ? 1 : 0;
                is( !!$request->provider_request_params->{options}->{with_winrate}, !!$with_winrate, "proper with_winrate: $with_winrate" );
            };

        }
    }
};

subtest OrderBy => sub {
    my $field_name_internal_map = {
        AdFormat               => 'banner_image_type',
        AdGroupId              => 'adgroup',
        AdGroupName            => 'adgroup_name',
        AdId                   => 'banner',
        AdNetworkType          => 'targettype',
        Age                    => 'age',
        AvgCpc                 => 'av_sum',
        AvgCpm                 => 'avg_cpm',
        AvgClickPosition       => 'fp_clicks_avg_pos',
        AvgImpressionPosition  => 'fp_shows_avg_pos',
        AvgImpressionFrequency => 'avg_view_freq',
        AvgPageviews           => 'adepth',
        BounceRate             => 'bounce_ratio',
        Bounces                => 'bounces',
        CampaignId             => 'campaign',
        StrategyId             => 'strategy_id',
        CampaignName           => 'camp_name',
        CampaignUrlPath        => 'camp_url_path',
        CampaignType           => 'campaign_type',
        CarrierType            => 'connection_type',
        Clicks                 => 'clicks',
        ClickType              => 'click_place',
        TargetingCategory      => 'targeting_category',
        IncomeGrade            => 'prisma_income_grade',
        Conversions            => 'agoalnum',
        ConversionRate         => 'aconv',
        Cost                   => 'sum',
        CostPerConversion      => 'agoalcost',
        CriteriaType           => 'contexttype_orig',
        CriterionType          => 'criterion_type',
        Ctr                    => 'ctr',
        Date                   => 'date',
        Device                 => 'device_type',
        ExternalNetworkName    => 'ssp',
        Gender                 => 'gender',
        GoalsRoi               => 'agoalroi',
        ImpressionShare        => 'winrate',
        Impressions            => 'shows',
        LocationOfPresenceId   => 'physical_region_id',
        LocationOfPresenceName => 'physical_region',
        MatchType              => 'match_type',
        MatchedKeyword         => 'matched_phrase',
        MobilePlatform         => 'detailed_device_type',
        Month                  => 'date',
        Placement              => 'page_group',
        Query                  => 'search_query',
        Quarter                => 'date',
        ImpressionReach        => 'uniq_viewers',
        Revenue                => 'agoalincome',
        Sessions               => 'asesnum',
        Slot                   => 'position',
        TargetingLocationId    => 'region_id',
        TargetingLocationName  => 'region',
        Week                   => 'date',
        Year                   => 'date',
        RlAdjustmentId         => 'coef_ret_cond_id',
        WeightedCtr            => 'ectr',
        WeightedImpressions    => 'eshows',
        AvgTrafficVolume       => 'avg_x',
        AvgTimeToConversion => 'avg_time_to_conv',
        Profit => 'agoals_profit',
        AvgEffectiveBid        => 'avg_bid',
        ClientLogin        => 'client_login',
    };

    my $direction_internal_map = {
        ASCENDING => 'asc',
        DESCENDING => 'desc',
    };

    do {
        my $guards = get_guards();
        my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

        my $request = $service->convert_client_request_to_internal_representation(
            build_request_data( {}, ['OrderBy'] ), lang => 'en' );

        subtest "no OrderBy" => sub {
            check_request_class($request);
            ok( $request->need_to_request_report_from_provider, "must request something from provider" );

            ok( ! exists $request->provider_request_params->{limits}->{order_by}, 'no order_by in provider request' );
        };
    };

    foreach my $report_type (@REPORT_TYPES) {
        my @covered_fields;
        for my $order_by ( sort ( all_fields() ) ) {
            next if $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_ORDER_BY{$report_type}{$order_by};

            my $internal_field_name = $field_name_internal_map->{$order_by};
            unless (defined $internal_field_name) {
                fail("$order_by is not covered");
                next;
            }

            push @covered_fields, $order_by;

            subtest $order_by => sub {
                do {
                    my $guards = get_guards();
                    my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

                    my $request = $service->convert_client_request_to_internal_representation(
                        build_request_data( { OrderBy => [ { Field => $order_by } ], ReportType => $report_type } ), lang => 'en' );

                    subtest "no direction" => sub {
                        check_request_class($request);
                        ok( $request->need_to_request_report_from_provider, "must request something from provider" );

                        cmp_deeply( $request->provider_request_params->{limits}->{order_by},
                            [ { field => $internal_field_name, dir => 'asc' } ],
                            'order_by must match in provider request' );
                    };
                };

                for my $direction ( qw( ASCENDING DESCENDING ) ) {
                    my $internal_direction_name = $direction_internal_map->{$direction}
                        or die "no internal_direction_name for direction=$direction";

                    my $guards = get_guards();
                    my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

                    my $request = $service->convert_client_request_to_internal_representation(
                        build_request_data( { OrderBy => [ { Field => $order_by, SortOrder => $direction } ], ReportType => $report_type } ), lang => 'en' );

                    subtest $direction => sub {
                        check_request_class($request);
                        ok( $request->need_to_request_report_from_provider, "must request something from provider" );

                        cmp_deeply( $request->provider_request_params->{limits}->{order_by},
                            [ { field => $internal_field_name, dir => $internal_direction_name } ],
                            'order_by must match in provider request' );
                    };
                }
            };
        }

        my $order_by_pairs_iterator = natatime 2, @covered_fields;
        while ( my @order_by_fields = $order_by_pairs_iterator->() ) {
            next unless defined $order_by_fields[1];

            my @internal_field_names;
            for my $order_by (@order_by_fields) {
                die "no internal_field_name for order_by=$field_name_internal_map->{$order_by}"
                    unless $field_name_internal_map->{$order_by};

                push @internal_field_names, $field_name_internal_map->{$order_by};
            }

            next if $internal_field_names[0] eq $internal_field_names[1];

            my $guards = get_guards();
            my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

            my $request = $service->convert_client_request_to_internal_representation(
                build_request_data( {
                    OrderBy => [
                        { Field => $order_by_fields[0], SortOrder => 'DESCENDING' },
                        { Field => $order_by_fields[1], SortOrder => 'ASCENDING' },
                    ],
                    ReportType => $report_type,
                } ),
                lang => 'en'
            );

            subtest "$order_by_fields[0] DESCENDING, $order_by_fields[1] ASCENDING" => sub {
                check_request_class($request);
                ok( $request->need_to_request_report_from_provider, "must request something from provider" );

                cmp_deeply( $request->provider_request_params->{limits}->{order_by},
                    [
                        { field => $internal_field_names[0], dir => 'desc' },
                        { field => $internal_field_names[1], dir => 'asc' },
                    ],
                    'order_by must match in provider request' );
            };
        }
    }
};

subtest Filter => sub {
    my $translocal_params = { tree => 'api' };
    my %cases = (
        AdId => {
            EQUALS => {
                input => [1943928610],
                output => { banner => { eq => [1943928610] } },
                mock => {
                    'API::Service::Reports::_restrict_campaign_ids_by_ad_ids' => [{
                        args => [ignore, undef, [1943928610]],
                        result => [201],
                    }],
                },
                expected_oids => [1],
            },
            IN => {
                input => [ qw( 1943928601 1943928611 1943928615 1943928618 1943928620 ) ],
                output => { banner => { eq => [ qw( 1943928601 1943928611 1943928615 1943928618 1943928620 ) ] } },
                mock => {
                    'API::Service::Reports::_restrict_campaign_ids_by_ad_ids' => [{
                        args => [ignore, undef, [1943928601, 1943928611, 1943928615, 1943928618, 1943928620]],
                        result => [201,202,204],
                    }],
                },
                expected_oids => [1,2,4],
            },
            NOT_EQUALS => {
                input => [1943928604],
                output => { banner => { ne => [1943928604] } },
            },
            NOT_IN => {
                input => [ qw( 1943928593 1943928591 1943928616 1943928571 ) ],
                output => { banner => { ne => [ qw( 1943928593 1943928591 1943928616 1943928571 ) ] } },
            }
        },
        AdGroupId => {
            EQUALS => {
                input => [1190453],
                output => { adgroup => { eq => [1190453] } },
                mock => {
                    'API::Service::Reports::_restrict_campaign_ids_by_adgroup_ids' => [{
                        args => [ignore, undef, [1190453]],
                        result => [201],
                    }],
                },
                expected_oids => [1],
            },
            IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { adgroup => { eq => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
                mock => {
                    'API::Service::Reports::_restrict_campaign_ids_by_adgroup_ids' => [{
                        args => [ignore, undef, [1190453, 1632308, 1688537, 1881096, 1979844]],
                        result => [201,202,204],
                    }],
                },
                expected_oids => [1,2,4],
            },
            NOT_EQUALS => {
                input => [1190453],
                output => { adgroup => { ne => [1190453] } },
            },
            NOT_IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { adgroup => { ne => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            }
        },
        AdNetworkType => {
            EQUALS => {
                input => ['SEARCH'],
                output => { targettype => { eq => ['SEARCH'] } },
            },
            IN => {
                input => [ qw( SEARCH AD_NETWORK ) ],
                output => { targettype => { eq => [ qw( SEARCH AD_NETWORK ) ] } },
            },
        },
        Age => {
            EQUALS => {
                input => ['AGE_18_24'],
                output => { age => { eq => ['AGE_18_24'] } },
            },
            IN => {
                input => [ qw( AGE_18_24 AGE_35_44 ) ],
                output => { age => { eq => [ qw( AGE_18_24 AGE_35_44 ) ] } },
            },
            NOT_EQUALS => {
                input => ['AGE_18_24'],
                output => { age => { ne => ['AGE_18_24'] } },
            },
            NOT_IN => {
                input => [ qw( AGE_18_24 AGE_35_44 ) ],
                output => { age => { ne => [ qw( AGE_18_24 AGE_35_44 ) ] } },
            }
        },
        AvgCpc => {
            GREATER_THAN => {
                input => [10000000],
                output => { av_sum => { gt => 10 } },
            },
            LESS_THAN => {
                input => [10000000],
                output => { av_sum => { lt => 10 } },
            }
        },
        AvgCpm => {
            GREATER_THAN => {
                input => [10000000],
                output => { avg_cpm => { gt => 10 } },
            },
            LESS_THAN => {
                input => [10000000],
                output => { avg_cpm => { lt => 10 } },
            }
        },
        AvgClickPosition => {
            GREATER_THAN => {
                input => [0.5],
                output => { fp_clicks_avg_pos => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { fp_clicks_avg_pos => { lt => 0.5 } },
            }
        },
        CostPerConversion => {
            GREATER_THAN => {
                input => [10000000],
                output => { agoalcost => { gt => 10 } },
            },
            LESS_THAN => {
                input => [10000000],
                output => { agoalcost => { lt => 10 } },
            }
        },
        AvgImpressionFrequency => {
            GREATER_THAN => {
                input => [0.5],
                output => { avg_view_freq => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { avg_view_freq => { lt => 0.5 } },
            }
        },
        AvgImpressionPosition => {
            GREATER_THAN => {
                input => [0.5],
                output => { fp_shows_avg_pos => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { fp_shows_avg_pos => { lt => 0.5 } },
            }
        },
        AvgPageviews => {
            GREATER_THAN => {
                input => [0.5],
                output => { adepth => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { adepth => { lt => 0.5 } },
            }
        },
        BounceRate => {
            GREATER_THAN => {
                input => [0.5],
                output => { bounce_ratio => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { bounce_ratio => { lt => 0.5 } },
            }
        },
        CampaignId => {
            EQUALS => {
                input => [219],
                output => {},
                expected_oids => [19],
            },
            IN => {
                input => [ qw( 202 204 218 219 222 ) ],
                output => {},
                expected_oids => [ qw( 2 4 18 19 22 ) ],
            }
        },
        CampaignType => {
            EQUALS => {
                input => {
                    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE() => ['CPM_BANNER_CAMPAIGN'],
                    default => ['TEXT_CAMPAIGN'],
                },
                output => {},
                expected_oids => [qw( 1 2 4 18 19 22 33 43 57 )],
            },
            IN => {
                input => {
                    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE() => [ qw( CPM_BANNER_CAMPAIGN CPM_DEALS_CAMPAIGN ) ],
                    default => [ qw( TEXT_CAMPAIGN MOBILE_APP_CAMPAIGN ) ],
                },
                output => {},
                expected_oids => [ qw( 1 2 4 18 19 22 33 43 57 ) ],
            }
        },
        CarrierType => {
            EQUALS => {
                input => ['STATIONARY'],
                output => { connection_type => { eq => ['STATIONARY'] } },
            },
            IN => {
                input => [ qw( STATIONARY CELLULAR ) ],
                output => { connection_type => { eq => [ qw( STATIONARY CELLULAR ) ] } },
            },
            NOT_EQUALS => {
                input => ['STATIONARY'],
                output => { connection_type => { ne => ['STATIONARY'] } },
            },
            NOT_IN => {
                input => [ qw( STATIONARY CELLULAR ) ],
                output => { connection_type => { ne => [ qw( STATIONARY CELLULAR ) ] } },
            }
        },
        ClickType => {
            EQUALS => {
                input => ['SITELINK2'],
                output => { click_place => { eq => ['SITELINK2'] } },
            },
            IN => {
                input => [ qw( SITELINK2 MOBILE_APP_ICON CHAT TURBO_GALLERY PRODUCT_EXTENSIONS MARKET PROMO_EXTENSION ) ],
                output => { click_place => { eq => [ qw( SITELINK2 MOBILE_APP_ICON CHAT TURBO_GALLERY PRODUCT_EXTENSIONS MARKET PROMO_EXTENSION ) ] } },
            },
            NOT_EQUALS => {
                input => ['SITELINK2'],
                output => { click_place => { ne => ['SITELINK2'] } },
            },
            NOT_IN => {
                input => [ qw( SITELINK2 MOBILE_APP_ICON ) ],
                output => { click_place => { ne => [ qw( SITELINK2 MOBILE_APP_ICON ) ] } },
            }
        },
        TargetingCategory => {
            EQUALS => {
                input => ['EXACT'],
                output => { targeting_category => { eq => ['EXACT'] } },
            },
            IN => {
                input => [ qw( EXACT ALTERNATIVE COMPETITOR ACCESSORY BROADER ) ],
                output => { targeting_category => { eq => [ qw( EXACT ALTERNATIVE COMPETITOR ACCESSORY BROADER ) ] } },
            },
            NOT_EQUALS => {
                input => ['EXACT'],
                output => { targeting_category => { ne => ['EXACT'] } },
            },
            NOT_IN => {
                input => [ qw( EXACT ALTERNATIVE ) ],
                output => { targeting_category => { ne => [ qw( EXACT ALTERNATIVE ) ] } },
            }
        },
        IncomeGrade => {
            EQUALS => {
                input => ['ABOVE_AVERAGE'],
                output => { prisma_income_grade => { eq => ['ABOVE_AVERAGE'] } },
            },
            IN => {
                input => [ qw( OTHER ABOVE_AVERAGE HIGH VERY_HIGH ) ],
                output => { prisma_income_grade => { eq => [ qw( OTHER ABOVE_AVERAGE HIGH VERY_HIGH ) ] } },
            },
            NOT_EQUALS => {
                input => ['OTHER'],
                output => { prisma_income_grade => { ne => ['OTHER'] } },
            },
            NOT_IN => {
                input => [ qw( HIGH VERY_HIGH ) ],
                output => { prisma_income_grade => { ne => [ qw( HIGH VERY_HIGH ) ] } },
            }
        },
        Clicks => {
            EQUALS => {
                input => [53],
                output => { clicks => { eq => [53] } },
            },
            GREATER_THAN => {
                input => [53],
                output => { clicks => { gt => 53 } },
            },
            IN => {
                input => [ 53, 65 ],
                output => { clicks => { eq => [ 53, 65 ] } },
            },
            LESS_THAN => {
                input => [53],
                output => { clicks => { lt => 53 } },
            }
        },
        Conversions => {
            EQUALS => {
                input => [53],
                output => { agoalnum => { eq => [53]  } },
            },
            GREATER_THAN => {
                input => [53],
                output => { agoalnum => { gt => 53 } },
            },
            IN => {
                input => [ 53, 65 ],
                output => { agoalnum => { eq => [ 53, 65 ] } },
            },
            LESS_THAN => {
                input => [53],
                output => { agoalnum => { lt => 53 } },
            }
        },
        ConversionRate => {
            GREATER_THAN => {
                input => [0.5],
                output => { aconv => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { aconv => { lt => 0.5 } },
            }
        },
        Cost => {
            GREATER_THAN => {
                input => [10000000],
                output => { sum => { gt => 10 } },
            },
            LESS_THAN => {
                input => [10000000],
                output => { sum => { lt => 10 } },
            }
        },
        CriteriaType => {
            EQUALS => {
                input => ['AUDIENCE_TARGET'],
                output => { contexttype_orig => { eq => ['AUDIENCE_TARGET'] } },
            },
            IN => {
                input => [ qw( AUDIENCE_TARGET SMART_BANNER_FILTER ) ],
                output => { contexttype_orig => { eq => [ qw( AUDIENCE_TARGET SMART_BANNER_FILTER ) ] } },
            },
            NOT_EQUALS => {
                input => ['SYNONYM'],
                output => { contexttype_orig => { ne => ['SYNONYM'] } },
            },
            NOT_IN => {
                input => [ qw( RELATED_KEYWORD KEYWORD ) ],
                output => { contexttype_orig => { ne => [ qw( RELATED_KEYWORD KEYWORD ) ] } },
            },
        },
        CriterionType => {
            EQUALS => {
                input => ['RETARGETING'],
                output => { criterion_type => { eq => ['RETARGETING'] } },
            },
            IN => {
                input => [ qw( FEED_FILTER WEBPAGE_FILTER ) ],
                output => { criterion_type => { eq => [ qw( FEED_FILTER WEBPAGE_FILTER ) ] } },
            },
            NOT_EQUALS => {
                input => ['MOBILE_APP_CATEGORY'],
                output => { criterion_type => { ne => ['MOBILE_APP_CATEGORY'] } },
            },
            NOT_IN => {
                input => [ qw( RELATED_KEYWORD KEYWORD ) ],
                output => { criterion_type => { ne => [ qw( RELATED_KEYWORD KEYWORD ) ] } },
            },
        },

        Ctr => {
            GREATER_THAN => {
                input => [0.5],
                output => { ctr => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { ctr => { lt => 0.5 } },
            }
        },
        Device => {
            EQUALS => {
                input => ['DESKTOP'],
                output => { device_type => { eq => ['DESKTOP'] } },
            },
            IN => {
                input => [ qw( DESKTOP TABLET ) ],
                output => { device_type => { eq => [ qw( DESKTOP TABLET ) ] } },
            },
            NOT_EQUALS => {
                input => ['DESKTOP'],
                output => { device_type => { ne => ['DESKTOP'] } },
            },
            NOT_IN => {
                input => [ qw( DESKTOP TABLET ) ],
                output => { device_type => { ne => [ qw( DESKTOP TABLET ) ] } },
            }
        },
        DynamicTextAdTargetId => {
            EQUALS => {
                input => [1190453],
                output => { dynamic => { eq => [1190453] } },
            },
            IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { dynamic => { eq => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            },
            NOT_EQUALS => {
                input => [1190453],
                output => { dynamic => { ne => [1190453] } },
            },
            NOT_IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { dynamic => { ne => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            }
        },
        ExternalNetworkName => {
            EQUALS => {
                input => ['foo'],
                output => { ssp => { eq => ['foo'] } },
            },
            IN => {
                input => [ qw( foo bar ) ],
                output => { ssp => { eq => [ qw( foo bar ) ] } },
            },
            NOT_EQUALS => {
                input => ['foo'],
                output => { ssp => { ne => ['foo'] } },
            },
            NOT_IN => {
                input => [ qw( foo bar ) ],
                output => { ssp => { ne => [ qw( foo bar ) ] } },
            }
        },
        Gender => {
            EQUALS => {
                input => ['GENDER_MALE'],
                output => { gender => { eq => ['GENDER_MALE'] } },
            },
            IN => {
                input => [ qw( GENDER_MALE GENDER_FEMALE ) ],
                output => { gender => { eq => [ qw( GENDER_MALE GENDER_FEMALE ) ] } },
            },
            NOT_EQUALS => {
                input => ['GENDER_MALE'],
                output => { gender => { ne => ['GENDER_MALE'] } },
            },
            NOT_IN => {
                input => [ qw( GENDER_MALE GENDER_FEMALE ) ],
                output => { gender => { ne => [ qw( GENDER_MALE GENDER_FEMALE ) ] } },
            }
        },
        GoalId => {
            EQUALS => {
                input => [1190453],
                output => { goal_id => { eq => 1190453 } },
            }
        },
        GoalsRoi => {
            GREATER_THAN => {
                input => [0.5],
                output => { agoalroi => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { agoalroi => { lt => 0.5 } },
            }
        },
        AdFormat => {
            EQUALS => {
                input => ['IMAGE'],
                output => { banner_image_type => { eq => ['IMAGE'] } },
            },
            IN => {
                input => [ qw( IMAGE TEXT VIDEO ) ],
                output => { banner_image_type => { eq => [ qw/IMAGE TEXT VIDEO/ ] } },
            }
        },
        ImpressionShare => {
            GREATER_THAN => {
                input => [0.5],
                output => { winrate => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { winrate => { lt => 0.5 } },
            },
        },
        Impressions => {
            EQUALS => {
                input => [53],
                output => { shows => { eq => [53]  } },
            },
            GREATER_THAN => {
                input => [53],
                output => { shows => { gt => 53 } },
            },
            IN => {
                input => [ 53, 65 ],
                output => { shows => { eq => [ 53, 65 ] } },
            },
            LESS_THAN => {
                input => [53],
                output => { shows => { lt => 53 } },
            }
        },
        Keyword => {
            DOES_NOT_START_WITH_ALL_IGNORE_CASE => {
                input => [ qw( foo bar ) ],
                output => { phrase_orig => { not_starts_with => [ qw( foo bar ) ] } },
            },
            DOES_NOT_START_WITH_IGNORE_CASE => {
                input => ['foo'],
                output => { phrase_orig => { not_starts_with => ['foo'] } },
            },
            EQUALS => {
                input => ['foo'],
                output => { phrase_orig => { eq => ['foo'] } },
            },
            IN => {
                input => [ qw( foo bar ) ],
                output => { phrase_orig => { eq => [ qw( foo bar ) ] } },
            },
            NOT_EQUALS => {
                input => ['foo'],
                output => { phrase_orig => { ne => ['foo'] } },
            },
            NOT_IN => {
                input => [ qw( foo bar ) ],
                output => { phrase_orig => { ne => [ qw( foo bar ) ] } },
            },
            STARTS_WITH_ANY_IGNORE_CASE => {
                input => [ qw( foo bar ) ],
                output => { phrase_orig => { starts_with => [ qw( foo bar ) ] } },
            },
            STARTS_WITH_IGNORE_CASE => {
                input => ['foo'],
                output => { phrase_orig => { starts_with => ['foo'] } },
            }
        },
        LocationOfPresenceId => {
            EQUALS => {
                input => [111],
                output => { physical_region => { eq => [111,112,114] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111], $translocal_params],
                        result => [111,112,114],
                    }],
                },
            },
            IN => {
                input => [111,201],
                output => { physical_region => { eq => [111,112,114,218,243,256] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111,201], $translocal_params],
                        result => [111,112,114,218,243,256],
                    }],
                },

            },
            NOT_EQUALS => {
                input => [111],
                output => { physical_region => { ne => [111,112,114] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111], $translocal_params],
                        result => [111,112,114],
                    }],
                },
            },
            NOT_IN => {
                input => [111,201],
                output => { physical_region => { ne => [111,112,114,218,243,256] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111,201], $translocal_params],
                        result => [111,112,114,218,243,256],
                    }],
                },
            },
        },
        AudienceTargetId => {
            EQUALS => {
                input => [1190453],
                output => { bs_criterion_id => { eq => [1190453] } },
            },
            IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { bs_criterion_id => { eq => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            },
            NOT_EQUALS => {
                input => [1190453],
                output => { bs_criterion_id => { ne => [1190453] } },
            },
            NOT_IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { bs_criterion_id => { ne => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            }
        },
        MatchType => {
            EQUALS => {
                input => ['SYNONYM'],
                output => { match_type => { eq => ['SYNONYM'] } },
            },
            IN => {
                input => [ qw( RELATED_KEYWORD EXPERIMENT ) ],
                output => { match_type => { eq => [ qw( RELATED_KEYWORD EXPERIMENT ) ] } },
            },
            NOT_EQUALS => {
                input => ['SYNONYM'],
                output => { match_type => { ne => ['SYNONYM'] } },
            },
            NOT_IN => {
                input => [ qw( RELATED_KEYWORD EXPERIMENT ) ],
                output => { match_type => { ne => [ qw( RELATED_KEYWORD EXPERIMENT ) ] } },
            }
        },
        MatchedKeyword => {
            DOES_NOT_START_WITH_ALL_IGNORE_CASE => {
                input => [ qw( foo bar ) ],
                output => { matched_phrase => { not_starts_with => [ qw( foo bar ) ] } },
            },
            DOES_NOT_START_WITH_IGNORE_CASE => {
                input => ['foo'],
                output => { matched_phrase => { not_starts_with => ['foo'] } },
            },
            EQUALS => {
                input => ['foo'],
                output => { matched_phrase => { eq => ['foo'] } },
            },
            IN => {
                input => [ qw( foo bar ) ],
                output => { matched_phrase => { eq => [ qw( foo bar ) ] } },
            },
            NOT_EQUALS => {
                input => ['foo'],
                output => { matched_phrase => { ne => ['foo'] } },
            },
            NOT_IN => {
                input => [ qw( foo bar ) ],
                output => { matched_phrase => { ne => [ qw( foo bar ) ] } },
            },
            STARTS_WITH_ANY_IGNORE_CASE => {
                input => [ qw( foo bar ) ],
                output => { matched_phrase => { starts_with => [ qw( foo bar ) ] } },
            },
            STARTS_WITH_IGNORE_CASE => {
                input => ['foo'],
                output => { matched_phrase => { starts_with => ['foo'] } },
            }
        },
        MobilePlatform => {
            EQUALS => {
                input => ['IOS'],
                output => { detailed_device_type => { eq => ['IOS'] } },
            },
            IN => {
                input => [ qw( IOS ANDROID ) ],
                output => { detailed_device_type => { eq => [ qw( IOS ANDROID ) ] } },
            },
            NOT_EQUALS => {
                input => ['IOS'],
                output => { detailed_device_type => { ne => ['IOS'] } },
            },
            NOT_IN => {
                input => [ qw( IOS ANDROID ) ],
                output => { detailed_device_type => { ne => [ qw( IOS ANDROID ) ] } },
            }
        },
        Placement => {
            EQUALS => {
                input => ['foo'],
                output => { page => { eq => [ qw( 123 456 ) ] } },
            },
            IN => {
                input => [ qw( foo bar ) ],
                output => { page => { eq => [ qw( 123 456 789 ) ] } },
            },
            NOT_EQUALS => {
                input => ['foo'],
                output => { page => { ne => [ qw( 123 456 ) ] } },
            },
            NOT_IN => {
                input => [ qw( foo bar ) ],
                output => { page => { ne => [ qw( 123 456 789 ) ] } },
            }
        },
        ImpressionReach => {
            EQUALS => {
                input => [53],
                output => { uniq_viewers => { eq => [53] } },
            },
            GREATER_THAN => {
                input => [53],
                output => { uniq_viewers => { gt => 53 } },
            },
            IN => {
                input => [ 53, 65 ],
                output => { uniq_viewers => { eq => [ 53, 65 ] } },
            },
            LESS_THAN => {
                input => [53],
                output => { uniq_viewers => { lt => 53 } },
            }
        },
        Revenue => {
            GREATER_THAN => {
                input => [10000000],
                output => { agoalincome => { gt => 10 } },
            },
            LESS_THAN => {
                input => [10000000],
                output => { agoalincome => { lt => 10 } },
            }
        },
        Slot => {
            EQUALS => {
                input => ['PREMIUMBLOCK'],
                output => { position => { eq => ['PREMIUMBLOCK'] } },
            },
            IN => {
                input => [ qw( PREMIUMBLOCK OTHER ALONE SUGGEST PRODUCT_GALLERY ) ],
                output => { position => { eq => [ qw( PREMIUMBLOCK OTHER ALONE SUGGEST PRODUCT_GALLERY ) ] } },
            },
            NOT_EQUALS => {
                input => ['PREMIUMBLOCK'],
                output => { position => { ne => ['PREMIUMBLOCK'] } },
            },
            NOT_IN => {
                input => [ qw( PREMIUMBLOCK OTHER ALONE SUGGEST PRODUCT_GALLERY ) ],
                output => { position => { ne => [ qw( PREMIUMBLOCK OTHER ALONE SUGGEST PRODUCT_GALLERY ) ] } },
            }
        },
        SmartBannerFilterId => {
            EQUALS => {
                input => [1190453],
                output => { performance => { eq => [1190453] } },
            },
            IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { performance => { eq => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            },
            NOT_EQUALS => {
                input => [1190453],
                output => { performance => { ne => [1190453] } },
            },
            NOT_IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { performance => { ne => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            }
        },
        SmartAdTargetId => {
            EQUALS => {
                input => [1190453],
                output => { performance => { eq => [1190453] } },
            },
            IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { performance => { eq => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            },
            NOT_EQUALS => {
                input => [1190453],
                output => { performance => { ne => [1190453] } },
            },
            NOT_IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { performance => { ne => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            }
        },
        TargetingLocationId => {
            EQUALS => {
                input => [111],
                output => { region_id => { eq => [111,112,114] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111], $translocal_params],
                        result => [111,112,114],
                    }],
                },
            },
            IN => {
                input => [111,201],
                output => { region_id => { eq => [111,112,114,218,243,256] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111,201], $translocal_params],
                        result => [111,112,114,218,243,256],
                    }],
                },
            },
            NOT_EQUALS => {
                input => [111],
                output => { region_id => { ne => [111,112,114] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111], $translocal_params],
                        result => [111,112,114],
                    }],
                },
            },
            NOT_IN => {
                input => [111,201],
                output => { region_id => { ne => [111,112,114,218,243,256] } },
                mock => {
                    'Stat::Tools::get_plus_regions_by_geo' => [{
                        args => [[111,201], $translocal_params],
                        result => [111,112,114,218,243,256],
                    }],
                },
            },
        },
        RlAdjustmentId => {
            EQUALS => {
                input => [1190453],
                output => { retargeting_coef => { eq => [1190453] } },
            },
            IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { retargeting_coef => { eq => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            },
            NOT_EQUALS => {
                input => [1190453],
                output => { retargeting_coef => { ne => [1190453] } },
            },
            NOT_IN => {
                input => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ],
                output => { retargeting_coef => { ne => [ qw( 1190453 1632308 1688537 1881096 1979844 ) ] } },
            }
        },
        Query => {
            DOES_NOT_START_WITH_ALL_IGNORE_CASE => {
                input => [ qw( foo bar ) ],
                output => { search_query => { not_starts_with => [ qw( foo bar ) ], ne => [ '__SIGNIFICANT_EMPTY_STRING__' ] } },
            },
            DOES_NOT_START_WITH_IGNORE_CASE => {
                input => ['foo'],
                output => { search_query => { not_starts_with => ['foo'], ne => [ '__SIGNIFICANT_EMPTY_STRING__' ] } },
            },
            EQUALS => {
                input => ['foo'],
                output => { search_query => { eq => ['foo'], ne => [ '__SIGNIFICANT_EMPTY_STRING__' ] } },
            },
            IN => {
                input => [ qw( foo bar ) ],
                output => { search_query => { eq => [ qw( foo bar ) ], ne => [ '__SIGNIFICANT_EMPTY_STRING__' ] } },
            },
            NOT_EQUALS => {
                input => ['foo'],
                output => { search_query => { ne => [ qw(foo __SIGNIFICANT_EMPTY_STRING__) ] } },
            },
            NOT_IN => {
                input => [ qw( foo bar ) ],
                output => { search_query => { ne => [ qw( foo bar __SIGNIFICANT_EMPTY_STRING__) ] } },
            },
            STARTS_WITH_ANY_IGNORE_CASE => {
                input => [ qw( foo bar ) ],
                output => { search_query => { starts_with => [ qw( foo bar ) ], ne => [ '__SIGNIFICANT_EMPTY_STRING__' ] } },
            },
            STARTS_WITH_IGNORE_CASE => {
                input => ['foo'],
                output => { search_query => { starts_with => ['foo'], ne => [ '__SIGNIFICANT_EMPTY_STRING__' ] } },
            },
         },
        WeightedCtr => {
            GREATER_THAN => {
                input => [0.5],
                output => { ectr => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { ectr => { lt => 0.5 } },
            }
        },
        WeightedImpressions => {
            GREATER_THAN => {
                input => [0.5],
                output => { eshows => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { eshows => { lt => 0.5 } },
            }
        },
        AvgTrafficVolume => {
            GREATER_THAN => {
                input => [0.5],
                output => { avg_x => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { avg_x => { lt => 0.5 } },
            }
        },
        AvgTimeToConversion => {
            GREATER_THAN => {
                input => [0.5],
                output => { avg_time_to_conv => { gt => 0.5 } },
            },
            LESS_THAN => {
                input => [0.5],
                output => { avg_time_to_conv => { lt => 0.5 } },
            }
        },
        Profit => {
            GREATER_THAN => {
                input => [-10000000],
                output => { agoals_profit => { gt => -10 } },
            },
            LESS_THAN => {
                input => [10000000],
                output => { agoals_profit => { lt => 10 } },
            }
        },
        AttributionModel => {
           EQUALS => {
               input  => [ 'FC' ],
               output => { attribution_model => { eq => 'FC' } }
           }
        },
        AvgEffectiveBid => {
            GREATER_THAN => {
                input => [10000000],
                output => { avg_bid => { gt => 10 } },
            },
            LESS_THAN => {
                input => [10000000],
                output => { avg_bid => { lt => 10 } },
            }
        },
        ClientLogin => {
            EQUALS => {
                input => ['l1'],
                output => { client_id => { eq => [101] } },
                mock => {
                    'API::Service::Reports::_get_accessible_client_ids' => [{ args => ignore, result => [101] }],
                },
            },
            IN => {
                input => [ qw( l1 l2 l3 ) ],
                output => { client_id => { eq => [ 102, 103, 104 ] } },
                mock => {
                    'API::Service::Reports::_get_accessible_client_ids' => [{ args => ignore, result => [102, 103, 104] }],
                },
            }
        },
        StrategyId => {
            EQUALS => {
                input  => [ 101 ],
                output => { strategy_id => { eq => [101]} }
            }
        }
    );
    lock_hash_recurse(%cases);

    foreach my $report_type ( @REPORT_TYPES ) {
        my $mandatory_filter = {
            AD_PERFORMANCE_REPORT_TYPE() => {},
            CAMPAIGN_PERFORMANCE_REPORT_TYPE() => {},
            CUSTOM_REPORT_TYPE() => {},
            SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => {
                search_query => { ne => [ '__SIGNIFICANT_EMPTY_STRING__' ] },
                targettype => { eq => 'SEARCH' },
            },
        }->{$report_type} // {};
        for my $field ( sort ( all_fields() ) ) {
            next if $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_FILTER{$report_type}{$field};

            my $field_cases = $cases{$field};
            unless ( $field_cases && %$field_cases ) {
                fail("$field is not covered");
                next;
            }

            subtest $report_type .' - '. $field => sub {
                for my $operator ( sort keys %$field_cases ) {
                    my $guards = get_guards();
                    my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

                    if (exists $field_cases->{$operator}{mock}) {
                        mock_subs(%{yclone($field_cases->{$operator}{mock})});
                    }

                    my $request = $service->convert_client_request_to_internal_representation(
                        build_request_data( {
                            SelectionCriteria => {
                                Filter => [ {
                                    Field => $field,
                                    Operator => $operator,
                                    Values => ref $field_cases->{$operator}->{input} eq 'HASH'
                                    ? exists $field_cases->{$operator}->{input}{$report_type}
                                        ? $field_cases->{$operator}->{input}{$report_type}
                                        : $field_cases->{$operator}->{input}{default}
                                    : $field_cases->{$operator}->{input}
                                } ]
                            },
                            DateRangeType => 'LAST_WEEK',
                            ReportType => $report_type,
                        } ),
                        lang => 'en'
                    );

                    subtest $report_type .' - '. $field .' - '. $operator => sub {
                        check_request_class($request);
                        ok( $request->need_to_request_report_from_provider, "must request something from provider" );

                        die "Don't know what filter to expect, field=$field, operator=$operator" unless exists $field_cases->{$operator}{output};
                        my $expected_filter = { %$mandatory_filter, %{$field_cases->{$operator}->{output}} };
                        cmp_deeply(
                            $request->provider_request_params->{filter},
                            $expected_filter,
                            'filter must match in provider request' );

                        cmp_deeply(
                            $request->provider_request_params->{oid},
                            exists $field_cases->{$operator}->{expected_oids} ? $field_cases->{$operator}->{expected_oids} : [ qw( 1 2 4 18 19 22 33 43 57 ) ],
                            "oids must match in provider response" );
                    };

                    if (exists $field_cases->{$operator}->{mock}) {
                        restore_mocked_subs();
                    }
                }
            };
        }
    }

    subtest complex_filter_by_ids => sub {
        my $guards = get_guards();
        my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

        mock_subs('API::Service::Reports',
            '_restrict_campaign_ids_by_ad_ids' => [{
                args => [ignore, [201,202,204,210,211], [1943928601, 1943928611, 1943928615, 1943928618, 1943928620]],
                result => [201,202,204],
            }],
            '_restrict_campaign_ids_by_adgroup_ids' => [{
                args => [ignore, [201,202,204], [1190453, 1632308, 1688537, 1881096, 1979844]],
                result => [202,204],
            }],
        );
        my $request = $service->convert_client_request_to_internal_representation(
            build_request_data( {
                SelectionCriteria => {
                    Filter => [
                        {
                            Field => "CampaignId",
                            Operator => "IN",
                            Values => [201,202,204,210,211]
                        },
                        {
                            Field => "AdId",
                            Operator => "IN",
                            Values => [1943928601, 1943928611, 1943928615, 1943928618, 1943928620]
                        },
                        {
                            Field => "AdGroupId",
                            Operator => "IN",
                            Values => [1190453, 1632308, 1688537, 1881096, 1979844]
                        },

                    ]
                },
                DateRangeType => 'LAST_WEEK',
                ReportType => CUSTOM_REPORT_TYPE,
            } ),
            lang => 'en'
        );

        check_request_class($request);
        ok( $request->need_to_request_report_from_provider, "must request something from provider" );

        cmp_deeply(
            $request->provider_request_params->{filter},
            {
                banner => { eq => [1943928601, 1943928611, 1943928615, 1943928618, 1943928620] },
                adgroup => { eq => [1190453, 1632308, 1688537, 1881096, 1979844] }
            },
            'filter must match in provider request' );

        cmp_deeply(
            $request->provider_request_params->{oid},
            [ qw( 2 4 ) ],
            "oids must match in provider response" );
        restore_mocked_subs();
    };
};

subtest DateRangeType => sub {
    do {
        my $guards = get_guards();
        my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

        my $request = $service->convert_client_request_to_internal_representation(
            build_request_data( {
                SelectionCriteria => {
                    Filter => [],
                    DateFrom => '2016-05-01',
                    DateTo => '2016-06-01',
                },
                DateRangeType => 'CUSTOM_DATE',
            } ),
            lang => 'en'
        );

        subtest 'CUSTOM_DATE' => sub {
            check_request_class($request);
            ok( $request->need_to_request_report_from_provider, "must request something from provider" );
            cmp_deeply(
                $request->provider_request_params,
                superhashof( { start_date => date_matcher('2016-05-01'), end_date => date_matcher('2016-06-01') } ),
                'start_date, end_date must match in provider request' );
        };
    };

    do {
        my $guards = get_guards();
        my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

        my $request = $service->convert_client_request_to_internal_representation(
            build_request_data( {
                SelectionCriteria => { Filter => [] },
                DateRangeType => 'ALL_TIME',
            } ),
            lang => 'en'
        );

        subtest 'ALL_TIME' => sub {
            check_request_class($request);
            ok( $request->need_to_request_report_from_provider, "must request something from provider" );
            ok( ! exists $request->provider_request_params->{start_date}, "must not have start_date in provider request" );
            ok( ! exists $request->provider_request_params->{end_date}, "must not have end_date in provider request" );
        };
    };

    # date_range_type => [ case, case, ... ]
    # case --- это { set_time => '2016-06-01 09:50:00', expect_range => [ start_date, end_date ] }
    my %cases = (
        TODAY => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-06-01', '2016-06-01' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-06-01', '2016-06-01' ] },
        ],
        YESTERDAY => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-31', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-31', '2016-05-31' ] },
        ],
        LAST_3_DAYS => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-29', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-29', '2016-05-31' ] },
        ],
        LAST_5_DAYS => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-27', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-27', '2016-05-31' ] },
        ],
        LAST_7_DAYS => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-25', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-25', '2016-05-31' ] },
        ],
        LAST_14_DAYS => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-18', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-18', '2016-05-31' ] },
        ],
        LAST_30_DAYS => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-02', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-02', '2016-05-31' ] },
        ],
        LAST_90_DAYS => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-03-03', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-03-03', '2016-05-31' ] },
        ],
        LAST_365_DAYS => [
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2015-06-02', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2015-06-02', '2016-05-31' ] },
        ],
        LAST_WEEK => [
            # среда
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-23', '2016-05-29' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-23', '2016-05-29' ] },

            # понедельник
            { set_time => '2016-06-06 00:00:00', expect_range => [ '2016-05-30', '2016-06-05' ] },
            { set_time => '2016-06-06 09:50:00', expect_range => [ '2016-05-30', '2016-06-05' ] },

            # воскресенье
            { set_time => '2016-06-05 00:00:00', expect_range => [ '2016-05-23', '2016-05-29' ] },
            { set_time => '2016-06-05 09:50:00', expect_range => [ '2016-05-23', '2016-05-29' ] },
        ],
        LAST_BUSINESS_WEEK => [
            # среда
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-23', '2016-05-27' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-23', '2016-05-27' ] },

            # пятница
            { set_time => '2016-06-03 00:00:00', expect_range => [ '2016-05-23', '2016-05-27' ] },
            { set_time => '2016-06-03 09:50:00', expect_range => [ '2016-05-23', '2016-05-27' ] },

            # воскресенье
            { set_time => '2016-06-05 00:00:00', expect_range => [ '2016-05-30', '2016-06-03' ] },
            { set_time => '2016-06-05 09:50:00', expect_range => [ '2016-05-30', '2016-06-03' ] },

            # понедельник
            { set_time => '2016-06-06 00:00:00', expect_range => [ '2016-05-30', '2016-06-03' ] },
            { set_time => '2016-06-06 09:50:00', expect_range => [ '2016-05-30', '2016-06-03' ] },
        ],
        THIS_MONTH => [
            # начало месяца
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-06-01', '2016-06-01' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-06-01', '2016-06-01' ] },

            # середина месяца
            { set_time => '2016-06-15 00:00:00', expect_range => [ '2016-06-01', '2016-06-15' ] },
            { set_time => '2016-06-15 09:50:00', expect_range => [ '2016-06-01', '2016-06-15' ] },

            # конец месяца
            { set_time => '2016-06-30 00:00:00', expect_range => [ '2016-06-01', '2016-06-30' ] },
            { set_time => '2016-06-30 09:50:00', expect_range => [ '2016-06-01', '2016-06-30' ] },
        ],
        LAST_MONTH => [
            # начало месяца
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-01', '2016-05-31' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-01', '2016-05-31' ] },

            # середина месяца
            { set_time => '2016-06-15 00:00:00', expect_range => [ '2016-05-01', '2016-05-31' ] },
            { set_time => '2016-06-15 09:50:00', expect_range => [ '2016-05-01', '2016-05-31' ] },

            # конец месяца
            { set_time => '2016-06-30 00:00:00', expect_range => [ '2016-05-01', '2016-05-31' ] },
            { set_time => '2016-06-30 09:50:00', expect_range => [ '2016-05-01', '2016-05-31' ] },
        ],
        THIS_WEEK_SUN_TODAY => [
            # среда
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-29', '2016-06-01' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-29', '2016-06-01' ] },

            # понедельник
            { set_time => '2016-06-06 00:00:00', expect_range => [ '2016-06-05', '2016-06-06' ] },
            { set_time => '2016-06-06 09:50:00', expect_range => [ '2016-06-05', '2016-06-06' ] },

            # воскресенье
            { set_time => '2016-06-05 00:00:00', expect_range => [ '2016-06-05', '2016-06-05' ] },
            { set_time => '2016-06-05 09:50:00', expect_range => [ '2016-06-05', '2016-06-05' ] },
        ],
        THIS_WEEK_MON_TODAY => [
            # среда
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-30', '2016-06-01' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-30', '2016-06-01' ] },

            # понедельник
            { set_time => '2016-06-06 00:00:00', expect_range => [ '2016-06-06', '2016-06-06' ] },
            { set_time => '2016-06-06 09:50:00', expect_range => [ '2016-06-06', '2016-06-06' ] },

            # воскресенье
            { set_time => '2016-06-05 00:00:00', expect_range => [ '2016-05-30', '2016-06-05' ] },
            { set_time => '2016-06-05 09:50:00', expect_range => [ '2016-05-30', '2016-06-05' ] },
        ],
        LAST_WEEK_SUN_SAT => [
            # среда
            { set_time => '2016-06-01 00:00:00', expect_range => [ '2016-05-22', '2016-05-28' ] },
            { set_time => '2016-06-01 09:50:00', expect_range => [ '2016-05-22', '2016-05-28' ] },

            # понедельник
            { set_time => '2016-06-06 00:00:00', expect_range => [ '2016-05-29', '2016-06-04' ] },
            { set_time => '2016-06-06 09:50:00', expect_range => [ '2016-05-29', '2016-06-04' ] },

            # воскресенье
            { set_time => '2016-06-05 00:00:00', expect_range => [ '2016-05-29', '2016-06-04' ] },
            { set_time => '2016-06-05 09:50:00', expect_range => [ '2016-05-29', '2016-06-04' ] },
        ],
    );
    lock_hash_recurse(%cases);

    for my $date_range_type ( sort( all_date_range_types() ) ) {
        ## CUSTOM_DATE, ALL_TIME покрыты в подтестах выше
        ## AUTO покрыт в date_range_types_type_auto.t
        next if $date_range_type eq 'CUSTOM_DATE' ||
            $date_range_type eq 'ALL_TIME' ||
            $date_range_type eq 'AUTO';

        my $cases_for_type = $cases{$date_range_type};
        unless ( $cases_for_type && @$cases_for_type ) {
            fail("$date_range_type not covered");
            next;
        }

        subtest $date_range_type => sub {
            for my $case (@$cases_for_type) {
                my $guards = get_guards( set_time => $case->{set_time} );
                my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

                my $request = $service->convert_client_request_to_internal_representation(
                    build_request_data( {
                        SelectionCriteria => { Filter => [] },
                        DateRangeType => $date_range_type,
                    } ),
                    lang => 'en'
                );

                subtest $case->{set_time} => sub {
                    check_request_class($request);
                    ok( $request->need_to_request_report_from_provider, "must request something from provider" );

                    my ( $start_date, $end_date ) = @{ $case->{expect_range} };

                    cmp_deeply(
                        $request->provider_request_params,
                        superhashof( { start_date => date_matcher($start_date), end_date => date_matcher($end_date) } ),
                        'start_date, end_date must match in provider request' );
                };
            }
        };
    }
};

subtest 'VAT and Discount' => sub {
    for my $include_vat ( qw( YES NO ) ) {
        for my $include_discount ( qw( YES NO ) ) {
            my $guards = get_guards();
            my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

            my $request = $service->convert_client_request_to_internal_representation(
                build_request_data( {
                    # в $SAMPLE_REQUEST уже RUB, но в будущем может стать не RUB;
                    # для этого теста принципиально важно, чтобы там были не фишки
                    Currency => 'RUB',

                    IncludeVAT => $include_vat,
                    IncludeDiscount => $include_discount,
                } ),
                lang => 'en'
            );

            subtest "IncludeVAT=$include_vat, IncludeDiscount=$include_discount" => sub {
                check_request_class($request);
                ok( $request->need_to_request_report_from_provider, "must request something from provider" );

                cmp_deeply(
                    $request->provider_request_params->{options},
                    superhashof( {
                        with_nds => $include_vat eq 'YES' ? 1 : 0,
                        with_discount => 0,
                    } ),
                    'options must have proper with_nds, with_discount in provider request'
                );
            };
        }
    }
};

subtest Page => sub {
    do {
        my $guards = get_guards();
        my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

        my $request = $service->convert_client_request_to_internal_representation(
            build_request_data( {}, ['Page'] ), lang => 'en' );

        subtest "no limit" => sub {
            check_request_class($request);
            ok( $request->need_to_request_report_from_provider, "must request something from provider" );

            is( $request->provider_request_params->{limits}->{limit}, 1000000, 'default limit of 1000000 rows in provider request' );
        };
    };

    for my $limit ( qw( 1 2 5 10 25 ) ) {
        my $guards = get_guards();
        my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

        my $request = $service->convert_client_request_to_internal_representation(
            build_request_data( { Page => { Limit => $limit } } ), lang => 'en' );

        subtest "limit=$limit" => sub {
            check_request_class($request);
            ok( $request->need_to_request_report_from_provider, "must request something from provider" );

            is( $request->provider_request_params->{limits}->{limit}, $limit, 'limit must match in provider request' );
        };
    }
};

subtest ReportName => sub {
    for my $report_name ( qw( Report Отчёт ), 'a' x 255 ) {
        my $guards = get_guards();
        my $service = API::Test::Reports::FakeService::Reports->new( $user_login, $request_http_headers );

        my $request = $service->convert_client_request_to_internal_representation(
            build_request_data( { ReportName => $report_name } ), lang => 'en' );

        subtest encode_utf8($report_name) => sub {
            check_request_class($request);
            ok( $request->need_to_request_report_from_provider, "must request something from provider" );

            is( $request->report_name, $report_name, "report name is the same" );
        };
    }
};

sub build_request_data {
    my ( $additional_data, $params_to_delete ) = @_;

    my $request_data = hash_merge( {}, yclone($SAMPLE_REQUEST), $additional_data );

    for my $param ( @{ $params_to_delete // [] } ) {
        delete $request_data->{$param};
    }

    return $request_data;
}

sub check_request_class {
    my ($request) = @_;
    my $class = 'API::Reports::InternalRequestRepresentation';
    my $ok = Scalar::Util::blessed $request && $request->isa($class);
    ok( $ok, "internal request needs to be a $class object" );
    Test::More::diag( Data::Dumper::Dumper($request) ) unless $ok;
}

sub get_guards {
    my %opts = @_;

    $API::Test::Reports::FakeService::Reports::NO_REAL_CALLS = 1;
    @API::Test::Reports::FakeService::Reports::UNAVAILABLE_CIDS = @{ $unavailable_cids };
    $API::Test::Reports::FakeService::Reports::CIDS_WITH_STATISTICS = yclone( $all_cids_with_statistics );
    $API::Test::Reports::FakeService::Reports::SUBLCIENT_CURRENCY = $subclient_currency;

    $API::Test::Reports::FakeUser::NO_REAL_CALLS = 1;
    %API::Test::Reports::FakeUser::USER_MAP = %{ $user_ids };

    $API::Test::Reports::FakeRBAC::NO_REAL_CALLS = 1;
    %API::Test::Reports::FakeRBAC::ALLOW_SHOW_STAT_CAMPS = %{ $rbac_allow_show_stat_camps };

    $API::Test::Reports::FakeCampaignIdLookup::NO_REAL_CALLS = 1;
    %API::Test::Reports::FakeCampaignIdLookup::CID_TO_ORDERID = %{ $cid_to_orderid_map };

    $API::Test::Reports::FakePageIdMap::NO_REAL_CALLS = 1;
    %API::Test::Reports::FakePageIdMap::PAGE_NAME_TO_PAGE_IDS = %$page_name_to_page_ids_map;

    my $guards = {
        'API::Test::Reports::FakeService::Reports' => API::Test::Reports::FakeService::Reports->get_override_guard,
        'API::Test::Reports::FakeUser' => API::Test::Reports::FakeUser->get_override_guard,
        'API::Test::Reports::FakeRBAC' => API::Test::Reports::FakeRBAC->get_override_guard,
        'API::Test::Reports::FakeCampaignIdLookup' => API::Test::Reports::FakeCampaignIdLookup->get_override_guard,
        'API::Test::Reports::FakePageIdMap' => API::Test::Reports::FakePageIdMap->get_override_guard,
    };

    if ( exists $opts{set_time} ) {
        set_fixed_time( $opts{set_time} // '2016-06-01 00:00:00', '%Y-%m-%d %H:%M:%S');
        $guards->{time} = guard { restore_time() };
    }

    return $guards;
}

sub date_matcher {
    my ($date) = @_;

    if ( $date =~ /^(\d{4})-(\d{2})-(\d{2})$/ ) {
        return any( $date, "$1-$2-$3" );
    }

    die "Invalid date: $date";
}
