#!/usr/bin/perl
use Direct::Modern;
use my_inc '../../../..', for => 'unit_tests';

use Test::More tests => 12;
use Test::Deep;

use API::Reports::DataRules qw/
    %OPERATORS_MAP_TO_INTERNAL
    %FILTER_PARAMS_MAP
    %REPORT_TYPE_CONF
    %GROUP_BY_FIELD_MAP
    %DATE_AGGREGATION_MAP
    %RESULT_FIELD_MAP
    %RESULT_MAP_BY_FIELD
    %MONEY_FIELD_MAP
    %MONEY_INTERNAL_FIELD_MAP

    :types
/;


use API::Test::MockHelper qw/mock_subs restore_mocked_subs/;
use API::Test::Reports::Fields qw/
REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE
CUSTOM_REPORT_TYPE
/;

use API::Service::Reports::ConvertSubs qw/
        convert_format
        convert_filter_items
        convert_field_names_and_report_type_to_group_by
        convert_field_names_to_date_aggregation
        convert_field_names_to_report_fields
        convert_field_names_to_countable_fields
        convert_yes_no
        process_report_row
        convert_order_by_items
        convert_page_filter
        convert_page_name_filters
        convert_none
/;

use API::Reports::ReportRowProcessor 'process_report_row';

use constant FALSE => 0;
use constant TRUE => 1;

subtest 'convert_format' => sub {
    is(convert_format('TSV'), 'tsv', 'internal value for "TSV" should be "tsv"');

    is(convert_format('JSON'), 'unknown', 'internal value for "JSON" should be "unknown"');

    is(convert_format('XML'), 'unknown', 'internal value for "XML" should be "unknown"');

    is(convert_format('XXX'), 'unknown', 'internal value for other formats should be "unknown"');
};


subtest 'convert_filter_items' => sub {
    SKIP: {
        skip('assumptions failed', 4) unless subtest 'assumptions' => sub {
            is($OPERATORS_MAP_TO_INTERNAL{'EQUALS'}, 'eq');
            cmp_deeply($FILTER_PARAMS_MAP{+CUSTOM_REPORT_TYPE}{'Clicks'}, superhashof {
                name => 'clicks',
            });
            cmp_deeply( $FILTER_PARAMS_MAP{+CUSTOM_REPORT_TYPE}{'Cost'}, superhashof {
                name => 'sum',
            });
            cmp_deeply( $FILTER_PARAMS_MAP{+CUSTOM_REPORT_TYPE}{'AdFormat'}, superhashof {
                name => 'banner_image_type',
                enum => bag(qw/TEXT ADAPTIVE_IMAGE IMAGE VIDEO SMART_SINGLE SMART_MULTIPLE SMART_VIDEO SMART_TILE/),
            });
            #cmp_deeply( $FILTER_PARAMS_MAP{+CUSTOM_REPORT_TYPE}{'GoalId'}, superhashof {
            #    name => 'goal_id',
            #    single_value => 1,
            #});
        };

        # assertions
        cmp_deeply(
            convert_filter_items([{operator =>'EQUALS', field => 'Clicks', values => [11]}], CUSTOM_REPORT_TYPE),
            {clicks => {'eq' => [11]}},
            'internal representation of filter should be valid'
        );
        cmp_deeply(
            convert_filter_items([{operator =>'EQUALS', field => 'Cost', values => [2000000]}], CUSTOM_REPORT_TYPE),
            {sum => {'eq' => [2]}},
            'money fields should converted from micros to float'
        );
        cmp_deeply(
            convert_filter_items([{operator =>'EQUALS', field => 'AdFormat', values => ['TEXT']}], CUSTOM_REPORT_TYPE),
            {banner_image_type => {'eq' => ['TEXT']}},
            'values of fields with value map should be converted through that map'
        );
        #cmp_deeply(
        #    convert_filter_items([{operator =>'EQUALS', field => 'GoalId', values => [123]}], CUSTOM_REPORT_TYPE),
        #    {goal_id => {'eq' => 123}},
        #    'values of field with flag "single_value" should be converted to scalar'
        #);

    }
};


subtest 'convert_field_names_and_report_type_to_group_by' => sub {
    SKIP: {
        cmp_deeply(
            convert_field_names_and_report_type_to_group_by([], CUSTOM_REPORT_TYPE),
            [],
            'CUSTOM_REPORT type should not add group_by field'
        );

        skip('group by AdGroupId not supported', 1) unless eq_deeply(\%GROUP_BY_FIELD_MAP, superhashof { AdGroupId => 'adgroup' });

        cmp_deeply(
            convert_field_names_and_report_type_to_group_by(['AdGroupId'], CUSTOM_REPORT_TYPE),
            ['adgroup'],
            'fields with group_by param should invoke grouping'
        );

        skip('ADGROUP_PERFORMANCE_REPORT not supported', 2)
            unless eq_deeply(\%REPORT_TYPE_CONF, superhashof { ADGROUP_PERFORMANCE_REPORT => { group_by => ['adgroup'] } });

        cmp_deeply(
            convert_field_names_and_report_type_to_group_by([], 'ADGROUP_PERFORMANCE_REPORT'),
            ['adgroup'],
            'other report types should add group_by field'
        );
        cmp_deeply(
            convert_field_names_and_report_type_to_group_by(['AdGroupId'], 'ADGROUP_PERFORMANCE_REPORT'),
            ['adgroup'],
            'grouping invoked by field and report type should not duplicate'
        );
    }
};


subtest 'convert_field_names_to_date_aggregation' => sub {
    SKIP: {
        skip('assumptions failed', 4) unless subtest 'assumptions' => sub {
            cmp_deeply({%DATE_AGGREGATION_MAP}, superhashof {
                Date => 'day',
                Week => 'week',
            });
            is( $DATE_AGGREGATION_MAP{'Clicks'}, undef);
        };

        is(convert_field_names_to_date_aggregation([]), 'none', 'empty list of fields should not invoke date_aggreagation');
        is(convert_field_names_to_date_aggregation(['Clicks', 'Date']), 'day', 'date-field should invoke proper date_aggregation');
        is(convert_field_names_to_date_aggregation(['Clicks']), 'none', 'non-date-fields should not invoke date_aggregation');
        is(convert_field_names_to_date_aggregation(['Week', 'Date']), 'week', 'only the first date-field should invoke date_aggregation');
    }
};


subtest 'convert_field_names_to_report_fields' => sub {
    SKIP: {
        skip('assumptions failed', 4) unless subtest 'assumptions' => sub {
            cmp_deeply($RESULT_FIELD_MAP{+CUSTOM_REPORT_TYPE}, superhashof {
                Ctr => 'ctr',
                Cost => 'sum',
            });
        };

        cmp_deeply(convert_field_names_to_report_fields([], CUSTOM_REPORT_TYPE), [], 'empty list should be converted to empty list');
        cmp_deeply(convert_field_names_to_report_fields(['Ctr', 'Cost'], CUSTOM_REPORT_TYPE), ['ctr', 'sum'], 'exteral fields should be converted to proper internal fields with the same order');
    }
};


subtest 'convert_yes_no' => sub {
    is(convert_yes_no('YES'), 1, 'YES should be converted to 1');
    is(convert_yes_no('NO'), 0, 'NO should be converted to 0');
    is(convert_yes_no('xxx'), 0, 'unknown value should be converted to 0');
};

subtest 'convert_none' => sub {
    is(convert_none('NONE'), 0, 'NONE should be converted to 0');
    is(convert_none('11111'), '11111', 'other values should not be converted');

};

subtest 'convert_order_by_items' => sub {
    cmp_deeply(
        convert_order_by_items([{ field => 'Impressions', dir => 'ASCENDING' }], CUSTOM_REPORT_TYPE),
        [{field => 'shows', dir => 'asc'}],
        'ASCENDING should be converted to asc'
    );
    cmp_deeply(
        convert_order_by_items([{ field => 'Clicks', dir => 'DESCENDING' }], CUSTOM_REPORT_TYPE),
        [{field => 'clicks', dir => 'desc'}],
        'DESCENDING should be converted to desc'
    );
    cmp_deeply(
        convert_order_by_items([{ field => 'Ctr' }], CUSTOM_REPORT_TYPE),
        [{field => 'ctr', dir => 'asc'}],
        'unspecified sort order should be converted to desc'
    );
};

sub get_processor {
    my ($field, $need_conversion_to_micro_units, $report_type) = @_;

    return API::Reports::ReportRowProcessor->new(
        requested_fields => defined $field ? [$field] : [],
        report_type => $report_type // CUSTOM_REPORT_TYPE,
        need_conversion_to_micro_units => $need_conversion_to_micro_units,
    );
}

## это тест, что process_report_row правильно работает с большими таблицами в ConvertSubs;
## что сами таблицы правильные, проверяет process_report_row.t
subtest 'process_report_row' => sub {
    SKIP: {
        skip('assumptions failed', 8) unless subtest 'assumptions' => sub {
                ok( exists $MONEY_INTERNAL_FIELD_MAP{+CUSTOM_REPORT_TYPE}{sum} );
        };

        cmp_deeply( get_processor()->process_report_row( { xxx => 1 } ), [ ],
            'unrequested fields should be ignored' );
        cmp_deeply( get_processor('phrase')->process_report_row( { phrase => 'xxx', PhraseID => 1 } ), [ '' ],
            'value of field phrase should be replaced by empty string if PhraseID <= 1' );
        cmp_deeply( get_processor('criteria_id')->process_report_row( { PhraseID => 1 } ), [ '--' ],
            'value of field phrase should be replaced by empty string if PhraseID <= 1' );
        cmp_deeply( get_processor('sum', TRUE)->process_report_row( { sum => 1 } ),
            [ 1000000 ],
            'money fields should be converted to micro units if flag need_money_fields_in_micros is set' );
        cmp_deeply( get_processor('sum', FALSE)->process_report_row( { sum => 1 } ), [ 1 ],
            'money fields should not be converted to micro units if flag need_money_fields_in_micros is not set' );
        cmp_deeply( get_processor('ctr')->process_report_row( { ctr => 1 } ), [ 1 ],
            'other fields should be returned without changes' );

        cmp_deeply( get_processor('region_name')->process_report_row( { region_name => 'aa',
                    region_not_exactly_detected       => 1,
                    original_region_name              =>
                    'bb' } ), [ 'bb' ],
            'original region name should be restored' );

        cmp_deeply( get_processor('physical_region_name')->process_report_row( { physical_region_name => 'xx',
                    physical_region_not_exactly_detected       => 1,
                    original_physical_region_name              =>
                    'zz' } ), [ 'zz' ],
            'original physical_region_name should be restored' );

        cmp_deeply(
            get_processor('region_name')->process_report_row( { region_name => 'aa' } )
            , [ 'aa' ],
            'region name should be returned unmodified if it is exactly detected' );
    }
};


subtest 'convert_page_filter' => sub {
    mock_subs(
        'Stat::Tools::convert_page_name_filter_to_ids' => [{
            args => [['aa', 'bb'], 'eq', 'ru'],
            result => [1, 2],
        }],
    );
    cmp_deeply(convert_page_filter({eq=>['aa', 'bb']}, 'ru'), {eq=>[1, 2]},
               'should call get_page_id_by_name with page names and use returned values as filter values');
    restore_mocked_subs();
};

subtest 'convert_page_name_filters' => sub {
    cmp_deeply(convert_page_name_filters({eq => ['aa', 'bb']}), {contains => ['aa', 'bb']});
    cmp_deeply(convert_page_name_filters({ne => ['aa', 'bb']}), {not_contains => ['aa', 'bb']});
};

subtest 'convert_field_names_to_countable_fields' => sub {
    cmp_deeply(
        convert_field_names_to_countable_fields([qw/AvgCpm AvgImpressionFrequency ImpressionReach/], REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE),
        [qw/avg_cpm avg_view_freq uniq_viewers/],
        'extra countable fields only'
    );
    cmp_deeply(
        convert_field_names_to_countable_fields([qw/AvgCpc AvgImpressionFrequency/], REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE),
        [qw/avg_view_freq/],
        'extra countable fields mixed'
    );
    cmp_deeply(
        convert_field_names_to_countable_fields([qw/AvgCpm AvgImpressionFrequency ImpressionReach/], CUSTOM_REPORT_TYPE),
        [],
        'extra countable fields with unsupported report type'
    );

};
