#!/usr/bin/perl
use Direct::Modern;

use Test::Deep;
use Test::Exception;
use Test::More tests => 6;

use my_inc '../../../..', for => 'api/t';

use API::Service::Reports::ConvertMap;

my @CALLED_FUNCS = qw(
    convert_format
    convert_yes_no
);

my %call_table;

do {
    no strict 'refs';
    no warnings 'redefine';

    for my $func (@CALLED_FUNCS) {
        my $fully_qualified_name = "API::Service::Reports::ConvertMap::$func";
        *$fully_qualified_name = sub {
            $call_table{$func} = [@_];
            return $_[0];
        };
    }
};

subtest build_to_internal_map => sub {
    plan tests => 2;

    do {
        my $converter = API::Service::Reports::ConvertMap->new( method => 'create' );
        cmp_deeply(
            $converter->to_internal_map,
            {
                SelectionCriteria => ignore(),
                Goals             => ignore(),
                AttributionModels => ignore(),
                FieldNames        => ignore(),
                DateRangeType     => ignore(),
                ReportName        => ignore(),
                ReportType        => ignore(),
                Currency          => ignore(),
                Format            => ignore(),
                IncludeVAT        => ignore(),
                OrderBy           => ignore(),
                Page              => ignore(),
            },
            'valid map for create',
        );
    };

    dies_ok {
        my $converter = API::Service::Reports::ConvertMap->new( method => 'bogus' );
        $converter->to_internal_map;
    } 'cannot build map for bogus method';
};

subtest _convert_selection_criteria => sub {
    plan tests => 2;

    do {
        %call_table = ();
        my $result = API::Service::Reports::ConvertMap::_convert_selection_criteria( {
            DateFrom => '2016-01-01',
            DateTo => '2016-02-01',
            Filter => [ { Field => 'Impressions', Operator => 'GREATER_THAN', Values => [0] } ]
        } );

        cmp_deeply(
            { result => $result, call_table => \%call_table },
            {
                result => {
                    start_date => '2016-01-01',
                    end_date => '2016-02-01',
                    filter => [ { field => 'Impressions', operator => 'GREATER_THAN', values => [0] } ],
                },
                call_table => {},
            },
            'with filter',
        );
    };

    do {
        %call_table = ();
        my $result = API::Service::Reports::ConvertMap::_convert_selection_criteria( {
            DateFrom => '2016-01-01',
            DateTo => '2016-02-01',
        } );

        cmp_deeply(
            { result => $result, call_table => \%call_table },
            {
                result => {
                    end_date => '2016-02-01',
                    start_date => '2016-01-01',
                },
                call_table => {},
            },
            'without filter',
        );
    };
};

do {
    %call_table = ();
    my $result = API::Service::Reports::ConvertMap::_convert_page( { Limit => 10 } );

    cmp_deeply(
        { result => $result, call_table => \%call_table },
        {
            result => { limit => 10 },
            call_table => {},
        },
        '_convert_page',
    );

    $result = API::Service::Reports::ConvertMap::_convert_page( { Limit => 10, Offset => 20 } );

    cmp_deeply(
        { result => $result, call_table => \%call_table },
        {
            result => { limit => 10, offset => 20 },
            call_table => {},
        },
        '_convert_page with offset',
    );

};

do {
    %call_table = ();
    my $result = API::Service::Reports::ConvertMap::_convert_goals([ 1, 2, 3 ]);

    cmp_deeply(
        { result => $result, call_table => \%call_table },
        {
            result => { goal_ids => { field => 'Goals', operator => "IN", values => [ 1, 2, 3 ] }},
            call_table => {},
        },
        '_convert_goals',
    );
};

do {
    %call_table = ();
    my $result = API::Service::Reports::ConvertMap::_convert_attribution_models([ 'LC', 'LSC' ]);

    cmp_deeply(
        { result => $result, call_table => \%call_table },
        {
            result => { attribution_models => { field => 'AttributionModels', operator => "IN", values => [ 'LC', 'LSC' ] }},
            call_table => {},
        },
        '_convert_attribution_models',
    );
};
