#!/usr/bin/perl
use Direct::Modern;

use Test::Deep;
use Test::More tests => 7;

use Yandex::DateTime;
use Yandex::DBTools;
use Yandex::DBUnitTest ':all';

use my_inc '../../../..', for => 'api/t';
use API::Reports::DataRules;
use Settings;

copy_table( PPCDICT, 'stat_rollbacks' );

my $tested_sub = $API::Reports::DataRules::DATE_RANGE_TYPE_MAP{'AUTO'};

my $now = now();

my %dates = (
    today => $now->ymd,
    yesterday => $now->clone->subtract( days => 1 )->ymd,
    three_days_ago => $now->clone->subtract( days => 3 )->ymd,
    month_ago => $now->clone->subtract( months => 1 )->ymd,
    two_months_ago => $now->clone->subtract( months => 2 )->ymd,
);

do_test(
    description => 'no rollbacks',
    data => [],
    OrderIds => [ qw( 1 2 ) ],
    expected_dates => [ @dates{ qw( three_days_ago yesterday ) } ],
);

do_test(
    description => 'a rollback earlier than yesterday',
    data => [
        { OrderID => 1, rollback_time => $dates{three_days_ago}, border_date => $dates{month_ago} },
    ],
    OrderIds => [ qw( 1 2 ) ],
    expected_dates => [ @dates{ qw( three_days_ago yesterday ) } ],
);

do_test(
    description => 'a rollback today',
    data => [
        { OrderID => 1, rollback_time => $dates{today}, border_date => $dates{month_ago} },
    ],
    OrderIds => [ qw( 1 2 ) ],
    expected_dates => [ @dates{ qw( three_days_ago yesterday ) } ],
);

do_test(
    description => 'a rollback yesterday, going back to yesterday',
    data => [
        { OrderID => 1, rollback_time => $dates{yesterday}, border_date => $dates{yesterday} },
    ],
    OrderIds => [ qw( 1 2 ) ],
    expected_dates => [ @dates{ qw( three_days_ago yesterday ) } ],
);

do_test(
    description => 'a rollback yesterday, going back a month ago',
    data => [
        { OrderID => 1, rollback_time => $dates{yesterday}, border_date => $dates{month_ago} },
    ],
    OrderIds => [ qw( 1 2 ) ],
    expected_dates => [ @dates{ qw( month_ago yesterday ) } ],
);

do_test(
    description => 'two rollbacks yesterday, going back a month and two months ago',
    data => [
        { OrderID => 1, rollback_time => $dates{yesterday}, border_date => $dates{month_ago} },
        { OrderID => 2, rollback_time => $dates{yesterday}, border_date => $dates{two_months_ago} },
    ],
    OrderIds => [ qw( 1 2 ) ],
    expected_dates => [ @dates{ qw( two_months_ago yesterday ) } ],
);

do_test(
    description => 'two rollbacks yesterday and today, going back a month and two months ago',
    data => [
        { OrderID => 1, rollback_time => $dates{yesterday}, border_date => $dates{month_ago} },
        { OrderID => 2, rollback_time => $dates{today}, border_date => $dates{two_months_ago} },
    ],
    OrderIds => [ qw( 1 2 ) ],
    expected_dates => [ @dates{ qw( month_ago yesterday ) } ],
);

sub do_test {
    my %args = @_;

    do_sql( PPCDICT, 'TRUNCATE TABLE stat_rollbacks' );

    if ( $args{data} && @{ $args{data} } ) {
        do_mass_insert_sql( PPCDICT, 'INSERT INTO stat_rollbacks (OrderID, rollback_time, border_date) VALUES %s',
            [ map { [ @$_{ qw( OrderID rollback_time border_date ) } ] } @{ $args{data} } ] );
    }

    cmp_deeply( [ $tested_sub->( $now->clone, $args{OrderIds} ) ], $args{expected_dates}, $args{description} );
}
