#!/usr/bin/perl
use Direct::Modern;
use 5.010;

use Template;
use Test::Deep;
use Test::More tests => 64;

use my_inc '../../../..', for => 'api/t';
use API::Reports::ReportRowProcessor;
use API::Service::Reports::ConvertSubs 'convert_field_names_to_report_fields';
use API::Test::Reports::Fields qw/
    @REPORT_TYPES
    all_fields
    CUSTOM_REPORT_TYPE
    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE
    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE
/;
use Settings;

my @NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES = (0, 1);
my $need_money_fields_in_micros;

my %field_covered;

# field => {
#     provider_name => '...',
#     type => 'int|float|date|money|str|str_take_original|mapped_enum',
#     map => { bs_value => 'value_client_sees' },
# }
my %field_type = (
    AdGroupId => { provider_name => 'adgroup_id', type => 'id' },
    AdGroupName => { provider_name => 'adgroup_name', type => 'str' },
    AdId => { provider_name => 'bid', type => 'id' },
    AdNetworkType => { provider_name => 'targettype', type => 'str', },
    Age => { provider_name => 'age', type => 'str', },
    AvgClickPosition => { provider_name => 'fp_clicks_avg_pos', type => 'float' },
    AvgCpc => { provider_name => 'av_sum', type => 'money' },
    AvgCpm => { provider_name => 'avg_cpm', type => 'money', report_type => REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE },
    AvgImpressionPosition => { provider_name => 'fp_shows_avg_pos', type => 'float' },
    AvgImpressionFrequency => { provider_name => 'avg_view_freq', type => 'float', report_type => REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE },
    AvgPageviews => { provider_name => 'adepth', type => 'float' },
    BounceRate => { provider_name => 'bounce_ratio', type => 'float' },
    Bounces => { provider_name => 'bounces', type => 'int' },
    CampaignId => { provider_name => 'cid', type => 'id' },
    StrategyId => { provider_name => 'strategy_id', type => 'id'},
    CampaignName => { provider_name => 'camp_name', type => 'str' },
    CampaignUrlPath => { provider_name => 'camp_url_path', type => 'str' },

    CampaignType => { provider_name => 'campaign_type', type => 'str', },

    CarrierType => { provider_name => 'connection_type', type => 'str', },

    ClickType => { provider_name => 'click_place', type => 'str', },
    TargetingCategory => {provider_name => 'targeting_category', type => 'str', },
    IncomeGrade => {provider_name => 'prisma_income_grade', type => 'str', },
    Clicks => { provider_name => 'clicks', type => 'int' },
    Conversions => { provider_name => 'agoalnum', type => 'int' },
    ConversionRate => { provider_name => 'aconv', type => 'float' },
    Cost => { provider_name => 'sum', type => 'money' },
    CostPerConversion => { provider_name => 'agoalcost', type => 'money' },
    Criteria => {
        provider_name => 'phrase',
        type => 'str',
        additional_fields => { ContextType => 1 },
        additional_tests => sub {
            my ($field_name_in_provider_response) = @_;
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                foreach my $report_type (@REPORT_TYPES) {
                    next if $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_FIELD_NAMES{$report_type}{Criteria};
                    my $is_empty = $report_type ne SEARCH_QUERY_PERFORMANCE_REPORT_TYPE;
                    my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
                    cmp_deeply(
                        $processor->process_report_row( { $field_name_in_provider_response => 'foo bar baz', PhraseID => 0 } ),
                        [$is_empty ? '' : 'foo bar baz'],
                        ($is_empty ? 'empty' : 'not empty')." for $report_type if PhraseID <= 1 in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
                    );
                }
            }

            return;
        }
    },
    CriteriaId => {
        provider_name => 'bs_criterion_id',
        type => 'id',
        additional_fields => { ContextType => 1 },
        additional_tests => sub {
            my ($field_name_in_provider_response) = @_;
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros);
                cmp_deeply(
                    $processor->process_report_row(
                        { $field_name_in_provider_response => 123, PhraseID => undef },
                    ),
                    ['--'],
                    "hyphens if PhraseID <= 1 in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
                );
            }

            return;
        }
    },
    CriteriaType => { provider_name => 'contexttype', type => 'str', },
    Criterion => { provider_name => 'phrase', type => 'str' },
    CriterionId => { provider_name => 'bs_criterion_id', type => 'id' },
    CriterionType => { provider_name => 'criterion_type', type => 'str', },
    Ctr => { provider_name => 'ctr', type => 'float' },
    Date => { provider_name => 'stat_date', type => 'date' },
    Device => { provider_name => 'device_type', type => 'str', },
    ExternalNetworkName => { provider_name => 'ssp', type => 'str' },
    Gender => { provider_name => 'gender', type => 'str', },
    GoalId => { provider_name => 'goal_id', type => 'id' },
    GoalsRoi => { provider_name => 'agoalroi', type => 'float' },
    AdFormat => { provider_name => 'banner_image_type', type => 'str' },
    ImpressionShare => { provider_name => 'winrate', type => 'float' },
    Impressions => { provider_name => 'shows', type => 'int' },
    LocationOfPresenceId => { provider_name => 'physical_region', type => 'region_id' },
    LocationOfPresenceName => {
        provider_name => 'physical_region_name',
        type => 'str_take_original',
        not_exactly_detected_field_name => 'physical_region_not_exactly_detected',
    },
    MatchType => { provider_name => 'match_type', type => 'str', },
    MatchedKeyword => { provider_name => 'matched_phrase', type => 'str', report_type => SEARCH_QUERY_PERFORMANCE_REPORT_TYPE },
    MobilePlatform => { provider_name => 'detailed_device_type', type => 'str', },
    Month => { provider_name => 'stat_date', type => 'date' },
    Placement => { provider_name => 'page_name', type => 'str' },
    Quarter => { provider_name => 'stat_date', type => 'date' },
    Query => { provider_name => 'search_query', type => 'str', report_type => SEARCH_QUERY_PERFORMANCE_REPORT_TYPE },
    ImpressionReach => { provider_name => 'uniq_viewers', type => 'int', report_type => REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE },
    Profit => { provider_name => 'agoals_profit', type => 'money' },
    Revenue => { provider_name => 'agoalincome', type => 'money' },
    Slot => { provider_name => 'position', type => 'str', },
    TargetingLocationId => { provider_name => 'region', type => 'region_id' },
    TargetingLocationName => {
        provider_name => 'region_name',
        type => 'str_take_original',
        not_exactly_detected_field_name => 'region_not_exactly_detected',
    },
    Week => { provider_name => 'stat_date', type => 'date' },
    Year => { provider_name => 'stat_date', type => 'date' },
    Sessions => { provider_name => 'asesnum', type => 'int' },
    RlAdjustmentId => { provider_name => 'coef_ret_cond_id', type => 'id' },
    WeightedCtr => { provider_name => 'ectr', type => 'float' },
    WeightedImpressions => { provider_name => 'eshows', type => 'float' },
    AvgTrafficVolume => { provider_name => 'avg_x', type => 'float' },
    AvgTimeToConversion => { provider_name => 'avg_time_to_conv', type => 'float' },
    AttributionModel => { provider_name => 'attribution_model', type => 'str' },
    AvgEffectiveBid => { provider_name => 'avg_bid', type => 'money' },
    ClientLogin => { provider_name => 'client_login', type => 'str' },
);

for my $field ( sort ( all_fields() ) ) {
    my $desc = $field_type{$field};
    my $report_type = $desc->{report_type} // CUSTOM_REPORT_TYPE;

    next if $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_FIELD_NAMES{$report_type}{$field};

    unless ($desc) {
        fail("$field is not covered");
        next;
    }

    subtest $field => sub {
        die "no provider_name for $field" unless $desc->{provider_name};

        my ($field_name_in_provider_response) = @{ convert_field_names_to_report_fields([$field], $report_type) };
        is( $field_name_in_provider_response, $desc->{provider_name}, 'field_name_in_provider_response' );

        return unless $desc->{type};

        $field_covered{$field} = 1;

        my %additional_fields = %{ $desc->{additional_fields} // {} };

        my $additional_tests = sub {
            if ( $desc->{additional_tests} ) {
                $desc->{additional_tests}->($field_name_in_provider_response);
            }
        };

        if ( $desc->{type} eq 'int' || $desc->{type} eq 'region_id' ) {
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => 123, %additional_fields },
                ),
                [123],
                "same as $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => 0, %additional_fields },
                ),
                [0],
                "wrong zero value for $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => undef, %additional_fields },
                ),
                ['--'],
                "wrong default value for $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );

            }

            $additional_tests->();

            return;
        }

        if ( $desc->{type} eq 'float' ) {
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => 0.123, %additional_fields },
                ),
                [0.123],
                "same as $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => undef, %additional_fields },
                ),
                [ '--' ],
                "wrong default value for $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );
            }

            $additional_tests->();

            return;
        }

        if ( $desc->{type} eq 'date' ) {
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => '2016-06-01', %additional_fields },
                ),
                ['2016-06-01'],
                "same as $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );
            }

            $additional_tests->();

            return;
        }

        if ( $desc->{type} eq 'money' ) {
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => 123, %additional_fields },
                ),
                [$need_money_fields_in_micros ? 123000000 : 123],
                "same as $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => undef, %additional_fields },
                ),
                ['--'],
                "wrong default value for $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
            );

            }

            $additional_tests->();

            return;
        }

        if ( $desc->{type} eq 'str' ) {
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
                cmp_deeply(
                    $processor->process_report_row(
                        { $field_name_in_provider_response => 'foo bar baz', %additional_fields },
                    ),
                    ['foo bar baz'],
                    "same as $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
                );
                cmp_deeply(
                    $processor->process_report_row(
                        { $field_name_in_provider_response => undef, %additional_fields },
                    ),
                    ['--'],
                    "wrong default value for $field_name_in_provider_response in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
                );
            }

            $additional_tests->();

            return;
        }

        if ( $desc->{type} eq 'str_take_original' ) {
            subtest 'exactly detected' => sub {
                foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                    my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
                cmp_deeply(
                    $processor->process_report_row(
                        {
                            $field_name_in_provider_response => "foo bar",
                            "original_$field_name_in_provider_response" => "baz quux",
                            $desc->{not_exactly_detected_field_name} => 0,
                            %additional_fields
                        },
                    ),
                    ["foo bar"],
                    "not same as original_$field_name_in_provider_response and same as $field_name_in_provider_response " .
                        "in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
                );
                }
            };

            subtest 'not exactly detected' => sub {
                foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                    my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
                cmp_deeply(
                    $processor->process_report_row(
                        {
                            $field_name_in_provider_response => "foo bar",
                            "original_$field_name_in_provider_response" => "baz quux",
                            $desc->{not_exactly_detected_field_name} => 1,
                            %additional_fields
                        },
                    ),
                    ["baz quux"],
                    "same as original_$field_name_in_provider_response and not the same as $field_name_in_provider_response " .
                        "in provider response [need_money_fields_in_micros=$need_money_fields_in_micros]"
                );
                }
            };

            $additional_tests->();

            return;
        }

        if ( $desc->{type} eq 'mapped_enum' ) {
            foreach $need_money_fields_in_micros (@NEED_MONEY_FIELDS_IN_MICROS_TEST_VALUES) {
                subtest "need_money_fields_in_micros=$need_money_fields_in_micros" => sub {
                    my $processor = get_processor($field_name_in_provider_response, $need_money_fields_in_micros, $report_type);
                    my %map = ( 'invalid_value' => '--', %{ $desc->{map} } );
                    for my $provider_value ( sort keys %map ) {
                        cmp_deeply(
                            $processor->process_report_row({ $field_name_in_provider_response => $provider_value, %additional_fields }),
                            [ $map{$provider_value} ],
                            "$provider_value in provider_response/$field_name_in_provider_response => $map{$provider_value} in client response"
                        );
                    }
                };
            }

            $additional_tests->();

            return;
        }

        if ( $desc->{type} eq 'id' ) {
            my $processor = get_processor($field_name_in_provider_response, 0, $report_type);
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => 123, %additional_fields },
                ),
                [123],
                "same as $field_name_in_provider_response in provider response"
            );
            cmp_deeply(
                $processor->process_report_row(
                    { $field_name_in_provider_response => 0, %additional_fields },
                ),
                ['--'],
                "'--' if $field_name_in_provider_response = 0"
            );

            $additional_tests->();

            return;
        }

        die "invalid type: $desc->{type}";
    };
}

sub get_processor {
    my ($field, $need_conversion_to_micro_units, $report_type) = @_;

    return API::Reports::ReportRowProcessor->new(
        requested_fields => [ $field ],
        report_type => $report_type // CUSTOM_REPORT_TYPE,
        need_conversion_to_micro_units => $need_conversion_to_micro_units
    );
}
