#!/usr/bin/perl
use Direct::Modern;

use Test::Deep;
use Test::More tests => 3;

use my_inc '../../../..', for => 'api/t';
use API::Reports::InternalRequestRepresentation;
use API::Reports::DataRules qw/
    CUSTOM_REPORT_TYPE
/;

use API::Service::Reports::ProcessingModeChooser;
use constant TEST_APPLICATION_ID => 'ae99016820074f809e5c268e564bebad';

my @MOBILE_APP_IDS = ('a4286bedd73840fb966a41277ffcbf40', '74a699cdfe004d6ca576cfe4a8bfd79a');
my @ALL_APP_IDS = (@MOBILE_APP_IDS, TEST_APPLICATION_ID);

{
    no warnings 'redefine';
    *Property::get = sub { undef };
}

subtest '_require_offline_mode' => sub {
    plan tests => 11;

    my $ok_request_data = { report_type => 'CUSTOM_REPORT', date_range_type => 'CUSTOM_DATE' };
    my $ok_provider_params = { start_date => '2016-12-01', end_date => '2016-12-03', oid => [ qw( 1 ) ] };
    my $ok_internal_rep = {
        need_to_request_report_from_provider => 1,
        lang => 'ru',
        provider_request_params => $ok_provider_params,
        displayed_field_names => [],
        field_names_in_provider_response => [],
        return_money_in_micros => 0,
        response_format => 'tsv',
        report_name => 'testReport',
        report_type => 'CUSTOM_REPORT',
    };

    ok(
        !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
            $_,
            $ok_request_data, API::Reports::InternalRequestRepresentation->new(%$ok_internal_rep) ),
        'online report OK for a base request with app_id '.$_
    ) foreach @ALL_APP_IDS;

    ok(
        API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
            $_,
            { %$ok_request_data, report_type => 'SEARCH_QUERY_PERFORMANCE_REPORT' },
            API::Reports::InternalRequestRepresentation->new( %$ok_internal_rep, report_type => 'SEARCH_QUERY_PERFORMANCE_REPORT' ) ),
        'online report not OK for a SEARCH_QUERY_PERFORMANCE_REPORT request with app_id'.$_
    ) foreach (TEST_APPLICATION_ID);

    ok(
        !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
            $_,
            { %$ok_request_data, report_type => 'SEARCH_QUERY_PERFORMANCE_REPORT' },
            API::Reports::InternalRequestRepresentation->new( %$ok_internal_rep, report_type => 'SEARCH_QUERY_PERFORMANCE_REPORT' ) ),
        'online report not OK for a SEARCH_QUERY_PERFORMANCE_REPORT request with app_id'.$_
    ) foreach @MOBILE_APP_IDS;

    ok(
        !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
            $_,
            $ok_request_data,
            API::Reports::InternalRequestRepresentation->new( %$ok_internal_rep, need_to_request_report_from_provider => 0 ) ),
        'online report OK for a request that doesn\'t actually need statistics with app_id '.$_
    ) foreach @ALL_APP_IDS;

    subtest 'time period' => sub {
        plan tests => 14;

        ok(
            API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                TEST_APPLICATION_ID,
                { %$ok_request_data, date_range_type => 'ALL_TIME' },
                API::Reports::InternalRequestRepresentation->new(%$ok_internal_rep) ),
            'online report not OK for an ALL_TIME request' );

        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                { %$ok_request_data, date_range_type => 'ALL_TIME' },
                API::Reports::InternalRequestRepresentation->new(%$ok_internal_rep) ),
            'online report OK for an ALL_TIME report with app_id '.$_
        ) foreach @MOBILE_APP_IDS;

        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, end_date => '2016-12-06' } ) ),
            'online report OK for an 6-day request with app_id '.$_
        ) foreach @ALL_APP_IDS;

        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, end_date => '2016-12-07' } ) ),
            'online report OK for a 7-day request with app_id '.$_
        ) foreach @ALL_APP_IDS;

        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, end_date => '2016-12-07' } ) ),
            'online report OK for a 7-day request with app_id '.$_
        ) foreach @MOBILE_APP_IDS;

        ok(
            API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                TEST_APPLICATION_ID,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, end_date => '2016-12-08' } ) ),
            'online report not OK for an 8-day request' );

        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, end_date => '2016-12-08' } ) ),
            'online report OK for an 8-day request with app_id '.$_
        ) foreach @MOBILE_APP_IDS;

    };

    subtest 'number of campaigns' => sub {
        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, oid => [ 1 .. 9 ] } ) ),
            'online report OK for a 9-campaign request with app_id '.$_
        ) foreach @ALL_APP_IDS;

        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, oid => [ 1 .. 10 ] } ) ),
            'online report OK for a 10-campaign request with app_id '.$_
        ) foreach @ALL_APP_IDS;

        ok(
            API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                TEST_APPLICATION_ID,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, oid => [ 1 .. 11 ] } ) ),
            'online report not OK for an 11-campaign request' );

        ok(
            !API::Service::Reports::ProcessingModeChooser->_require_offline_mode(
                $_,
                $ok_request_data,
                API::Reports::InternalRequestRepresentation->new(
                    %$ok_internal_rep,
                    provider_request_params => { %$ok_provider_params, oid => [ 1 .. 11 ] } ) ),
            'online report OK for an 11-campaign request with app_id '.$_
        ) foreach @MOBILE_APP_IDS;
    };

};

subtest 'choose_processing_mode' => sub {
    plan tests => 3;

    cmp_deeply(
        API::Service::Reports::ProcessingModeChooser->choose_processing_mode(TEST_APPLICATION_ID, 'illegal_mode'),
        defect_matcher( 8000, 'BadRequest' ),
        'illegal processingMode' );

    no warnings 'redefine';

    subtest '_require_offline_mode = always true' => sub {
        plan tests => 3;
        local *API::Service::Reports::ProcessingModeChooser::_require_offline_mode = sub { 1 };

        is(
            API::Service::Reports::ProcessingModeChooser->choose_processing_mode(TEST_APPLICATION_ID, 'auto'),
            API::Service::Reports::ProcessingModeChooser::PROCESSING_MODE_OFFLINE,
            'processingMode = auto' );

        cmp_deeply(
            API::Service::Reports::ProcessingModeChooser->choose_processing_mode(TEST_APPLICATION_ID, 'online'),
            defect_matcher( 8312, 'ReportCannotBeBuiltOnline' ),
            'processingMode = online' );

        is(
            API::Service::Reports::ProcessingModeChooser->choose_processing_mode(TEST_APPLICATION_ID, 'offline'),
            API::Service::Reports::ProcessingModeChooser::PROCESSING_MODE_OFFLINE,
            'processingMode = offline' );
    };

    subtest '_require_offline_mode = always false' => sub {
        plan tests => 3;
        local *API::Service::Reports::ProcessingModeChooser::_require_offline_mode = sub { 0 };

        is(
            API::Service::Reports::ProcessingModeChooser->choose_processing_mode(TEST_APPLICATION_ID, 'auto'),
            API::Service::Reports::ProcessingModeChooser::PROCESSING_MODE_ONLINE,
            'processingMode = auto' );

        is(
            API::Service::Reports::ProcessingModeChooser->choose_processing_mode(TEST_APPLICATION_ID, 'online'),
            API::Service::Reports::ProcessingModeChooser::PROCESSING_MODE_ONLINE,
            'processingMode = online' );

        is(
            API::Service::Reports::ProcessingModeChooser->choose_processing_mode(TEST_APPLICATION_ID, 'offline'),
            API::Service::Reports::ProcessingModeChooser::PROCESSING_MODE_OFFLINE,
            'processingMode = offline' );
    };
};

subtest 'require_offline_mode_adjusted' => sub {
    plan tests => 3;
    my $ok_provider_params = { start_date => '2016-12-01', end_date => '2016-12-03', oid => [ qw( 1 ) ] };
    ok(
        !API::Service::Reports::ProcessingModeChooser->require_offline_mode_adjusted(
            { start_date => '2016-12-01', end_date => '2016-12-07', oid => [ 1..10 ] },
            CUSTOM_REPORT_TYPE
        ),
        'by default 7 days and 10 campaigns goes online'
    );
    ok(
        API::Service::Reports::ProcessingModeChooser->require_offline_mode_adjusted(
            { start_date => '2016-12-01', end_date => '2016-12-08', oid => [ 1..11 ] },
            CUSTOM_REPORT_TYPE
        ),
        'by default 7 days and 11 campaigns goes offline'
    );

    ok(
        API::Service::Reports::ProcessingModeChooser->require_offline_mode_adjusted(
            { start_date => '2016-12-01', end_date => '2016-12-09', oid => [ 1..10 ] },
            CUSTOM_REPORT_TYPE
        ),
        'by default 8 days and 10 campaigns goes offline'
    );
};

sub defect_matcher {
    my ( $code, $name ) = @_;

    return all(
        obj_isa('Direct::Defect'),
        code( sub {
            my ($error) = @_;
            my ( $ok, $stack ) = Test::Deep::cmp_details( { %$error }, {
                code => $code,
                description => ignore(),
                name => $name,
                text => ignore(),
                type => 'error',
                suffix => '',
            } );

            return 1 if $ok;
            return ( 0, Test::Deep::deep_diag($stack) );
        } ),
    );
}
