#!/usr/bin/perl
use Direct::Modern;
use my_inc '../../../..', for => 'unit_tests';
use utf8;

use Test::More;
use Test::Deep;
use Test::MockObject;

use Direct::Errors::Messages;

use API::Test::MockHelper qw/mock_subs restore_mocked_subs/;
use API::Test::Utils qw/get_error_with_ignored_description/;
use API::Service::Reports;
use API::Reports::DataRules qw/CUSTOM_DATE_RANGE_TYPE/;
use API::Response::Headers;
use API::Request::Headers;

use Rbac qw/:const/;

use Settings;
use Yandex::DBUnitTest qw(init_test_dataset);

use constant API_CONSUMER_NAME => 'api';
use constant TRUE => 1;
use constant FALSE => 0;

use constant CLIENT_ID => 1001;
use constant CLIENT_ID2 => 1002;
use constant CLIENT_ID3 => 1003;
use constant CLIENT_ID4 => 1004;
use constant UID => 10011;
use constant UID2 => 10012;
use constant UID3 => 10013;
use constant UID4 => 10014;
use constant LOGIN => 'l';
use constant LOGIN2 => 'l2';
use constant LOGIN3 => 'l3';
use constant LOGIN4 => 'l4';

plan tests => 7;

init_test_dataset(
  {
        shard_client_id => {
            original_db => PPCDICT(),
            rows => [
                { ClientID => CLIENT_ID, shard => 1 },
                { ClientID => CLIENT_ID2, shard => 1 },
                { ClientID => CLIENT_ID3, shard => 1 },
                { ClientID => CLIENT_ID4, shard => 1 },
            ]
        },
        shard_uid => {
            original_db => PPCDICT(),
            rows => [
                { uid => UID, ClientID => CLIENT_ID },
                { uid => UID2, ClientID => CLIENT_ID2 },
                { uid => UID3, ClientID => CLIENT_ID3 },
                { uid => UID4, ClientID => CLIENT_ID4 },
            ]
        },
        shard_login => {
            original_db => PPCDICT(),
            rows => [
                { login => LOGIN, uid => UID },
                { login => LOGIN2, uid => UID2 },
                { login => LOGIN3, uid => UID3 },
                { login => LOGIN4, uid => UID4 },
            ]
        },
        users => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    { uid => UID, ClientID => CLIENT_ID, login => LOGIN },
                    { uid => UID2, ClientID => CLIENT_ID2, login => LOGIN2 },
                    { uid => UID3, ClientID => CLIENT_ID3, login => LOGIN3 },
                    { uid => UID4, ClientID => CLIENT_ID4, login => LOGIN4 },
                ]
            }
        },
        banners => {
            original_db => PPC( shard => 1 ),
            rows => [
                {
                    bid => 11111,
                    cid => 111,
                },
                {
                    bid => 33333,
                    cid => 333,
                },
                {
                    bid => 55555,
                    cid => 555,
                },
            ],
        },
        phrases => {
            original_db => PPC( shard => 1 ),
            rows => {
                1 => [
                    {
                        pid => 1111,
                        cid => 111,
                    },
                    {
                        pid => 2222,
                        cid => 222,
                    },
                    {
                        pid => 6666,
                        cid => 666,
                    },
                ],
            }
        },
    }
);

my $S = API::Service::Reports->new('Reports');
$S->http_request_headers_for_log(API::Request::Headers->new());

subtest '_convert_request_data_to_report_params' => sub{
    mock_subs(
        'API::Service::Reports',
        'get_cid2orderid' => [{
            args => [cid => [111, 222]],
            result => {111 => 333, 222 => 444},
        }],
        'mass_get_is_uac_campaign_by_order_ids' => [{
            args => [[333, 444]],
            result => {333 => 1},
        }],
        'convert_page_filter' => [{
            args => [{eq => ['xxx', 'yyy']}, 'en'],
            result => [eq => [555, 666]],
        }],
        'convert_field_names_and_report_type_to_group_by' => [{
            args => [['f1', 'f2'], 'ZZZ'],
            result => ['sss'],
        }],
        'convert_field_names_to_date_aggregation' => [{
            args => [['f1', 'f2']],
            result => 'none',
        }],
        'convert_field_names_to_countable_fields' => [{
            args => [['f1', 'f2'], 'ZZZ'],
            result => ['nnn'],
        }],
        'API::Service::Base::subclient_client_id' => [
            ({args => [ignore], result => CLIENT_ID}) x 3
        ],
        'API::Service::Base::operator_client_id' => [
            ({args => ignore, result => CLIENT_ID}) x 2
        ],
        'Client::ClientFeatures::has_access_to_new_feature_from_java' => [
            ({args => ignore, result => 0}) x 3,
        ],
    );
    my $request = {
        report_type => 'ZZZ',
        field_names => ['f1', 'f2'],
        filter => {
            cid => {
                eq => [111, 222],
            },
            page => {
                eq => ['xxx', 'yyy'],
            },
        },
        date_range_type => CUSTOM_DATE_RANGE_TYPE,
        start_date => '2016-01-01',
        end_date => '2016-01-01',
        currency => 'XXX',
        with_nds => 1,
        with_discount => 0,
        order_by => [
            {
               dir => 'asc',
               field => 'position',
            }
        ],
        page => {
            limit => 5,
            offset => 10,
        },
    };

    cmp_deeply($S->_convert_request_data_to_report_params($request, 'en'), {
        oid => [333, 444],
        uac_oid => [333],
        filter => {
            page => [
                eq => [555, 666],
            ],
        },
        start_date => '2016-01-01',
        end_date => '2016-01-01',

        date_aggregation_by => 'none',
        group_by => ['sss'],

        options => {
            without_totals => TRUE,
            consumer => API_CONSUMER_NAME,
            filter_by_consumer_values => TRUE,
            currency => 'XXX',
            with_nds => TRUE,
            with_discount => FALSE,
            with_winrate => '',
            extra_countable_fields => ['nnn'],
            four_digits_precision => 0,
            operator_ClientID => CLIENT_ID,
        },

        limits => {
            order_by => [
                {
                   dir => 'asc',
                   field => 'position',
                }
            ],
            limit => 5,
            offset => 10,
        },
        translocal_params => {
            tree => 'api',
        },
        ClientID_for_stat_experiments => CLIENT_ID,
    }, 'it should properly prepare report params');
    restore_mocked_subs();
};

subtest '_populate_request_data' => sub{
    subtest 'use client currency as report currency' => sub{
        my $req_data = {};
        mock_subs(
            'API::Service::Reports',
            '_get_subclient_currency' => [{
                args => [ignore],
                result => 'YYY',
            }]
        );
        $S->_populate_request_data($req_data);
        is($req_data->{Currency}, 'YYY');
        restore_mocked_subs();
    };
};


subtest '_validate_create' => sub{
    subtest 'for valid data is should return undef' => sub{
        mock_subs(
            'API::Service::Reports::Validation::validate_create_report_request' => [{
                args => [ignore(), {Currency => 'XXX'}],
                result => Test::MockObject->new()->set_true( 'is_valid' ),
            }],
        );
        is($S->_validate_create({Currency => 'XXX'}), undef);
        restore_mocked_subs();
    };
    subtest 'for invalid data is should return result of get_errors method' => sub{
        mock_subs(
            'API::Service::Reports::Validation::validate_create_report_request' => [{
                args => [ignore(), {Currency => 'XXX'}],
                result => Test::MockObject->new()->set_false( 'is_valid' )->set_series( 'get_errors', ['ErroR']),
            }],
        );
        is($S->_validate_create({Currency => 'XXX'}), 'ErroR');
        restore_mocked_subs();
    };
};

subtest '_restrict_campaign_ids_by_ad_ids' => sub{
    mock_subs('API::Service::Base::subclient_client_id' => [{ args => ignore, result => CLIENT_ID }]);
    cmp_deeply($S->_restrict_campaign_ids_by_ad_ids([111,222,333,444],[11111,33333,55555]), [111,333], 'get valid intersection of campaign_ids');
    cmp_deeply($S->_restrict_campaign_ids_by_ad_ids([],[11111,33333,55555]), [], 'get empty list of campaign_ids');
    cmp_deeply($S->_restrict_campaign_ids_by_ad_ids(undef,[11111,33333,55555]), [111,333,555], 'get valid list of campaign_ids');
    restore_mocked_subs();
};

subtest '_restrict_campaign_ids_by_adgroup_ids' => sub{
    mock_subs('API::Service::Base::subclient_client_id' => [{ args => ignore, result => CLIENT_ID }]);
    cmp_deeply($S->_restrict_campaign_ids_by_adgroup_ids([111,222,333,444],[1111,2222,6666]), [111,222], 'get valid intersection of campaign_ids');
    cmp_deeply($S->_restrict_campaign_ids_by_adgroup_ids([],[1111,2222,6666]), [], 'get empty list of campaign_ids');
    cmp_deeply($S->_restrict_campaign_ids_by_adgroup_ids(undef,[1111,2222,6666]), [111,222,666], 'get valid list of campaign_ids');
    restore_mocked_subs();
};

subtest '_get_accessible_client_ids' => sub{
    mock_subs(
        'API::Service::Base::operator_role' => [({ args => ignore, result => $ROLE_CLIENT }) x 3],
        'API::Service::Base::operator_uid' => [({ args => ignore, result => UID }) x 3],
        'API::Service::Reports::get_perminfo' => [({ args => ignore, result => { ClientID => CLIENT_ID, mcc_client_ids => [CLIENT_ID2, CLIENT_ID3] } }) x 3],
    );
    cmp_deeply($S->_get_accessible_client_ids([LOGIN, LOGIN2, LOGIN3, LOGIN4]), bag(CLIENT_ID, CLIENT_ID2, CLIENT_ID3), 'get valid list of client_ids');
    cmp_deeply($S->_get_accessible_client_ids([LOGIN4]), [], 'get empty list of client_ids');
    cmp_deeply($S->_get_accessible_client_ids([]), [], 'get empty list of client_ids for empty list of logins');
    restore_mocked_subs();
};

subtest 'check report with existing name' => sub {
    my $request = {
        DateRangeType => 'LAST_30_DAYS',
        FieldNames => ['CampaignId','CriteriaType','CriteriaId','Criteria','Impressions','Clicks','Cost','Bounces','BounceRate'],
        Format => 'TSV',
        IncludeDiscount => 'NO',
        IncludeVAT => 'YES',
        Page => {
            Limit => '10'
        },
        ReportName => "Отчет 20170217-0003",
        ReportType => 'CUSTOM_REPORT',
        SelectionCriteria => {
            Filter => [
                {
                    Field => 'CampaignId',
                    Operator => 'IN',
                    Values => ['16573126']
                }
            ]
        }
    };
    mock_subs(
        'API::Service::Reports',
        'DirectRedis::get_redis' => [{ args => ignore, result => bless {}, 'Redis' }],
        'Yandex::Redis::RateLimit::request_ok' => [{ args => ignore, result => 1 }],
        'PPC' => [{ args => ignore, result => undef }],
        'Yandex::DBQueue::new' => [{ args => ignore, result => bless { count_jobs => 1 }, 'Yandex::DBQueue' }],
        'Yandex::DBQueue::count_jobs' => [{ args => ignore, result => 1 }],
        'API::Service::Base::subclient_client_id' => [
            ({ args => ignore, result => CLIENT_ID }) x 4
        ],
        'API::Service::Base::get_unavailable_cids_map' => [
            { args => [ignore(),[16573126], supported_camp_kind => 'stat_stream_ext', pass_archived => 1], result_set => [] },
        ],
        'API::Service::Base::get_http_request_header' => [
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },

            { args => ignore, result => 'true' },
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
        ],
        'API::Service::Base::is_operator_super' => [{ args => ignore, result => 0 },{ args => ignore, result => 0 },{ args => ignore, result => 0 }],
        'API::Service::Reports::ProcessingModeChooser::choose_processing_mode' => [{ args => ignore, result => 2 }],
        _populate_request_data => [{ args => ignore, result => undef }],
        _validate_create => [{ args => ignore, result => undef }],
        convert_client_request_to_internal_representation => [
            {
                args => ignore,
                result => bless { return_money_in_micros => 1, lang => 'en', report_name => '' }, 'API::Reports::InternalRequestRepresentation'
            }
        ],
        'API::Reports::OfflineReportTask::get_or_create' => [
            {
                args => [
                    ignore,
                    bless {
                        request_data_md5 => '59cff0b16c0dc851372a89a88a72ff60',
                        client_id => CLIENT_ID,
                        job_args => ignore,
                        report_name => '',
                        dont_check_limit => 0
                    }, 'API::Reports::OfflineReportTaskParameters'
                ],
                result_in_args => { 4 => sub { die API::Reports::ReportNameCollisionException->new(defect => error_BadParams()) } }
            }
        ],
        'API::Service::Base::application_id' => [{ args => ignore, result => 1 }],
        'API::Service::Base::http_response_headers' => [
            { args => ignore, result => API::Response::Headers->new(reportsInQueue => 1) },
        ],
    );
    cmp_deeply($S->create($request), get_error_with_ignored_description(error_BadParams), 'get error for report with existing name but different params');
    restore_mocked_subs();

    mock_subs(
        'API::Service::Reports',
        'DirectRedis::get_redis' => [{ args => ignore, result => bless {}, 'Redis' }],
        'Yandex::Redis::RateLimit::request_ok' => [{ args => ignore, result => 1 }],
        'PPC' => [{ args => ignore, result => undef }],
        'Yandex::DBQueue::new' => [{ args => ignore, result => bless { count_jobs => 1 }, 'Yandex::DBQueue' }],
        'Yandex::DBQueue::count_jobs' => [{ args => ignore, result => 1 }],
        'API::Service::Base::subclient_client_id' => [
            ({ args => ignore, result => CLIENT_ID }) x 4
        ],
        'API::Service::Base::get_unavailable_cids_map' => [
            { args => [ignore(),[16573126], supported_camp_kind => 'stat_stream_ext', pass_archived => 1], result_set => [] },
        ],
        'API::Service::Base::get_http_request_header' => [
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },	    

            { args => ignore, result => 'false' },
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
        ],
        'API::Service::Base::is_operator_super' => [{ args => ignore, result => 0 },{ args => ignore, result => 0 },{ args => ignore, result => 0 }],
        'API::Service::Reports::ProcessingModeChooser::choose_processing_mode' => [{ args => ignore, result => 2 }],
        _populate_request_data => [{ args => ignore, result => undef }],
        _validate_create => [{ args => ignore, result => undef }],
        convert_client_request_to_internal_representation => [
            {
                args => ignore,
                result => bless { return_money_in_micros => 0, lang => 'en', report_name => '' }, 'API::Reports::InternalRequestRepresentation'
            }
        ],
        'API::Reports::OfflineReportTask::get_or_create' => [
            {
                args => [
                    ignore,
                    bless {
                        request_data_md5 => 'c9b0c92e143ff9a7d806f796e616bc07',
                        client_id => CLIENT_ID,
                        job_args => ignore,
                        report_name => '',
                        dont_check_limit => 0
                    }, 'API::Reports::OfflineReportTaskParameters'
                ],
                result_in_args => { 4 => sub { die API::Reports::ReportNameCollisionException->new(defect => error_BadParams()) } }
            }
        ],
        'API::Service::Base::application_id' => [{ args => ignore, result => 1 }],
        'API::Service::Base::http_response_headers' => [
            { args => ignore, result => API::Response::Headers->new(reportsInQueue => 1) },
        ],
    );

    cmp_deeply($S->create($request), get_error_with_ignored_description(error_BadParams), 'get error for report with existing name but not micros');
    restore_mocked_subs();

    mock_subs(
        'API::Service::Reports',
        'DirectRedis::get_redis' => [{ args => ignore, result => bless {}, 'Redis' }],
        'Yandex::Redis::RateLimit::request_ok' => [{ args => ignore, result => 1 }],
        'PPC' => [{ args => ignore, result => undef }],
        'Yandex::DBQueue::new' => [{ args => ignore, result => bless { count_jobs => 1 }, 'Yandex::DBQueue' }],
        'Yandex::DBQueue::count_jobs' => [{ args => ignore, result => 1 }],
        'API::Service::Base::subclient_client_id' => [
            ({ args => ignore, result => CLIENT_ID }) x 4
        ],
        'API::Service::Base::get_unavailable_cids_map' => [
            { args => [ignore(),[16573126], supported_camp_kind => 'stat_stream_ext', pass_archived => 1], result_set => [] },
        ],
        'API::Service::Base::get_http_request_header' => [
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },

            { args => ignore, result => 'true' },
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
        ],
        'API::Service::Base::is_operator_super' => [{ args => ignore, result => 0 },{ args => ignore, result => 0 },{ args => ignore, result => 0 }],
        'API::Service::Reports::ProcessingModeChooser::choose_processing_mode' => [{ args => ignore, result => 2 }],
        _populate_request_data => [{ args => ignore, result => undef }],
        _validate_create => [{ args => ignore, result => undef }],
        convert_client_request_to_internal_representation => [
            {
                args => ignore,
                result => bless { return_money_in_micros => 1, lang => 'ru', report_name => '' }, 'API::Reports::InternalRequestRepresentation'
            }
        ],
        'API::Reports::OfflineReportTask::get_or_create' => [
            {
                args => [
                    ignore,
                    bless {
                        request_data_md5 => '52761e0f85cfba7f052fd358fd2af2d8',
                        client_id => CLIENT_ID,
                        job_args => ignore,
                        report_name => '',
                        dont_check_limit => 0
                    }, 'API::Reports::OfflineReportTaskParameters'
                ],
                result_in_args => { 4 => sub { die API::Reports::ReportNameCollisionException->new(defect => error_BadParams()) } }
            }
        ],
        'API::Service::Base::application_id' => [{ args => ignore, result => 1 }],
        'API::Service::Base::http_response_headers' => [
            { args => ignore, result => API::Response::Headers->new(reportsInQueue => 1) },
        ],
    );

    cmp_deeply($S->create($request), get_error_with_ignored_description(error_BadParams), 'get error for report with existing name but different lang');
    restore_mocked_subs();

    mock_subs(
        'API::Service::Reports',
        'DirectRedis::get_redis' => [{ args => ignore, result => bless {}, 'Redis' }],
        'Yandex::Redis::RateLimit::request_ok' => [{ args => ignore, result => 1 }],
        'PPC' => [{ args => ignore, result => undef }],
        'Yandex::DBQueue::new' => [{ args => ignore, result => bless { count_jobs => 1 }, 'Yandex::DBQueue' }],
        'Yandex::DBQueue::count_jobs' => [{ args => ignore, result => 1 }],
        'API::Service::Base::subclient_client_id' => [
            ({ args => ignore, result => CLIENT_ID }) x 4
        ],
        'API::Service::Base::get_unavailable_cids_map' => [
            { args => [ignore(),[16573126], supported_camp_kind => 'stat_stream_ext', pass_archived => 1], result_set => [] },
        ],
        'API::Service::Base::get_http_request_header' => [
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },

            { args => ignore, result => 'true' },
            { args => ignore, result => 'auto' },
            { args => ignore, result => 'true' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
        ],
        'API::Service::Base::is_operator_super' => [{ args => ignore, result => 0 },{ args => ignore, result => 0 },{ args => ignore, result => 0 }],
        'API::Service::Reports::ProcessingModeChooser::choose_processing_mode' => [{ args => ignore, result => 2 }],
        _populate_request_data => [{ args => ignore, result => undef }],
        _validate_create => [{ args => ignore, result => undef }],
        convert_client_request_to_internal_representation => [
            {
                args => ignore,
                result => bless { return_money_in_micros => 1, lang => 'en', report_name => '', skip_report_header => 1 }, 'API::Reports::InternalRequestRepresentation'
            }
        ],
        'API::Reports::OfflineReportTask::get_or_create' => [
            {
                args => [
                    ignore,
                    bless {
                        request_data_md5 => 'e904737eb70b300d79a33c7c872140a8',
                        client_id => CLIENT_ID,
                        job_args => ignore,
                        report_name => '',
                        dont_check_limit => 0
                    }, 'API::Reports::OfflineReportTaskParameters'
                ],
                result_in_args => { 4 => sub { die API::Reports::ReportNameCollisionException->new(defect => error_BadParams()) } }
            }
        ],
        'API::Service::Base::application_id' => [{ args => ignore, result => 1 }],
        'API::Service::Base::http_response_headers' => [
            { args => ignore, result => API::Response::Headers->new(reportsInQueue => 1) },
        ],
    );

    cmp_deeply($S->create($request), get_error_with_ignored_description(error_BadParams), 'get error for report with existing name skipReportHeader=true');
    restore_mocked_subs();

    mock_subs(
        'API::Service::Reports',
        'DirectRedis::get_redis' => [{ args => ignore, result => bless {}, 'Redis' }],
        'Yandex::Redis::RateLimit::request_ok' => [{ args => ignore, result => 1 }],
        'PPC' => [{ args => ignore, result => undef }],
        'Yandex::DBQueue::new' => [{ args => ignore, result => bless { count_jobs => 1 }, 'Yandex::DBQueue' }],
        'Yandex::DBQueue::count_jobs' => [{ args => ignore, result => 1 }],
        'API::Service::Base::subclient_client_id' => [
            ({ args => ignore, result => CLIENT_ID }) x 4
        ],
        'API::Service::Base::get_http_request_header' => [
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },

            { args => ignore, result => 'true' },
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => 'true' },
            { args => ignore, result => undef },
        ],
        'API::Service::Base::is_operator_super' => [{ args => ignore, result => 0 },{ args => ignore, result => 0 },{ args => ignore, result => 0 }],
        'API::Service::Reports::ProcessingModeChooser::choose_processing_mode' => [{ args => ignore, result => 2 }],
        _populate_request_data => [{ args => ignore, result => undef }],
        _validate_create => [{ args => ignore, result => undef }],
        convert_client_request_to_internal_representation => [
            {
                args => ignore,
                result => bless { return_money_in_micros => 1, lang => 'en', report_name => '', skip_column_header => 1 }, 'API::Reports::InternalRequestRepresentation'
            }
        ],
        'API::Reports::OfflineReportTask::get_or_create' => [
            {
                args => [
                    ignore,
                    bless {
                        request_data_md5 => 'a823302db2856398ff51dca5b56c9940',
                        client_id => CLIENT_ID,
                        job_args => ignore,
                        report_name => '',
                        dont_check_limit => 0
                    }, 'API::Reports::OfflineReportTaskParameters'
                ],
                result_in_args => { 4 => sub { die API::Reports::ReportNameCollisionException->new(defect => error_BadParams()) } }
            }
        ],
        'API::Service::Base::application_id' => [{ args => ignore, result => 1 }],
        'API::Service::Base::http_response_headers' => [
            { args => ignore, result => API::Response::Headers->new(reportsInQueue => 1) },
        ],
    );

    cmp_deeply($S->create($request), get_error_with_ignored_description(error_BadParams), 'get error for report with existing name skipColumnHeader=true');
    restore_mocked_subs();

    mock_subs(
        'API::Service::Reports',
        'DirectRedis::get_redis' => [{ args => ignore, result => bless {}, 'Redis' }],
        'Yandex::Redis::RateLimit::request_ok' => [{ args => ignore, result => 1 }],
        'PPC' => [{ args => ignore, result => undef }],
        'Yandex::DBQueue::new' => [{ args => ignore, result => bless { count_jobs => 1 }, 'Yandex::DBQueue' }],
        'Yandex::DBQueue::count_jobs' => [{ args => ignore, result => 1 }],
        'API::Service::Base::subclient_client_id' => [
            ({ args => ignore, result => CLIENT_ID }) x 4
        ],
        'API::Service::Base::get_unavailable_cids_map' => [
            { args => [ignore(),[16573126], supported_camp_kind => 'stat_stream_ext', pass_archived => 1], result_set => [] },
        ],
        'API::Service::Base::get_http_request_header' => [
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => undef },

            { args => ignore, result => 'true' },
            { args => ignore, result => 'auto' },
            { args => ignore, result => undef },
            { args => ignore, result => undef },
            { args => ignore, result => 'true' },
        ],
        'API::Service::Base::is_operator_super' => [{ args => ignore, result => 0 },{ args => ignore, result => 0 },{ args => ignore, result => 0 }],
        'API::Service::Reports::ProcessingModeChooser::choose_processing_mode' => [{ args => ignore, result => 2 }],
        _populate_request_data => [{ args => ignore, result => undef }],
        _validate_create => [{ args => ignore, result => undef }],
        convert_client_request_to_internal_representation => [
            {
                args => ignore,
                result => bless { return_money_in_micros => 1, lang => 'en', report_name => '', skip_report_summary => 1 }, 'API::Reports::InternalRequestRepresentation'
            }
        ],
        'API::Reports::OfflineReportTask::get_or_create' => [
            {
                args => [
                    ignore,
                    bless {
                        request_data_md5 => '320150763338feb3197a81368f84d851',
                        client_id => CLIENT_ID,
                        job_args => ignore,
                        report_name => '',
                        dont_check_limit => 0
                    }, 'API::Reports::OfflineReportTaskParameters'
                ],
                result_in_args => { 4 => sub { die API::Reports::ReportNameCollisionException->new(defect => error_BadParams()) } }
            }
        ],
        'API::Service::Base::application_id' => [{ args => ignore, result => 1 }],
        'API::Service::Base::http_response_headers' => [
            { args => ignore, result => API::Response::Headers->new(reportsInQueue => 1) },
        ],
    );

    cmp_deeply($S->create($request), get_error_with_ignored_description(error_BadParams), 'get error for report with existing name skipReportSummary=true');
    restore_mocked_subs();

};
