#!/usr/bin/perl
use Direct::Modern;

use List::MoreUtils qw/pairwise uniq/;
use Template;
use Test::Deep;
use Test::More tests => 9;
use XML::Compile::Schema;

use my_inc '../../../..', for => 'unit_tests';

use Yandex::DateTime;
use Direct::Errors::Messages;
use API::Test::MockHelper qw/mock_subs restore_mocked_subs/;
use API::Test::Reports::Xsd qw/get_enum_values/;
use API::Test::Reports::Fields qw/
    @REPORT_TYPES
    all_fields
    CAMPAIGN_PERFORMANCE_REPORT_TYPE
    CRITERIA_PERFORMANCE_REPORT_TYPE
    CUSTOM_REPORT_TYPE
    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE
    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE
/;
use API::Service::Reports::Validation;
use API::Reports::DataRules qw/
    CUSTOM_DATE_RANGE_TYPE
    %FILTER_PARAMS_MAP
/;
use Settings;

my @XSD_NAMES = qw( reports general );

my %XSD_NAME_TO_FILENAME = (
    'reports' => "$Settings::ROOT/api/wsdl/v5/reports.xsd",
    'general' => "$Settings::ROOT/api/wsdl/v5/general.xsd",
);

my %XSD_NAME_TO_NAMESPACE = (
    'reports' => 'http://api.direct.yandex.com/v5/reports',
    'general' => 'http://api.direct.yandex.com/v5/general',
);

my %FILTER_ENUMS = (
    AdFormat => [qw/ADAPTIVE_IMAGE IMAGE SMART_MULTIPLE SMART_SINGLE SMART_TILE SMART_VIDEO TEXT VIDEO/],
    AdNetworkType => [qw/AD_NETWORK SEARCH/],
    Age => [qw/AGE_0_17 AGE_18_24 AGE_25_34 AGE_35_44 AGE_45 AGE_45_54 AGE_55 UNKNOWN/],
    AttributionModels => [qw/FC FCCD LC LSC LSCCD LYDC LYDCCD LYDVCD/],
    CarrierType => [qw/CELLULAR STATIONARY UNKNOWN/],
    CampaignType => [qw/CONTENT_PROMOTION_CAMPAIGN CPM_BANNER_CAMPAIGN CPM_DEALS_CAMPAIGN CPM_FRONTPAGE_CAMPAIGN CPM_PRICE_CAMPAIGN DYNAMIC_TEXT_CAMPAIGN MCBANNER_CAMPAIGN MOBILE_APP_CAMPAIGN SMART_CAMPAIGN TEXT_CAMPAIGN/],
    ClickType => [qw/BUTTON CHAT DISPLAY_URL_PATH MARKET MOBILE_APP_ICON PHONE PRODUCT_EXTENSIONS PROMO_EXTENSION SITELINK1 SITELINK2 SITELINK3 SITELINK4 SITELINK5 SITELINK6 SITELINK7 SITELINK8 TITLE TURBO_GALLERY UNKNOWN VCARD/],
    TargetingCategory => [qw /ACCESSORY ALTERNATIVE BROADER COMPETITOR EXACT UNKNOWN/],
    IncomeGrade => [qw /ABOVE_AVERAGE HIGH OTHER VERY_HIGH/],
    CriteriaType => {
        CUSTOM_REPORT_TYPE() => [qw/AUDIENCE_TARGET AUTOTARGETING DYNAMIC_TEXT_AD_TARGET KEYWORD SMART_BANNER_FILTER/],
        SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => [qw/AUTOTARGETING DYNAMIC_TEXT_AD_TARGET KEYWORD/],
    },
    CriterionType => {
        CUSTOM_REPORT_TYPE() => [qw/AUTOTARGETING FEED_FILTER INTERESTS_AND_DEMOGRAPHICS KEYWORD MOBILE_APP_CATEGORY RETARGETING WEBPAGE_FILTER/],
        SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => [qw/AUTOTARGETING FEED_FILTER KEYWORD WEBPAGE_FILTER/],
    },
    Device => [qw/DESKTOP MOBILE TABLET/],
    Gender => [qw/GENDER_FEMALE GENDER_MALE UNKNOWN/],
    MatchType => [qw/KEYWORD NONE RELATED_KEYWORD SYNONYM/],
    MobilePlatform => [qw/ANDROID IOS OTHER UNKNOWN/],
    Slot => [qw/ALONE OTHER PREMIUMBLOCK PRODUCT_GALLERY SUGGEST/],
);

my %MANDATORY_FIELDS = (
    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE() => ['CampaignId'],
);

my $template = Template->new( ABSOLUTE => 1, LOAD_PERL => 1 );

my $schema = XML::Compile::Schema->new;
for my $xsd_name (@XSD_NAMES) {
    my $xsd_content;
    $template->process($XSD_NAME_TO_FILENAME{$xsd_name}, {
        api_version => 5,
        is_beta => 0,
        lang => 'ru',
        API_SERVER_PATH => "https://api.direct.yandex.ru",
    }, \$xsd_content);

    $schema->importDefinitions(\$xsd_content);
}

sub ok_validation_success {
    my ( $code, $description ) = @_;
    my $vr = $code->();
    ok($vr->is_valid, "$description: validation succeeded" );
}

sub ok_validation_fail {
    my ( $code, $description ) = @_;
    my $vr = $code->();
    ok(!$vr->is_valid, "$description: validation failed" );
}

sub ok_error {
    my ( $code, $description ) = @_;
    my $error = $code->();
    ok( $error && eval { $error->isa('Direct::Defect') }, "$description: expecting error" );
}

sub ok_no_error {
    my ( $code, $description ) = @_;
    ok( !$code->(), "$description: expecting no error" );
}

my $valclass = 'API::Service::Reports::Validation';

subtest '_validate_create_date_range' => sub {
    my @date_range_types = get_enum_values( 'reports', 'DateRangeTypeEnum' );

    plan tests => scalar(@date_range_types);

    my $VALID_DATE_EARLIER = Yandex::DateTime->now()->truncate(to => 'month')->subtract(months => 1)->ymd('-');
    my $VALID_DATE_LATER = Yandex::DateTime->now()->truncate(to => 'month')->subtract(months => 1)->add(days => 7)->ymd('-');
    my $INVALID_DATE = 'Apr 1, 2016';
    my $INVALID_FORMAT_DATE_FROM = Yandex::DateTime->now()->truncate(to => 'month')->subtract(months => 6)->ymd('');
    my $INVALID_FORMAT_DATE_TO = Yandex::DateTime->now()->truncate(to => 'month')->subtract(months => 6)->add(days => 7)->ymd('');
    my $DATE_IN_FUTURE = Yandex::DateTime->now()->add(days => 1)->ymd();
    my $DATE_LONG_AGO = Yandex::DateTime->now()->subtract(years => 5)->ymd('-');
    my $DATE_NOT_ALLOWED_FOR_SEARCH_QUERY_REPORT = Yandex::DateTime->now()->subtract(months => 6)->ymd('-');

    subtest CUSTOM_DATE_RANGE_TYPE() => sub {
        plan tests => 15 * scalar(@REPORT_TYPES) + 1;

        foreach my $report_type (@REPORT_TYPES) {
            ok_validation_success sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $VALID_DATE_EARLIER, DateTo => $VALID_DATE_LATER },
                    $report_type
                );
            }, 'two valid dates in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateTo => $VALID_DATE_LATER },
                    $report_type
                );
            }, 'missing DateFrom in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $INVALID_DATE, DateTo => $VALID_DATE_LATER },
                    $report_type
                );
            }, 'invalid DateFrom in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $VALID_DATE_EARLIER },
                    $report_type
                );
            }, 'missing DateTo in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $VALID_DATE_EARLIER, DateTo => $INVALID_DATE },
                    $report_type
                );
            }, 'invalid DateTo in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    {},
                    $report_type
                );
            }, 'missing DateFrom and DateTo in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $INVALID_DATE, DateTo => $INVALID_DATE },
                    $report_type
                );
            }, 'invalid DateFrom and DateTo in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $INVALID_FORMAT_DATE_FROM, DateTo => $VALID_DATE_LATER },
                    $report_type
                );
            }, 'invalid format DateFrom in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $VALID_DATE_EARLIER, DateTo => $INVALID_FORMAT_DATE_TO },
                    $report_type
                );
            }, 'invalid format DateTo in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $INVALID_FORMAT_DATE_FROM, DateTo => $INVALID_FORMAT_DATE_TO },
                    $report_type
                );
            }, 'invalid format DateFrom and DateTo in '.$report_type;

            ok_validation_success sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $VALID_DATE_EARLIER, DateTo => $VALID_DATE_EARLIER },
                    $report_type
                );
            }, 'DateFrom = DateTo in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $VALID_DATE_LATER, DateTo => $VALID_DATE_EARLIER },
                    $report_type
                );
            }, 'DateFrom > DateTo in '.$report_type;

            ok_validation_success sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $VALID_DATE_EARLIER, DateTo => $DATE_IN_FUTURE },
                    $report_type
                );
            }, 'DateTo in future in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $DATE_IN_FUTURE, DateTo => $DATE_IN_FUTURE },
                    $report_type
                );
            }, 'DateFrom in future in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_date_range(
                    CUSTOM_DATE_RANGE_TYPE,
                    { DateFrom => $DATE_LONG_AGO, DateTo => $DATE_LONG_AGO },
                    $report_type
                );
            }, 'period too far ago in '.$report_type;
        }

        ok_validation_fail sub {
            $valclass->_validate_create_date_range(
                CUSTOM_DATE_RANGE_TYPE,
                { DateFrom => $DATE_NOT_ALLOWED_FOR_SEARCH_QUERY_REPORT, DateTo => $DATE_NOT_ALLOWED_FOR_SEARCH_QUERY_REPORT },
                SEARCH_QUERY_PERFORMANCE_REPORT_TYPE
            );
        }, 'period too far ago in '.SEARCH_QUERY_PERFORMANCE_REPORT_TYPE;
    };

    for my $date_range_type (@date_range_types) {
        next if $date_range_type eq CUSTOM_DATE_RANGE_TYPE;

        subtest $date_range_type => sub {
            plan tests => scalar(@REPORT_TYPES) * 4;

            foreach my $report_type (@REPORT_TYPES) {
                ok_validation_success sub {
                    $valclass->_validate_create_date_range(
                        $date_range_type,
                        {},
                        $report_type
                    );
                }, 'missing DateFrom and DateTo in '.$report_type;

                ok_validation_fail sub {
                    $valclass->_validate_create_date_range(
                        $date_range_type,
                        { DateFrom => $VALID_DATE_EARLIER },
                        $report_type
                    );
                }, 'present DateFrom in '.$report_type;

                ok_validation_fail sub {
                    $valclass->_validate_create_date_range(
                        $date_range_type,
                        { DateTo => $VALID_DATE_LATER },
                        $report_type
                    );
                }, 'present DateTo in '.$report_type;

                ok_validation_fail sub {
                    $valclass->_validate_create_date_range(
                        $date_range_type,
                        { DateFrom => $VALID_DATE_EARLIER, DateTo => $VALID_DATE_LATER },
                        $report_type
                    );
                }, 'present DateFrom and DateTo in '.$report_type;
            }
        };
    }
};

subtest '_validate_create_filter_items' => sub {
    my @criteria_id_fields = qw/AudienceTargetId DynamicTextAdTargetId Keyword SmartBannerFilterId SmartAdTargetId/;
    my @criteria_id_and_type_fields = (@criteria_id_fields, 'CriteriaType', 'CriterionType');
    my @criteria_id_and_type_field_pairs;
    foreach my $field (@criteria_id_and_type_fields) {
        push @criteria_id_and_type_field_pairs, map { $field ne $_ ? [$field, $_] : () } @criteria_id_and_type_fields;
    }
    foreach my $report_type (@REPORT_TYPES) {
        my $not_available_fields = $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_FIELD_NAMES{$report_type};
        my $field_not_available_in_filter = $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_FILTER{$report_type};
        foreach my $field (sort keys %$field_not_available_in_filter) {
            ok_validation_fail sub {
                $valclass->_validate_create_filter_items([{
                    Field => $field,
                    Operator => 'IN',
                    Values => [1]
                }], ['Clicks'], $report_type);
            }, "filter by $field inaccessible in $report_type";
        }

        ok_validation_success sub {
            $valclass->_validate_create_filter_items([{
                Field => 'Clicks',
                Operator => 'IN',
                Values => [100, 200],
            }], ['Clicks'], $report_type);
        }, 'all valid in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_filter_items([{
                Field => 'Clicks',
                Operator => 'SOME_OPERATOR',
            }], ['Clicks'], $report_type);
        }, 'forbidden operator in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_filter_items([{
                Field => 'Clicks',
                Operator => 'EQUALS',
                Values => [qw(more than one)]
            }], ['Clicks'], $report_type);
        }, 'check operator takes only one value in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_filter_items([
            {
                Field => 'Clicks',
                Operator => 'GREATER_THAN',
                Values => [10]
            },
            {
                Field => 'Impressions',
                Operator => 'EQUALS',
                Values => [30]
            },
            {
                Field => 'Clicks',
                Operator => 'LESS_THAN',
                Values => [20]
            },
            ], ['Clicks'], $report_type);
        }, 'only one filter per field allowed in '.$report_type;

        ok_validation_fail sub {
            mock_subs(
                $valclass,
                '_validate_filter_item_value' => [{
                    args   => [ignore(), 'Xxx', 1, {type => 'SomeType', ops => ['SOME_OPERATOR'], p => [1]}, 'SOME_OPERATOR'],
                    result => undef,
                }, {
                    args   => [ignore(), 'Xxx', 2, {type => 'SomeType', ops => ['SOME_OPERATOR'], p => [1]}, 'SOME_OPERATOR'],
                    result => error_BadParams('aaa'),
                }],
            );
            my $res = $valclass->_validate_create_filter_items([{
                Field => 'Xxx',
                Operator => 'SOME_OPERATOR',
                Values => [1, 2],
            }], ['Xxx'], $report_type);
            restore_mocked_subs();
            return $res;
        }, 'invalid value in '.$report_type;

        if (!exists $field_not_available_in_filter->{ClickType}) {
            ok_validation_fail sub {
                $valclass->_validate_create_filter_items([
                {
                    Field => 'ClickType',
                    Operator => 'EQUALS',
                    Values => ['VCARD']
                },
                {
                    Field => 'Impressions',
                    Operator => 'GREATER_THAN',
                    Values => [5]
                },
                ], ['Impressions'], $report_type);
            }, 'forbidden simultaneous use of ClickType and Impressions in Filter in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_filter_items([
                {
                    Field => 'Ctr',
                    Operator => 'GREATER_THAN',
                    Values => [0.05]
                },
                ], ['ClickType'], $report_type);
            }, 'forbidden simultaneous use of ClickType in FieldNames and Ctr in Filter in '.$report_type;

            ok_validation_fail sub {
                $valclass->_validate_create_filter_items([
                {
                    Field => 'ClickType',
                    Operator => 'EQUALS',
                    Values => ['VCARD']
                },
                ], ['AvgImpressionPosition'], $report_type);
            }, 'forbidden simultaneous use of ClickType in Filter and AvgImpressionPosition in FieldNames in '.$report_type;
        }

        foreach my $field_pair (@criteria_id_and_type_field_pairs) {
            next if List::MoreUtils::any { $field_not_available_in_filter->{$_} } @$field_pair;
            ok_validation_fail sub {
                $valclass->_validate_create_filter_items([
                {
                    Field => $field_pair->[0],
                    Operator => 'EQUALS',
                    Values => ['KEYWORD']
                },
                {
                    Field => $field_pair->[1],
                    Operator => 'EQUALS',
                    Values => ['KEYWORD']
                },
                ], ['Impressions'], $report_type);
            }, "forbidden simultaneous use of $field_pair->[0] and $field_pair->[1] in Filter in $report_type";
        }

        if (!$not_available_fields->{CriteriaType}) {
            foreach my $field (@criteria_id_fields) {
                next if $field_not_available_in_filter->{$field};
                ok_validation_success sub {
                    $valclass->_validate_create_filter_items([
                        {
                            Field => $field,
                            Operator => 'EQUALS',
                            Values => [1]
                        },
                    ], ['CriteriaType'], $report_type);
                }, "$field in Filter and CriteriaType in FieldNames for $report_type";
            }
        }

        if (!$not_available_fields->{CriterionType}) {
            foreach my $field (@criteria_id_fields) {
                next if $field_not_available_in_filter->{$field};
                ok_validation_success sub {
                    $valclass->_validate_create_filter_items([
                        {
                            Field => $field,
                            Operator => 'EQUALS',
                            Values => [1]
                        },
                    ], ['CriterionType'], $report_type);
                }, "$field in Filter and CriterionType in FieldNames for $report_type";
            }
        }
    }
};

subtest 'validate_filter_enums' => sub {
    foreach my $report_type (@REPORT_TYPES) {
        my $filter_params_map = $FILTER_PARAMS_MAP{$report_type};
        foreach my $field (sort grep {exists $filter_params_map->{$_}{enum}} keys %$filter_params_map) {
            my $filter_enum = exists $FILTER_ENUMS{$field} ? $FILTER_ENUMS{$field} : undef;
            $filter_enum = $filter_enum->{$report_type} // $filter_enum->{+CUSTOM_REPORT_TYPE} if ref $filter_enum eq 'HASH';
            if ($filter_enum) {
                cmp_deeply($filter_params_map->{$field}{enum}, $filter_enum, "correct contents of enum for $field in $report_type");
            } else {
                fail("filter enum for $field not found in $report_type");
            }
        }
    }
};

subtest 'validate_value_against_type' => sub {

    local $API::Service::Reports::Validation::MAX_STRING_LENGTH = 11;

    my %test_types = (
        Id => [
            # is_valid, value, extra, operator, comment
            [1, 1],

            [0, -1],
            [0, 0],
            [0, 'xx'],
        ],
        Boolean => [
            [1, 'Yes'],
            [1, 'No'],

            [0, 'xx'],
        ],
        Integer => [
            [1, -1],
            [1, 1],
            [1, 0],
            [1, 0, {non_negative => 1}, undef, 'non_negative'],
            [1, 0, {positive_lt => 1}, 'GREATER_THAN', 'positive_lt'],

            [0, -1, {non_negative => 1}, undef, 'non_negative'],
            [0, 0, {positive_lt => 1}, 'LESS_THAN', 'positive_lt'],
            [0, 0.5],
            [0, 'xx'],
        ],
        Float => [
            [1, -1],
            [1, 1],
            [1, 0],
            [1, 0.5],

            [0, 'xx'],
        ],
        String => [
            [1, 'o'],
            [1, 'o', { enum => ['o'] }, undef, 'valid enum'],
            [1, 'o' x 11],

            [0, 'm', { enum => ['o'] }, undef, 'invalid enum'],
            [0, 'o' x 12, undef, undef, 'too long string'],
        ],
        Region => [
            [1, 0],
            [1, 1],

            [0, 1048576],
            [0, 'xxx'],
        ],
        Money => [
            [1, 0],
            [1, 1000000],
            [1, 0, {positive_lt => 1}, 'GREATER_THAN', 'positive_lt'],

            [0, 0, {positive_lt => 1}, 'LESS_THAN', 'positive_lt'],
            [0, 0.5],
            [0, -1],
        ],
    );

    foreach my $type (sort keys %test_types) {
        foreach my $params (@{$test_types{$type}}) {
            my ($is_valid, $value, $filter_params, $operator, $comment) = @$params;
            $comment //= '';
            $filter_params //= {};

            foreach my $report_type (@REPORT_TYPES) {
                if ($is_valid) {
                    ok_no_error sub {
                        $valclass->_validate_filter_item_value('Field', $value, $filter_params, $type, $operator, $report_type)
                    }, "${type} ($value) $comment in $report_type";
                } else {
                    ok_error sub {
                        $valclass->_validate_filter_item_value('Field', $value, $filter_params, $type, $operator, $report_type)
                    }, "${type} ($value) $comment in $report_type";
                }
            }
        }
    }
};

subtest '_validate_create_page' => sub {
    plan tests => 5;

    ok_validation_success sub {
        $valclass->_validate_create_page({
            Limit => 1
        });
    }, 'positive limit';

    ok_validation_fail sub {
        $valclass->_validate_create_page({
            Limit => 0
        });
    }, 'zero limit';

    ok_validation_fail sub {
        $valclass->_validate_create_page({
            Limit => -1
        });
    }, 'negative limit';

    ok_validation_success sub {
        $valclass->_validate_create_page({
            Limit => 2 ** 31 - 1
        });
    }, 'maxint limit';

    ok_validation_fail sub {
        $valclass->_validate_create_page({
            Limit => 2 ** 31
        });
    }, 'over maxint limit';
};

subtest 'validate_create_report_request' => sub {
    plan tests => 4 * scalar(@REPORT_TYPES);

    sub mock_and_check_calls {
        my %methods_with_expected_args = @_;
        my @mock_subs_args = ($valclass,);
        my $call_desc;

        for my $method (keys %methods_with_expected_args) {
            my $expected_args = $methods_with_expected_args{$method};

            if (defined $expected_args) {
                $call_desc = [{
                    args => [ignore(), @{$expected_args}],
                    result => Direct::ValidationResult->new,
                }];
            } else {
                $call_desc = [];
            };
            push @mock_subs_args, $method => $call_desc;
        }
        mock_subs(@mock_subs_args);
    };

    foreach my $report_type (@REPORT_TYPES) {
        my $required_fields = {
            FieldNames => ['a', 'b', 'c'],
            DateRangeType => 'XXX',
            SelectionCriteria => {a => 'b', 'Filter' => [{Field => 1},{Field => 2}]},
            ReportName => 'abc',
            Currency => 'CURRENCY',
            IncludeDiscount => 'YES',
            IncludeVAT => 'YES',
            ReportType => $report_type,
        };
        my $required_calls = {
            '_validate_create_field_names' => [['a', 'b', 'c'], $report_type],
            '_validate_create_date_range' => ['XXX', {a => 'b', 'Filter' => [{Field => 1},{Field => 2}]}, $report_type],
            '_validate_create_report_name' => ['abc'],
            '_validate_create_filter_items' => [[{Field => 1},{Field => 2}], ['a', 'b', 'c'], $report_type, 0],
            '_is_valid_filter_age' => ['XXX', {a => 'b', 'Filter' => [{Field => 1},{Field => 2}]}],
        };

        subtest 'required params' => sub {
            mock_and_check_calls(
                %$required_calls,
                '_validate_create_page' => undef,
                '_validate_create_order_by' => undef,
            );
            $valclass->validate_create_report_request($required_fields);
            ok(1);
            restore_mocked_subs();
        };

        subtest 'validate optional page' => sub {
            mock_and_check_calls(
                %$required_calls,
                '_validate_create_page' => [{z => 'q'}, 0],
            );
            $valclass->validate_create_report_request({
                %$required_fields,
                'Page' => {
                    z => 'q'
                },
            });
            ok(1);
            restore_mocked_subs();
        };

        subtest 'validate optional order_by' => sub {
            mock_and_check_calls(
                %$required_calls,
                '_validate_create_order_by' => [[{Field => 'Zzz'}], ['a', 'b', 'c'], $report_type, 0],
            );
            $valclass->validate_create_report_request({
                %$required_fields,
                'OrderBy' => [
                    {Field => 'Zzz'}
                ]
            });
            ok(1);
            restore_mocked_subs();
        };

        # https://st.yandex-team.ru/DIRECT-54157
        subtest 'validate Currency with IncludeDiscount and IncludeVAT' => sub {
            my @test_params = (
                # is_valid?, Currency, IncludeDiscount, IncludeVat
                [1, 'OTHER_CURRENCY', 'NO', 'NO'],
                [1, 'OTHER_CURRENCY', 'NO', 'YES'],
                [1, 'OTHER_CURRENCY', 'YES', 'NO'],
                [1, 'OTHER_CURRENCY', 'YES', 'YES'],
            );
            foreach my $params (@test_params) {
                my ($is_valid, $currency, $include_discount, $include_vat) = @$params;

                my $valid_txt = $is_valid ? 'should be valid' : 'should be invalid';

                subtest "Currency $currency, IncludeDiscount $include_discount, IncludeVAT $include_vat: $valid_txt" => sub {
                    mock_and_check_calls(%$required_calls);
                    my $res = $valclass->validate_create_report_request({
                        %$required_fields,
                        Currency => $currency,
                        IncludeDiscount => $include_discount,
                        IncludeVAT => $include_vat,
                    });
                    ok($is_valid == $res->is_valid);
                    restore_mocked_subs();
                };
            };
        };
    }
};

subtest '_validate_create_report_name' => sub {
    plan tests => 5;

    ok_validation_success sub {
        $valclass->_validate_create_report_name('aaa');
    }, 'valid string';

    ok_validation_fail sub {
        $valclass->_validate_create_report_name("a\tb");
    }, 'string with tab symbol';

    ok_validation_fail sub {
        $valclass->_validate_create_report_name("x\nx");
    }, 'string with new line symbol';

    ok_validation_fail sub {
        $valclass->_validate_create_report_name('x' x 256);
    }, 'too long string';

    ok_validation_success sub {
        $valclass->_validate_create_report_name('x' x 255);
    }, 'quite long string';
};

subtest '_validate_create_order_by' => sub {
    plan tests => 6 * scalar(@REPORT_TYPES);

    foreach my $report_type (@REPORT_TYPES) {
        ok_validation_success sub {
            $valclass->_validate_create_order_by([
                {
                    Field => 'Clicks',
                    SortOrder => 'DESCENDING',
                }, {
                    Field => 'Impressions'
                },
            ], ['Impressions'], $report_type);
        }, 'correct order_by in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_order_by([
                {
                    Field => 'Clicks',
                    SortOrder => 'DESCENDING',
                }, {
                    Field => 'Clicks'
                },
            ], ['Clicks'], $report_type);
        }, 'duplicate order_by field in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_order_by([
                {
                    Field => 'Date',
                },
            ], ['Clicks'], $report_type);
        }, 'absent date related field in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_order_by([
                {
                    Field => 'GoalId',
                },
            ], ['GoalId'], $report_type);
        }, 'unsupported field in order_by: GoalId in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_order_by([
                {
                    Field => 'Criteria',
                },
            ], ['Criteria'], $report_type);
        }, 'unsupported field in order_by: Criteria in '.$report_type;

        ok_validation_success sub {
            $valclass->_validate_create_order_by([
                {
                    Field => 'Week',
                },
            ], ['Week'], $report_type);
        }, 'date related field both in order_by and field_names in '.$report_type;
    }
};

subtest '_validate_create_field_names' => sub {
    my @all_fields = all_fields();

    plan tests => (scalar(@all_fields) + 3 + 3) * scalar(@REPORT_TYPES);
    foreach my $report_type (@REPORT_TYPES) {
        my $not_available_fields = $API::Test::Reports::Fields::FIELD_NOT_AVAILABLE_IN_FIELD_NAMES{$report_type};
        my @mandatory_fields = @{$MANDATORY_FIELDS{$report_type} // []};
        foreach my $field_name (@all_fields) {
            if ($not_available_fields->{$field_name}) {
                ok_validation_fail sub {
                    $valclass->_validate_create_field_names([uniq($field_name, @mandatory_fields)], $report_type)
                }, "unsupported field $field_name in $report_type";
            } else {
                ok_validation_success sub {
                    $valclass->_validate_create_field_names([uniq($field_name, @mandatory_fields)], $report_type)
                }, "correct field $field_name in $report_type";
            }
        }

        ok_validation_fail sub {
            $valclass->_validate_create_field_names([qw/SmartBannerFilterId AudienceTargetId DynamicTextAdTargetId SmartAdTargetId/], $report_type)
        }, 'several unsupported fields in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_field_names(['Date', 'Week'], $report_type)
        }, 'only one date related field in '.$report_type;

        ok_validation_fail sub {
            $valclass->_validate_create_field_names(['Aa', 'Bb', 'Aa'], $report_type)
        }, 'fields can not be repeated in '.$report_type;

        foreach (
            [ ClickType => [qw/Impressions Ctr AvgImpressionPosition/] ] )
        {
            my $field = $_->[0];
            foreach (@{$_->[1]}) {
                ok_validation_fail sub {
                    $valclass->_validate_create_field_names([$field, $_], $report_type)
                }, "incompatible fields: $field and $_ in $report_type";
            }
        }
    }
};
