#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Scalar::Util qw/looks_like_number/;

use Test::More tests => 18;
use Test::Exception;

BEGIN {
    use_ok('API::Service::Request::Get');
}

my $request = {
    FieldNames => ["Id", "Status", "Type"],
    Page => {
        Limit => 5,
        Offset => 2
    },
    SelectionCriteria => {
        Ids => [153095521],
        Statuses => ['ARCHIVED', 'MODERATION']
    }
};
my $request_without_selection_criteria = {
    FieldNames => ["Id", "Status", "Type"],
    Page => {
        Limit => 5,
        Offset => 2
    },
};
my $request_without_field_names = {
    Page => {
        Limit => 5,
        Offset => 2
    },
    SelectionCriteria => {
        Ids => [153095521],
        Statuses => ['ARCHIVED', 'MODERATION']
    }
};
my $request_without_page = {
    FieldNames => ["Id", "Status", "Type"],
    SelectionCriteria => {
        Ids => [153095521],
        Statuses => ['ARCHIVED', 'MODERATION']
    }
};
my $request_without_ids = {
    FieldNames => ["Id", "Status", "Type"],
    SelectionCriteria => {
        Statuses => ['ARCHIVED', 'MODERATION']
    }
};

subtest 'creates correct object' => sub {
    isa_ok(API::Service::Request::Get->new({}), 'API::Service::Request::Get');
};

subtest 'create correct object with not empty request' => sub {
    lives_ok{ API::Service::Request::Get->new($request) };
};

subtest 'correct selection_criteria' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    is( ref $get_request->selection_criteria(), 'HASH');
    is_deeply( $get_request->selection_criteria(),
        { Ids => [153095521], Statuses => ['ARCHIVED', 'MODERATION'] }
    );
};

subtest 'missing selection_criteria' => sub {
    my $get_request = API::Service::Request::Get->new($request_without_selection_criteria);
    is( ref $get_request->selection_criteria(), 'HASH');
    is_deeply( $get_request->selection_criteria(), {});
};

subtest 'exists selection_ids' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    ok( is_array_of_numbers($get_request->selection_ids('Ids')) );
    is_deeply( [$get_request->selection_ids('Ids')], [153095521] );
};

subtest 'not exists selection_ids' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    is_deeply( [$get_request->selection_ids('StrangeIds')], [] );
};

subtest 'selection_attribute is array' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    is( ref [$get_request->selection_attribute('Statuses')], 'ARRAY' );
};

subtest 'exists selection_attribute' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    is_deeply( [$get_request->selection_attribute('Statuses')], ['ARCHIVED', 'MODERATION'] );
};

subtest 'not exists selection_attribute' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    is_deeply( [$get_request->selection_attribute('StrangeFields')], [] );
};

subtest 'exists page_limit' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    cmp_ok( $get_request->page_limit(), '==', 5);
};

subtest 'not exists page_limit' => sub {
    my $get_request = API::Service::Request::Get->new($request_without_page);
    cmp_ok( $get_request->page_limit(), '>', 0);
};

subtest 'exists page_offset' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    cmp_ok( $get_request->page_offset(), '==', 2);
};

subtest 'not exists page_offset' => sub {
    my $get_request = API::Service::Request::Get->new($request_without_page);
    cmp_ok( $get_request->page_offset(), '==', 0);
};

subtest 'exists field_names' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    is_deeply( [$get_request->field_names()], ["Id", "Status", "Type"] );
};

subtest 'not exists field_names' => sub {
    my $get_request = API::Service::Request::Get->new($request_without_field_names);
    is_deeply( [$get_request->field_names()], [] );
};

subtest 'has_selection_ids' => sub {
    my $get_request = API::Service::Request::Get->new($request);
    ok( $get_request->has_selection_ids());
};

subtest 'not has_selection_ids' => sub {
    my $get_request = API::Service::Request::Get->new($request_without_ids);
    ok( !$get_request->has_selection_ids());
};

sub is_array_of_numbers($) {
    for (@_){
        return 0 if !looks_like_number($_);
    }
    return 1;
}
