#!/usr/bin/perl

use strict;
use warnings;
use utf8;

use Scalar::Util qw/looks_like_number/;
use List::MoreUtils qw/all/;
use Yandex::ListUtils qw/xdiff/;

use Test::More tests => 33;
use Test::Exception;

BEGIN {
    use_ok('API::Service::ResultSet');
}
my $id1 = 10;
my $id2 = 11;

my $result = [
    {
        Id => $id1,
        Name => "New Group 2.1",
        RegionIds => [0],
    },
    {
        Id => $id2,
        RegionIds => [0],
        Name => "New Group 3.1",
    },
];

my $result_with_duplicated_items = [
    {
        Id => 12,
        Name => "New Group 2.1",
        RegionIds => [0],
    },
    {
        Id => 12,
        RegionIds => [0],
        Name => "New Group 3.1",
    },
];

my $warning = MockDefect->new(code => 42, text => 'Warning', description => 'Oops!');
my $error = MockDefect->new(code => 77, text => 'Error', description => 'Oops!');

subtest 'creates correct object' => sub {
    isa_ok( API::Service::ResultSet->new(()), 'API::Service::ResultSet');
};

subtest 'create object with not empty parameters' => sub {
    lives_ok{ API::Service::ResultSet->new(@$result) };
};

subtest 'create object with wrong parameters' => sub {
    dies_ok{ API::Service::ResultSet->new([]) };
};

subtest 'get ok' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    isa_ok( $result_set->get($id1), 'API::Service::ResultSet::Item' );
    is( $result_set->get($id1)->object->{Id}, $id1);
};

subtest 'get not found id' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is( $result_set->get(33), undef );
};

subtest 'list ok' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    isa_ok( ($result_set->list())[0], 'API::Service::ResultSet::Item');
};

subtest 'list empty ResultSet' => sub {
    my $result_set = API::Service::ResultSet->new(());
    is_deeply( [$result_set->list()], [] );
};

subtest '_id_from_object ok' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    my $item = ($result_set->list())[0];
    ok( looks_like_number($result_set->_id_from_object($item->object)) );
    is( $result_set->_id_from_object($item->object), $id1 );
};

subtest 'ids ok' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    my $is_list_of_numbers = all {looks_like_number($_)} $result_set->ids();
    ok( $is_list_of_numbers == 1, 'is list of numbers' );
    is_deeply( [$result_set->ids()], [$id1, $id2] );
};

subtest 'ids empty ResultSet' => sub {
    my $result_set = API::Service::ResultSet->new(());
    is_deeply( [$result_set->ids()], [] );
};

subtest 'has_errors with errors' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->get($id2)->{errors} = [$error];
    is( $result_set->has_errors($id2), 1 );
};

subtest 'has_errors with errors' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is( $result_set->has_errors($id2), 0 );
};

subtest 'has_warnings without warnings' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->get($id1)->{warnings} = [$warning];
    is( $result_set->has_warnings($id1), 1 );
};

subtest 'has_warnings without warnings' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is( $result_set->has_warnings($id1), 0 );
};

subtest 'list_failed with errors' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->get($id2)->{errors} = [$error];
    my @items = $result_set->list_failed();
    isa_ok( $_, 'API::Service::ResultSet::Item' ) for @items;
    is( scalar @items, 1);
};

subtest 'list_failed without errors' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is_deeply( [$result_set->list_failed()], [] );
};

subtest 'list_warned with warnings' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->get($id1)->{warnings} = [$warning];
    my @items = $result_set->list_warned();
    isa_ok( $_, 'API::Service::ResultSet::Item' ) for @items;
    is( scalar @items, 1);
};

subtest 'list_warned without warnings' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is_deeply( [$result_set->list_warned()], [] );
};

subtest 'list_failed_ids with errors' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->get($id2)->{errors} = [$error];
    my @ids = $result_set->list_failed_ids();
    is_deeply( \@ids, [$id2] );
};

subtest 'list_failed_ids without errors' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    my @ids = $result_set->list_failed_ids();
    is_deeply( \@ids, [] );
};

subtest 'list_ok' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    my @items = $result_set->list_ok();
    isa_ok( $_, 'API::Service::ResultSet::Item' ) for @items;
    is( scalar @items, 2);
    $result_set->get($id2)->{errors} = [$error];
    is( scalar $result_set->list_ok(), 1);
};

subtest 'list_ok_ids' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is_deeply( [$result_set->list_ok_ids()], [$id1, $id2] );
    $result_set->get($id2)->{errors} = [$error];
    is_deeply( [$result_set->list_ok_ids()], [$id1] );
};

subtest 'count_ok' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is( $result_set->count_ok(), 2);
    $result_set->get($id2)->{errors} = [$error];
    is( $result_set->count_ok(), 1);
};

subtest 'count_failed' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is( $result_set->count_failed(), 0);
    $result_set->get($id2)->{errors} = [$error];
    is( $result_set->count_failed(), 1);
};

subtest 'item_arrays_by_ids' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    is_deeply( xdiff([keys $result_set->item_arrays_by_ids()], [$id1, $id2]), [] );
    is( ref $_, 'ARRAY' ) for values $result_set->item_arrays_by_ids();
};

subtest 'list_id_duplicated_items with duplicated items' => sub {
    my $result_set = API::Service::ResultSet->new(@$result_with_duplicated_items);
    my @ids = map { $_->object->{Id} } $result_set->list_id_duplicated_items();
    is_deeply(\@ids, [12, 12]);
};

subtest 'list_id_duplicated_items without duplicated items' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    my @ids = map { $_->object->{Id} } $result_set->list_id_duplicated_items();
    is_deeply( \@ids, []);
};

subtest 'add_error' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->add_error($error, $id2);
    is_deeply( $result_set->get($id2)->{errors}, [$error] );
};

subtest 'add_warning' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->add_warning($warning, $id1);
    is_deeply( $result_set->get($id1)->{warnings}, [$warning] );
};

subtest 'add_error_for_id_dups with duplicated items' => sub {
    my $result_set = API::Service::ResultSet->new(@$result_with_duplicated_items);
    $result_set->add_error_for_id_dups($error);
    is_deeply( $result_set->get(12)->{errors}, [$error] );
};

subtest 'add_warning_for_id_dups with duplicated items' => sub {
    my $result_set = API::Service::ResultSet->new(@$result_with_duplicated_items);
    $result_set->add_warning_for_id_dups($warning);
    is_deeply( $result_set->get(12)->{warnings}, [$warning] );
};

subtest 'id_item_to_xml' => sub {
    my $result_set = API::Service::ResultSet->new(@$result);
    $result_set->get($id2)->{errors} = [$error];
    $result_set->get($id1)->{warnings} = [$warning];
    is_deeply( $result_set->id_item_to_xml($id2, $result_set->get($id2)),
        { 
            Errors => [ {Code => 77, Details => "Oops!", Message => "Error"} ]
    });
    is_deeply( $result_set->id_item_to_xml($id1, $result_set->get($id1)),
        { Id => $id1, Warnings => [{Code => 42, Details => "Oops!", Message => "Warning"}] }
        
    );
};

package MockDefect;
sub new {
    my $class = shift;
    my (%params) = @_;

    my $self = \%params;
    bless $self, $class;

    return $self;
}
sub code {
    my $self = shift;
    return $self->{code};
}

sub text {
    my $self = shift;
    return $self->{text};
}

sub description {
    my $self = shift;
    return $self->{description};
}


