#!/usr/bin/env perl

use Direct::Modern;
use open ':std' => ':utf8'; # указываем кодировку utf8 для STDOUT и STDERR

use Carp;
use Try::Tiny;
use Test::More tests => 4;
use Test::MockTime ':all';
use Test::Deep;

use Yandex::I18n;

use Settings;

use API::Authorization::User;
use API::Authorization;
use API::Service::Changes;
use Direct::Errors::Messages;

use API::Test::MockHelper qw/
    mock_subs
    restore_mocked_subs
/;
use API::Test::Utils qw/get_campaign_info_map_from_db_mock_info/;

{
    no warnings qw/redefine once/;
    *API::Service::Changes::ppc_shard = sub { 'ppc' };

    no strict 'refs';
    for (qw/rbac_check_allow_show_camps _get_changed_cids _get_campaigns_stat_changes _get_adgid2cid _get_cid2complete_changed_bids_for_adgids _get_banners_data _get_changed_bids _get_client_cids_and_states _get_cids_with_changed_adgroups _get_cids_with_changed_banners _get_cids_with_stat_changes _get_cid2changed_adgids_for_cids _get_cid2changed_bids check_campaigns_read _get_changed_adgids/) {
        my $method_name = $_;
        *{"API::Service::Changes::$method_name"} = sub { check_method_args_and_get_reply($method_name, @_) };
    }
}

my %methods_replies;

sub check_method_args_and_get_reply {
    my $method_name = shift;

    my ($args, $reply);

    if ($methods_replies{$method_name} and @{$methods_replies{$method_name}}) {
        ($args, $reply) = splice(@{$methods_replies{$method_name}}, 0, 2);
        eq_deeply(\@_, $args) or fail("Wrong args in $method_name");
        # use DDP; cmp_deeply(\@_, $args, 'failed method '.$method_name) || p @_;
    } else {
        fail("No args for $method_name");
    }

    return wantarray ? @$reply : $reply->[0];
}

my $error_BadParams = error_BadParams();
$error_BadParams->{description} = ignore(); # проверяем объект ошибки, исключая текст

subtest 'Testing API::Service::Request::Changes' => sub {
    plan tests => 14;

    ok(API::Service::Request::Changes->new({}), 'had to get API::Service::Request::Changes object');

    try { API::Service::Request::Changes->new({ Timestamp => '' })->mysql_timestamp(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_BadParams, 'had to get the error about the wrong format for empty string in Timestamp') }; # проверяем генерацию исключения для пустого значения Timestamp в запросе

    try { API::Service::Request::Changes->new({ Timestamp => '2015-02-01T08:56:36' })->mysql_timestamp(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_BadParams, 'had to get the error about the wrong format of Timestamp') }; # проверяем генерацию исключения для неверного формата Timestamp в запросе

    try { API::Service::Request::Changes->new({ Timestamp => [ '2015-02-01T08:56:36' ] })->mysql_timestamp(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_BadParams, 'had to get the error about the wrong type of Timestamp') }; # проверяем генерацию исключения при передаче в Timestamp списка значения Timestamp в запросе

    try { API::Service::Request::Changes->new({ Timestamp => '2015-02-01T08:66:36Z' })->mysql_timestamp(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_BadParams, 'had to get the error about an incorrect value for wrong date in Timestamp') }; # проверяем генерацию исключения для неправильной даты в поле Timestamp в запросе

    my $error_RequiredAtLeastOneOfParameters = error_RequiredAtLeastOneOfParameters();
    $error_RequiredAtLeastOneOfParameters->{description} = ignore(); # проверяем объект ошибки, исключая текст

    try { API::Service::Request::Changes->new({})->has_cids(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_RequiredAtLeastOneOfParameters, 'should have received the error object due to lack of field CampaingIds') }; # проверяем генерацию исключения из-за отсутствия поля CampaingIds в запросе

    try { API::Service::Request::Changes->new({})->has_adgids(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_RequiredAtLeastOneOfParameters, 'should have received the error object due to lack of field AdGroupIds') }; # проверяем генерацию исключения из-за отсутствия поля AdGroupIds в запросе

    try { API::Service::Request::Changes->new({})->has_bids(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_RequiredAtLeastOneOfParameters, 'should have received the error object due to lack of field AdIds') }; # проверяем генерацию исключения из-за отсутствия поля AdIds в запросе

    my $error_PossibleOnlyOneParameter = error_PossibleOnlyOneParameter();
    $error_PossibleOnlyOneParameter->{description} = ignore(); # проверяем объект ошибки, исключая текст

    try { API::Service::Request::Changes->new({ CampaignIds => [1], AdGroupIds => [1], AdIds => [1] })->has_cids(); fail('had to get an exception for wrong request') } catch { cmp_deeply($_, $error_PossibleOnlyOneParameter, 'had to get the error object about the wrong request') }; # проверяем генерацию исключения для нескольких взаимоисключающих полей в запросе

    try { API::Service::Request::Changes->new({ CampaignIds => ['test'] })->uniq_cids(); fail('had to get an exception about no int values in CampaignIds') } catch { cmp_deeply($_, $error_BadParams, 'had to get the error object about no int values in CampaignIds') }; # проверяем генерацию исключения для списка идентификаторов в поле CampaignIds, не содержащего числовых значений

    my $error_RequestLimitExceeded = error_RequestLimitExceeded();
    $error_RequestLimitExceeded->{description} = ignore(); # текст ошибки не проверяем

    try { API::Service::Request::Changes->new({ CampaignIds => [1..3001] })->uniq_cids(); fail('had to get an exception for too many CampaignIds in the request') } catch { cmp_deeply($_, $error_RequestLimitExceeded, 'had to get the error object about too many ids in the field CampaignIds') }; # проверяем генерацию исключения при превышении ограничения на количество идентификаторов кампаний в запросе

    try { API::Service::Request::Changes->new({ AdGroupIds => [1..10001] })->uniq_adgids(); fail('had to get an exception for too many AdGroupIds in the request') } catch { cmp_deeply($_, $error_RequestLimitExceeded, 'had to get the error object about too many ids in the field AdGroupIds') }; # проверяем генерацию исключения при превышении ограничения на количество идентификаторов групп в запросе

    try { API::Service::Request::Changes->new({ AdIds => [1..50001] })->uniq_bids(); fail('had to get an exception for too many AdIds in the request') } catch { cmp_deeply($_, $error_RequestLimitExceeded, 'had to get the error object about too many ids in the field AdIds') }; # проверяем генерацию исключения при превышении ограничения на количество идентификаторов объявлений в запросе

    ok(API::Service::Request::Changes->new({ FieldNames => [] })->field_names(), 'processing empty list in FieldNames'); # проверяем нормальную обработку пустого списка в поле FieldNames
};

my $service = API::Service::Changes->new('Changes'); # создаем объект сервиса

# задаем параметры тестового клиента
my $uid = 9600;
my $login = 'fakeuser';

my $user = API::Authorization::User->new(
    uid => $uid,
    login => $login,
    ClientID => 1024,
);
$service->set_subclient($user);
$service->set_protocol('soap');
$service->set_authorization( API::Authorization->fake_new( $uid, $login, { chief_rep_user => $user, operator_user => $user, karma => 0 } ) );

my $request = {};
my ($response, $expected_response);

subtest 'Testing checkDictionaries' => sub {
    plan tests => 6;

    $service->set_current_operation('checkDictionaries');

    mock_subs('API::Service::Changes', _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:01Z' }]);

    ok($response = $service->checkDictionaries($request), 'call to Changes/checkDictionaries with an empty request');
    $expected_response = { Timestamp => '2015-02-17T08:52:01Z' };
    cmp_deeply($response, $expected_response, 'had to get the server timestamp');

    mock_subs(
        'API::Service::Changes',
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:01Z' }],
        _get_dictionaries_changes => [{ args => [ignore, '2015-02-01 11:56:36'], result => {} }],
    );
    $expected_response = {
        Timestamp => '2015-02-17T08:52:01Z',
        InterestsChanged => "NO",
        RegionsChanged   => "NO",
        TimeZonesChanged => "NO",
    };
    $response = $service->checkDictionaries({ Timestamp => '2015-02-01T08:56:36Z' }); # запрос на отсутствие изменений
    cmp_deeply($response, $expected_response, 'had to get information on the absence of changes in both dictionaries');
    restore_mocked_subs();

    mock_subs(
        'API::Service::Changes',
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:01Z' }],
        _get_dictionaries_changes => [{ args => [ignore, '2015-02-01 11:56:36'], result => { geo_regions => 1 } }],
    );
    $response = $service->checkDictionaries({ Timestamp => '2015-02-01T08:56:36Z' }); # запрос на наличие изменений в справочнике регионов
    $expected_response = {
        Timestamp => '2015-02-17T08:52:01Z',
        InterestsChanged => "NO",
        RegionsChanged   => "YES",
        TimeZonesChanged => "NO",
    };
    cmp_deeply($response, $expected_response, 'had to get information on changes in the regions dictionary');
    restore_mocked_subs();

    mock_subs(
        'API::Service::Changes',
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:01Z' }],
        _get_dictionaries_changes => [{ args => [ignore, '2015-02-01 11:56:36'], result => { geo_regions => 1, geo_timezones => 1 } }],
    );
    $response = $service->checkDictionaries({ Timestamp => '2015-02-01T08:56:36Z' }); # запрос на наличие изменений в обоих справочниках
    $expected_response = {
        Timestamp => '2015-02-17T08:52:01Z',
        InterestsChanged => "NO",
        RegionsChanged   => "YES",
        TimeZonesChanged => "YES",
    };
    cmp_deeply($response, $expected_response, 'had to get information on changes in both dictionaries');
    restore_mocked_subs();

    set_fixed_time('2015-02-17 08:52:00', '%Y-%m-%d %H:%M:%S');
    mock_subs(
        'API::Service::Changes',
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
    );
    $response = $service->checkDictionaries({ Timestamp => '2015-02-17T08:52:01Z' }); # запрос на наличие изменений в обоих справочниках
    $expected_response = {
        Timestamp => '2015-02-17T08:52:00Z',
        InterestsChanged => "NO",
        RegionsChanged   => "NO",
        TimeZonesChanged => "NO",
    };
    cmp_deeply($response, $expected_response, 'get checkDictionaries for timestamp in the future');
    restore_time();
};

subtest 'Testing check' => sub {
    plan tests => 13;

    set_fixed_time('2015-02-17 08:52:00', '%Y-%m-%d %H:%M:%S');

    $service->set_current_operation('check');

    $request = {
        CampaignIds => [261],
        FieldNames => [ qw/CampaignIds/ ],
        Timestamp => '2015-02-17T08:52:01Z'
    };
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261]),
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        _split_inaccessible_cids => [{ args => ignore, result_set => [[261],[]]}],
    );
    $expected_response = {
        Modified => {
            CampaignIds => [],
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($service->check($request), $expected_response, 'check by CampaignId with timestamp in the future');
    restore_mocked_subs();

    $request = {
        AdGroupIds => [26101],
        FieldNames => [ qw/AdGroupIds/ ],
        Timestamp => '2015-02-17T08:52:01Z'
    };
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261]),
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        _get_adgid2cid => [{ args => [ignore, [26101]], result => { 26101 => 261 } }],
        _split_inaccessible_cids => [{ args => ignore, result => [261]}],
    );
    $expected_response = {
        Modified => {
            AdGroupIds => [],
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($service->check($request), $expected_response, 'check by AdGroupId with timestamp in the future');
    restore_mocked_subs();

    $request = {
        AdIds => [2610101],
        FieldNames => [ qw/AdIds/ ],
        Timestamp => '2015-02-17T08:52:01Z'
    };
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261]),
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        _get_banners_data => [{ args => [ignore, [2610101]], result => [{ bid => 2610101, pid => 26101, cid => 261 }] }],
        _split_inaccessible_cids => [{ args => ignore, result => [261]}],
    );
    $expected_response = {
        Modified => {
            AdIds => [],
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($service->check($request), $expected_response, 'check by AdId with timestamp in the future');
    restore_mocked_subs();

    $request = {
        CampaignIds => [261,301,265,264,263,270,'test',262,266,302,262,267,268,301,269], # запрос изменений в кампаниях
        FieldNames => [ qw/CampaignIds AdGroupIds AdIds CampaignsStat/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    %methods_replies = (
        check_campaigns_read => [
            bag(ignore(),261,264,265,266,267,263,270,262,268,269,301,302) => []
        ],
        _get_changed_cids => [
            [ignore(),bag(261,264,263,270,262,268,269),'2015-01-01 13:56:36'] => [{ 263 => 1, 264 => 1, 270 => 1 }]
        ],
        _get_campaigns_stat_changes => [
            [ignore(),bag(261,264,263,270,262,268,269),'2015-01-01 13:56:36'] => [{ 262 => '2015-01-28', 264 => '2015-01-28' }]
        ],
        _get_cid2changed_adgids_for_cids => [
            [ignore(),bag(261,264,263,270,262,268,269),'2015-01-01 13:56:36'] => [ { 262 => [26202,26203], 264 => [26402,26403], 269 => [26901], 270 => [27001] }, undef ]        ],
        _get_cid2changed_bids => [
            [ignore(),bag(261,264,263,270,262,268,269),'2015-01-01 13:56:36'] => [ { 262 => [2620101,2620102,2620103,2620104,2620302,2620303], 264 => [2640101,2640102,2640301,2640302,2640303], 269 => [2690101], 270 => [2700101] }, undef ]
        ],
    );
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,264,263,270,262,268,269,270]),
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        'Property::get' => [{ args => [ignore, ignore], result => 0 }],
    );

    ok($response = $service->check($request), 'call to Changes/check with the normal request');
    $expected_response = {
        Modified => {
            CampaignIds => [263,264,270],
            AdGroupIds => [26202,26203,26402,26403,26901,27001],
            AdIds => [2620101,2620102,2620103,2620104,2620302,2620303,2640101,2640102,2640301,2640302,2640303,2690101,2700101],
            CampaignsStat => [
                {
                    CampaignId => 262,
                    BorderDate => '2015-01-28'
                },
                {
                    CampaignId => 264,
                    BorderDate => '2015-01-28'
                }
            ]
        },
        NotFound => {
            CampaignIds => [265,266,267,301,302]
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'had to get the changed and not found campaigns');

    $request = {
        CampaignIds => [261,301,265,264,272,263,270,'test',262,266,271,302,262,267,268,301,269], # запрос изменений в кампаниях
        FieldNames => [ qw/CampaignIds AdGroupIds AdIds CampaignsStat/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    %methods_replies = (
        check_campaigns_read => [
            bag(ignore(),261,264,265,266,267,263,270,271,272,262,268,269,301,302) => []
        ],
        _get_changed_cids => [
            [ignore(),bag(261,264,272,263,270,262,271,268,269),'2015-01-01 13:56:36'] => [{ 263 => 1, 264 => 1, 270 => 1, 271 => 1 }]
        ],
        _get_campaigns_stat_changes => [
            [ignore(),bag(261,264,272,263,270,262,271,268,269),'2015-01-01 13:56:36'] => [{ 262 => '2015-01-28', 264 => '2015-01-28' }]
        ],
        _get_cid2changed_adgids_for_cids => [
            [ignore(),bag(261,264,272,263,270,262,271,268,269),'2015-01-01 13:56:36'] => [ { 262 => [26202,26203], 264 => [26402,26403], 269 => [26901], 270 => [27001], 271 => [2710001..2711000] }, undef ]
        ],
        _get_cid2changed_bids => [
            [ignore(),bag(261,264,272,263,270,262,271,268,269),'2015-01-01 13:56:36'] => [ { 262 => [2620101,2620102,2620103,2620104,2620302,2620303], 264 => [2640101,2640102,2640301,2640302,2640303], 269 => [2690101], 270 => [2700101], 271 => [271000001..271030000] }, 272 ]
        ],
    );
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,264,272,263,270,262,271,268,269]),
        'Property::get' => [{ args => [ignore, ignore], result => 0 }],
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
    );

    $response = $service->check($request);
    $expected_response = {
        Modified => {
            CampaignIds => [263,264,270,271],
            AdGroupIds => [26202,26203,26402,26403,26901,27001,2710001..2711000],
            AdIds => [2620101,2620102,2620103,2620104,2620302,2620303,2640101,2640102,2640301,2640302,2640303,2690101,2700101,271000001..271030000],
            CampaignsStat => [
                {
                    CampaignId => 262,
                    BorderDate => '2015-01-28'
                },
                {
                    CampaignId => 264,
                    BorderDate => '2015-01-28'
                }
            ]
        },
        Unprocessed => {
            CampaignIds => [272]
        },
        NotFound => {
            CampaignIds => [265,266,267,301,302]
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'had to get the first part of the limited response');

    $request = {
        CampaignIds => [272],
        FieldNames => [ qw/CampaignIds AdGroupIds AdIds CampaignsStat/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    $request->{CampaignIds} = [272];
    %methods_replies = (
        check_campaigns_read => [
            [ignore(),272] => []
        ],
        _get_changed_cids => [
            [ignore(),[272],'2015-01-01 13:56:36'] => [{ 272 => 1 }]
        ],
        _get_campaigns_stat_changes => [
            [ignore(),[272],'2015-01-01 13:56:36'] => [{}]
        ],
        _get_cid2changed_adgids_for_cids => [
            [ignore(),[272],'2015-01-01 13:56:36'] => [ { 272 => [2720001..2721000] }, undef ]
        ],
        _get_cid2changed_bids => [
            [ignore(),[272],'2015-01-01 13:56:36'] => [ { 272 => [272000001..272030000] }, undef ]
        ],
    );
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [272]),
        'Property::get' => [{ args => [ignore, ignore], result => 0 }],
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
    );

    $response = $service->check($request);
    $expected_response = {
        Modified => {
            CampaignIds => [272],
            AdGroupIds => [2720001..2721000],
            AdIds => [272000001..272030000],
            CampaignsStat => []
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'had to get the second part of the limited response');

    $request = {
        AdGroupIds => [26101,26501,26202,26301,'test',26201,26402,27001,26203,26403,26701,26801,26601,26901,26201],
        FieldNames => [ qw/AdGroupIds AdIds/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    %methods_replies = (
        check_campaigns_read => [
            bag(ignore(),261,263,262,269,264,265,266,267,270) => []
        ],
        _get_adgid2cid => [
            [ignore(),bag(26101,26501,26202,26301,26201,26402,27001,26203,26403,26701,26801,26601,26901)] => [{ 26101 => 261, 26501 => 265, 26202 => 262, 26301 => 263, 26201 => 262, 26402 => 264, 27001 => 270, 26203 => 262, 26403 => 264, 26701 => 267, 26601 => 266, 26901 => 269 }]
        ],
        _get_changed_adgids => [
            [ignore(),bag(26101,26201,26202,26203,26301,26402,26403,26901,27001),'2015-01-01 13:56:36'] => [[ 26202,26403,26901,26402,27001,26203 ]]
        ],
        _get_cid2complete_changed_bids_for_adgids => [
            [ignore(),bag(26101,26301,26202,26201,26203,26901,26403,26402,27001),'2015-01-01 13:56:36'] => [{ 262 => [2620101,2620102,2620103,2620104,2620302,2620303], 264 => [2640301,2640302,2640303], 269 => [2690101], 270 => [2700101] }],
        ]
    );
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,263,262,269,264,270]),
        'Property::get' => [{ args => [ignore, ignore], result => 0 }],
        _get_current_timestamp_iso8601 => [{ args => [ignore()], result => '2015-02-17T08:52:00Z', }],
    );

    $response = $service->check($request); # получаем наличие изменений в группах и баннерах
    $expected_response = {
        Modified => {
            AdGroupIds => [26202,26203,26402,26403,26901,27001],
            AdIds => [2620101,2620102,2620103,2620104,2620302,2620303,2640301,2640302,2640303,2690101,2700101]
        },
        NotFound => {
            AdGroupIds => [26501,26601,26701,26801]
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'had to get the ids of changed adgoups and banners'); # проверяем, что вернулся правильный объект ошибки

    $request = {
        AdIds => [2610101,2640303,2650101,2620103,2680101,'test',2610102,2620103,2660101,2640301,2620201,2670101,2700101],
        FieldNames => [ qw/CampaignIds AdGroupIds AdIds CampaignsStat/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    %methods_replies = (
        _get_banners_data => [
            [ignore(),[2610101,2640303,2650101,2620103,2680101,2610102,2660101,2640301,2620201,2670101,2700101]] => [[
                { bid => 2610101, pid => 26101, cid => 261 },
                { bid => 2640303, pid => 26403, cid => 264 },
                { bid => 2650101, pid => 26501, cid => 265 },
                { bid => 2620103, pid => 26201, cid => 262 },
                { bid => 2610102, pid => 26101, cid => 261 },
                { bid => 2660101, pid => 26601, cid => 266 },
                { bid => 2640301, pid => 26403, cid => 264 },
                { bid => 2620201, pid => 26202, cid => 262 },
                { bid => 2670101, pid => 26701, cid => 267 },
                { bid => 2700101, pid => 27001, cid => 270 },
            ]]
        ],
        check_campaigns_read => [
            bag(ignore(),261,262,264,265,266,267,270) => []
        ],
        _get_adgid2cid => [
            [ignore(),bag(26101,26501,26202,26301,26201,26402,27001,26203,26403,26701,26801,26601,26901)] => [{ 26101 => 261, 26501 => 265, 26202 => 262, 26301 => 263, 26201 => 262, 26402 => 264, 27001 => 270, 26203 => 262, 26403 => 264, 26701 => 267, 26601 => 266, 26901 => 269 }]
        ],
        _get_changed_cids => [
            [ignore(),bag(261,262,264,270),'2015-01-01 13:56:36'] => [{ 270 => 1, 264 => 1 }]
        ],
        _get_campaigns_stat_changes => [
            [ignore(),bag(261,262,264,270),'2015-01-01 13:56:36'] => [{ 262 => '2015-01-28', 264 => '2015-01-28' }]
        ],
        _get_changed_adgids => [
            [ignore(),bag(26101,26201,26202,26403,27001),'2015-01-01 13:56:36'] => [[ 26202,26403,27001 ]]
        ],
        _get_changed_bids => [
            [ignore(),bag(2610101,2610102,2620103,2620201,2640303,2640301,2700101),'2015-01-01 13:56:36'] => [[2620103,2640303,2700101]]
        ]
    );
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,262,264,270]),
        'Property::get' => [{ args => [ignore, ignore], result => 0 }],
        _get_current_timestamp_iso8601 => [{ args => [ignore()], result => '2015-02-17T08:52:00Z', }],
    );

    $response = $service->check($request); # получаем наличие изменений в баннерах
    $expected_response = {
        Modified => {
            CampaignIds => [264,270],
            AdGroupIds => [26202,26403,27001],
            AdIds => [2620103,2640303,2700101],
            CampaignsStat => [
                {
                    CampaignId => 262,
                    BorderDate => '2015-01-28'
                },
                {
                    CampaignId => 264,
                    BorderDate => '2015-01-28'
                }
            ]
        },
        NotFound => {
            AdIds => [2650101,2660101,2670101,2680101]
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'had to get the ids of changed banners'); # проверяем, что вернулся правильный ответ

    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,264,265]),
        'Property::get' => [{ args => [ignore, ignore], result => 1 }],
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        _get_banners_data => [{ args => [ignore, [261,264,265]], result => [] }],
        check_campaigns_read => [{ args => [ignore, 261, 264, 265], result_set => [] }],
        _get_cid2changed_adgids_for_cids => [{ args => [ignore, [261,264,265], '2015-01-01 13:56:36'], result_set => [{264 => [26401, 26402]}, undef] }],
    );
    $request = {
        CampaignIds => [261,264,265],
        FieldNames => [ qw/AdGroupIds/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    $response = $service->check($request); # получаем наличие изменений в баннерах
    $expected_response = {
        Modified => {
            AdGroupIds => [26401,26402],
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'get changed adgroups by CampaignIds with camp_aggregated_lastchange'); # проверяем, что вернулся правильный ответ
    restore_mocked_subs();

    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,264,265]),
        'Property::get' => [{ args => [ignore, ignore], result => 1 }],
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        _filter_inaccessible_cids => [{ args => [ignore,bag(261,264,265)], result => [261,264,265]}],
        _get_adgid2cid => [{ args => [ignore, [26101,26401,26501]], result => {26101 => 261, 26401 => 264, 26501 => 265}}],
        _get_changed_adgids => [{ args => [ignore, bag(26101,26401,26501), '2015-01-01 13:56:36'], result => [26401,26501]}],
    );
    $request = {
        AdGroupIds => [26101,26401,26501],
        FieldNames => [ qw/AdGroupIds/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    $response = $service->check($request); # получаем наличие изменений в баннерах
    $expected_response = {
        Modified => {
            AdGroupIds => [26401,26501],
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'get changed adgroups by AdGroupIds with camp_aggregated_lastchange'); # проверяем, что вернулся правильный ответ
    restore_mocked_subs();

    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,264,265]),
        'Property::get' => [{ args => [ignore, ignore], result => 1 }],
        _get_banners_data => [{ args => [ignore, [2610101,2640101,2650101]], result => [{bid => 2610101, pid => 26101, cid => 261},{bid => 2640101, pid => 26401, cid => 264},{bid => 2650101, pid => 26501, cid => 265}] }],
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        _filter_inaccessible_cids => [{ args => [ignore,bag(261,264,265)], result => [261,264,265]}],
        _get_cid2changed_adgids_for_adgids => [{ args => [ignore,{26101 => 261, 26401 => 264, 26501 => 265},'2015-01-01 13:56:36'], result => {264 => [26401], 265 => [26501]} }],
        _get_changed_bids => [{ args => [ignore,[2610101,2640101,2650101],'2015-01-01 13:56:36'], result => [2640101,2650101] }],
    );
    $request = {
        AdIds => [2610101,2640101,2650101],
        FieldNames => [ qw/AdGroupIds AdIds/ ],
        Timestamp => '2015-01-01T10:56:36Z'
    };
    $response = $service->check($request); # получаем наличие изменений в баннерах
    $expected_response = {
        Modified => {
            AdGroupIds => [26401,26501],
            AdIds => [2640101, 2650101],
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'get changed adgroups and ads by AdIds with camp_aggregated_lastchange'); # проверяем, что вернулся правильный ответ
    restore_mocked_subs();

    $request = {
        CampaignIds => [261],
        FieldNames => [ qw/CampaignIds/ ],
        Timestamp => '2014-12-20T00:00:00Z'
    };
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261]),
        _get_current_timestamp_iso8601 => [{ args => [ignore()], result => '2015-02-17T08:52:00Z', }],
        _get_changed_cids => [{ args => [ignore(),[261],'2014-12-20 03:00:00'], result => { 261 => 1 } }],
        check_campaigns_read => [{ args => [ignore(),261], result_set => [] }],
        'Property::get' => [{ args => [ignore, ignore], result => 0 }],
    );

    $response = $service->check($request); # получаем наличие изменений в баннерах
    $expected_response = {
        Modified => {
            CampaignIds => [261],
        },
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'had to get the id of changed campaign'); # проверяем, что вернулся правильный ответ

    restore_time();
    restore_mocked_subs();
};

subtest 'Testing checkCampaigns' => sub {
    plan tests => 6;

    $service->set_current_operation('checkCampaigns');
    $request = { Timestamp => '2015-01-01T10:56:36Z' };

    %methods_replies = (
        _get_client_cids_and_states => [
            [ignore(),'2015-01-01 13:56:36'] => [[
                { cid => 261, OrderID => 1261, updated => 0 },
                { cid => 262, OrderID => 1262, updated => 0 },
                { cid => 263, OrderID => 1263, updated => 1 },
                { cid => 264, OrderID => 1264, updated => 1 },
                { cid => 265, OrderID => 1265, updated => 0 },
                { cid => 266, OrderID => 1266, updated => 1 },
                { cid => 267, OrderID => 1267, updated => 1 },
                { cid => 269, OrderID => 1269, updated => 0 },
                { cid => 270, OrderID => 1270, updated => 1 },
                { cid => 271, OrderID => 1271, updated => 0 },
                { cid => 272, OrderID => 1272, updated => 0 }
            ]]
        ],
        check_campaigns_read => [
            [ignore(),261,262,263,264,265,266,267,269,270,271,272] => []
        ],
        _get_cids_with_changed_adgroups => [
            [ignore(),[261,262,263,264,265,266,267,269,270,271,272],'2015-01-01 13:56:36'] => [[262,265,270]]
        ],
        _get_cids_with_changed_banners => [
            [ignore(),[261,263,264,266,267,269,271,272],'2015-01-01 13:56:36'] => [[264,267,269]]
        ],
        _get_cids_with_stat_changes => [
            [ignore(),{1261 => 261,1262 => 262,1263 => 263,1264 => 264,1265 => 265,1266 => 266,1267 => 267,1269 => 269,1270 => 270,1271 => 271,1272 => 272},'2015-01-01 13:56:36'] => [[262,264,265,267]]
        ]
    );
    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,262,263,264,265,266,267,269,270,271,272]),
        'Property::get' => [{ args => ignore, result => 0 }, { args => ignore, result => 0 }],
        _get_current_timestamp_iso8601 => [{ args => [ignore()], result => '2015-02-17T08:52:00Z', }],
        'API::Service::Base::get_http_request_header' => [{ args => [ignore, 'useCampAggregatedLastChange'], result => undef }, { args => [ignore, 'useCampAggregatedLastChange'], result => undef }],
    );

    ok($response = $service->checkCampaigns($request), 'call to Changes/checkCampaigns with the normal request');
    $expected_response = {
        Campaigns => [
            {
                CampaignId => 262,
                ChangesIn  => [ 'CHILDREN', 'STAT' ]
            },
            {
                CampaignId => 263,
                ChangesIn  => [ 'SELF' ]
            },
            {
                CampaignId => 264,
                ChangesIn  => [ 'SELF', 'CHILDREN', 'STAT' ]
            },
            {
                CampaignId => 265,
                ChangesIn  => [ 'CHILDREN', 'STAT' ]
            },
            {
                CampaignId => 266,
                ChangesIn  => [ 'SELF' ]
            },
            {
                CampaignId => 267,
                ChangesIn  => [ 'SELF', 'CHILDREN', 'STAT' ]
            },
            {
                CampaignId => 269,
                ChangesIn  => [ 'CHILDREN' ]
            },
            {
                CampaignId => 270,
                ChangesIn  => [ 'SELF', 'CHILDREN' ]
            },
        ],
        Timestamp => '2015-02-17T08:52:00Z'
    };
    cmp_deeply($response, $expected_response, 'had to get types of changes in campaigns'); # проверяем, что вернулся правильный результат

    cmp_deeply($service->checkCampaigns({}), $error_BadParams, 'had to get the error object for empty request'); # проверяем получение объекта ошибки для пустого запроса

    mock_subs(
        'API::Service::Changes',
        'API::Service::CampaignsAvailabilityChecker::get_campaign_info_map_from_db' => get_campaign_info_map_from_db_mock_info(uid => $uid, cids => [261,262,263,264,265,266,267,269,270,271,272]),
        'Property::get' => [{ args => ignore, result => 1 }, { args => ignore, result => '2015-03-17T11:51:57' }, { args => ignore, result => 1 }],
        _get_client_cids_and_states => [{
            args => [ignore(),'2015-01-01 13:56:36'],
            result => [
                { cid => 261, OrderID => 1261, updated => 0 },
                { cid => 262, OrderID => 1262, updated => 0 },
                { cid => 263, OrderID => 1263, updated => 1 },
                { cid => 264, OrderID => 1264, updated => 1 },
                { cid => 265, OrderID => 1265, updated => 0 },
                { cid => 266, OrderID => 1266, updated => 1 },
                { cid => 267, OrderID => 1267, updated => 1 },
                { cid => 269, OrderID => 1269, updated => 0 },
            ],
        }],
        check_campaigns_read => [{ args => [ignore(),261,262,263,264,265,266,267,269], result_set => [] }],
        _get_cids_with_stat_changes => [{
            args => [ignore(),{1261 => 261,1262 => 262,1263 => 263,1264 => 264,1265 => 265,1266 => 266,1267 => 267,1269 => 269},'2015-01-01 13:56:36'],
            result => [262,264,265,267],
        }],
        _get_cids_with_changed_children => [{ args => [ignore, bag(261,262,263,264,265,266,267,269),'2015-01-01 13:56:36'], result => [265,267,269] }],
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-03-17T08:52:00Z' }],
        'Yandex::DBShards::get_shard_multi' => [{ args => [ ClientID => [1024], undef ], result => { 1024 => 8 } }],
        'API::Service::Base::get_http_request_header' => [{ args => [ignore, 'useCampAggregatedLastChange'], result => undef }, { args => [ignore, 'useCampAggregatedLastChange'], result => undef }],
    );

    $expected_response = {
        Campaigns => [
            {
                CampaignId => 262,
                ChangesIn  => [ 'STAT' ]
            },
            {
                CampaignId => 263,
                ChangesIn  => [ 'SELF' ]
            },
            {
                CampaignId => 264,
                ChangesIn  => [ 'SELF', 'STAT' ]
            },
            {
                CampaignId => 265,
                ChangesIn  => [ 'CHILDREN', 'STAT' ]
            },
            {
                CampaignId => 266,
                ChangesIn  => [ 'SELF' ]
            },
            {
                CampaignId => 267,
                ChangesIn  => [ 'SELF', 'CHILDREN', 'STAT' ]
            },
            {
                CampaignId => 269,
                ChangesIn  => [ 'CHILDREN' ]
            },
        ],
        Timestamp => '2015-03-17T08:51:57Z'
    };
    cmp_deeply($service->checkCampaigns($request), $expected_response, 'had to get types of changes in campaigns using camp_aggregated_lastchange'); # проверяем, что вернулся правильный результат

    set_fixed_time('2015-02-17 08:52:00', '%Y-%m-%d %H:%M:%S');
    mock_subs(
        'API::Service::Changes',
        'Property::get' => [{ args => ignore, result => 0 }],
        _get_current_timestamp_iso8601 => [{ args => ignore, result => '2015-02-17T08:52:00Z' }],
        'Yandex::DBShards::get_shard_multi' => [{ args => [ ClientID => [1024], undef ], result => { 1024 => 8 } }],
        'API::Service::Base::get_http_request_header' => [{ args => [ignore, 'useCampAggregatedLastChange'], result => undef }],
    );
    $response = $service->checkCampaigns({ Timestamp => '2015-02-17T08:52:01Z' });
    $expected_response = {
        Campaigns => [],
        Timestamp => '2015-02-17T08:52:00Z',
    };
    cmp_deeply($response, $expected_response, 'get checkCampaigns for timestamp in the future');

    mock_subs(
        'API::Service::Changes',
        'Property::get' => [{ args => ignore, result => 1 }, { args => ignore, result => '2015-02-17T11:51:47' }],
        'Yandex::DBShards::get_shard_multi' => [{ args => [ ClientID => [1024], undef ], result => { 1024 => 8 } }],
        'API::Service::Base::get_http_request_header' => [{ args => [ignore, 'useCampAggregatedLastChange'], result => undef }],
    );
    $response = $service->checkCampaigns({ Timestamp => '2015-02-17T08:52:01Z' });
    $expected_response = {
        Campaigns => [],
        Timestamp => '2015-02-17T08:51:47Z',
    };
    cmp_deeply($response, $expected_response, 'get checkCampaigns for timestamp in the future with camp_aggregated_lastchange');
    restore_time();

};

done_testing;
