#!/usr/bin/perl

use Direct::Modern;

use Try::Tiny;
use Test::More;
use Test::Deep;

use Yandex::Test::UTF8Builder;

use API::Authorization::User;
use API::Authorization;
use API::UserConnectionLock::Fake;

my $login = 'fake_agency';
my $uid = 12345;
my $ClientID = 15155;
my $subclient_allowed_concurrent_calls = 5;

my $agency1 = API::Authorization::User->new(
    uid => $uid,
    login => $login,
    ClientID => $ClientID,
    special_options => {
        concurrent_calls => 1
    },
);

my $agency2 = API::Authorization::User->new(
    uid => $uid+1,
    login => $login . "_2",
    ClientID => $ClientID+1,
    special_options => {
        concurrent_calls => 3
    },
);

my $subclient = API::Authorization::User->new(
    uid => $uid+100,
    login => "${login}_subclient",
    ClientID => $ClientID+100,
    special_options => {
        concurrent_calls => $subclient_allowed_concurrent_calls
    },
);

my $subclient2 = API::Authorization::User->new(
    uid => $uid+101,
    login => "${login}_subclient2",
    ClientID => $ClientID+101,
    special_options => {
        concurrent_calls => 7
    },
);

{
    no warnings;
    *API::Authorization::subclient_by_login = sub { $subclient };
    *API::Authorization::run_service_operation = sub { $subclient };
};


ok($subclient->allowed_concurrent_calls == $subclient_allowed_concurrent_calls, "allowed concurrent calls");

subtest limiter => sub {
    my $limiter = API::UserConnectionLock::Fake->new($subclient, $subclient->allowed_concurrent_calls);

    ok($limiter->current == 0, "zero counter at start");
    ok($limiter->get == 1, "got first lock");
    ok($limiter->get == 2, "got second lock");
    ok((my $current_locks = $limiter->get) == 3, "got third lock");
    ok($limiter->release == 2, "release lock");
    ok(($current_locks - $limiter->current) == 1, "locks decreased by one after release");
};

subtest max_subclient_connections => sub {
    flush_counters();
    my $limiter = API::UserConnectionLock::Fake->new($subclient, $subclient->allowed_concurrent_calls);
    foreach my $i (1..$subclient_allowed_concurrent_calls) {
        ok($limiter->get, "$i lock ok");
    }

    ok(!$limiter->get, ($subclient_allowed_concurrent_calls + 1) . " lock fails");
    $limiter->release;
    ok($limiter->get, $limiter->current . " lock ok again after release");
};

# Тут будет метод вызова API от оператора и сабклиента проверяюащий лимит но, не сбрасывающий лок по выходу.

# тест который проверяет, что при обращении двух разных агентств из под одного сабклиента и превышении лимита не влезшие в лимит вызовы падают
subtest agency_shares_same_subclient => sub {
    flush_counters();
    my $auth_agency1 = API::Authorization->fake_new($uid, $login, { chief_rep_user => $agency1, operator_user => $agency1, karma => 0 });
    my $auth_agency2 = API::Authorization->fake_new($uid, $login, { chief_rep_user => $agency2, operator_user => $agency2, karma => 0 });
    my $api = api_psgi_mock->new();

    do_max_counter($auth_agency1, $subclient, 1);
    #ok(!$failed, "made $subclient_allowed_concurrent_calls concurrent calls, for subclient while agency has settings for only 3");
    ok($api->try_concurrency_lock_and_run(undef , undef, $auth_agency2, $subclient) eq 'operator failure', "agency2 can't use api for client if it's concurrent limit exceeded");
};

# тест, который проверяет что один клиент агентсва может обращаться к двум разным сабклиентам одновременно с количеством параллельных соединений N+M > MAX, где N и M кол-во параллельных коннектов на каждого клиента
subtest agency_with_two_subclients => sub {
    flush_counters();
    my $auth = API::Authorization->fake_new($uid, $login, { chief_rep_user => $agency1, operator_user => $agency1, karma => 0 });
    ok($agency1->allowed_concurrent_calls < ($subclient->allowed_concurrent_calls + $subclient2->allowed_concurrent_calls), " agency limit less that sum of subclient limits");
    do_max_counter($auth, $subclient);
    do_max_counter($auth, $subclient2);
    ok(do_connection($auth, $subclient) eq 'operator failure', "agency can't use api for subclient if it's concurrent limit exceeded");
    ok(do_connection($auth, $subclient2) eq 'operator failure', "agency can't use api for subclient2 if it's concurrent limit exceeded");
};

done_testing;

sub do_connection {
    my ($operator, $subclient, $dont_release_lock) = @_;
    my $request = $dont_release_lock ?  { dont_release_lock => 1 } : undef;
    return api_psgi_mock->new()->try_concurrency_lock_and_run($request, undef, $operator, $subclient);
}

sub do_max_counter { # забиваем счетчик соединений о максимума
    my ($operator, $subclient) = @_;
    my $max = $subclient->allowed_concurrent_calls;
    foreach my $i (1..$max) {
        my $result;
        try {
            $result = do_connection($operator, $subclient, 1);
        };
        die "somehow counter full already" if $result && $result eq 'operator failure';
    }
    ok(current_locks($subclient) == $max, $subclient->login . " counter maxed");
}

sub lock_storage { $API::UserConnectionLock::Fake::USERS }
sub flush_counters { API::UserConnectionLock::Fake::FLUSH() }
sub current_locks {
    my $user = shift;
    lock_storage()->{ $user->client_id };
}

{
    package api_psgi_mock;

    use base qw/API::PSGI::Base/;

    sub user_connection_lock {
        my ($self, $user) = @_;
        return API::UserConnectionLock::Fake->new($user, $user->allowed_concurrent_calls);
    }

    sub run_service_operation {
        my ($self, $request) = @_;
        if ($request->{dont_release_lock}) {
            die 'wont relese lock';
        }
        return 1;
    }

    sub handle_operator_failure {
        return "operator failure";
    }

    sub client_fault {'client fault' }

    sub server_fault { 'server fault' }
}

