#!/usr/bin/env perl

=head1 NAME

apache_reloader - перезапускает апач при изменении .pm-файлов.

=head1 SYNOPSIS

    perl bin/apache_reloader.pl

или

    direct-mk apache-reloader

=head1 DESCRIPTION

Висит запущенным, когда меняются файлы (пере)запускает апач. По
умолчанию пытается выполнить мягкий перезапуск, когда это не удается
(не был запущен или упал из-за ошибки синтаксиса во время предыдущей
перезагрузки) запускает его заново.

Ждёт 0.1 секунды после измения последнего файла в серии.

Удалось или нет апачу перезапуститься можно оценивать на глаз по его error.log

=cut
use strict;
use warnings;
use utf8;

use FindBin;
use AnyEvent;
use AnyEvent::Handle;
use Linux::Inotify2;
use File::Find;
use File::Slurp;
use POSIX;
use Data::Dumper;
use 5.14.0;

use my_inc for => 'api/t';

{
    our %MASKS;
    for my $mask (@Linux::Inotify2::EXPORT) {
        next if $mask !~ /^IN_/ or $mask eq 'IN_ALL_EVENTS';
        $MASKS{Linux::Inotify2->$mask} = $mask;
    }
}

my $reload_timer;

sub main {
    our %W; # место для хранения AnyEvent watcher'ов, чтобы они не уничтожились раньше времени.
    my $inotify = Linux::Inotify2->new or die "inotify: $!";
    find(
        {
            wanted => sub {
                if ($_ eq '.svn') {
                    $File::Find::prune = 1;
                    return;
                }
                if (-d $_) {
                    $W{$File::Find::name} = create_watcher($inotify, $File::Find::fullname);
                }
            },
            follow => 1,
        },
        @my_inc::MY_INC
    );

    my $inotify_w = AnyEvent->io(
        fh => $inotify->fileno, poll => 'r', cb => sub { $inotify->poll },
    );

    my $log_tail = create_log_watcher();
    AnyEvent->condvar->recv;
}

sub create_log_watcher {
    open my $log_fh, "|-", "tail -F $FindBin::Bin/../apache/logs/error.log";
    my $log_tail; $log_tail = AnyEvent::Handle->new(
        fh => $log_fh,
        on_error => sub {
            warn "Log reading error!";
            $log_tail->destroy;
        },
        on_eof  => sub {
            warn "The end of log";
            $log_tail->destroy;
        },
        on_read  => sub {
            my $chunk = $_[0]->rbuf;
            $_[0]->rbuf = "";
            print $chunk;
        },
    );
    return $log_tail;
}

sub create_watcher {
    my ($inotify, $dir) = @_;
    say "Watching $dir";
    return $inotify->watch(
        $dir, IN_CLOSE_WRITE | IN_MOVED_TO | IN_DELETE, sub {
            my $e = shift;
            my $name = $e->fullname;
            my @mask_names;
            while (my($mask, $name) = each %::MASKS) {
                if ($e->mask & $mask) {
                    push @mask_names, $name;
                }
            }
            if ($name =~ /\.pm$/) {
                say "[@{[dt()]}] $name: @{[join ',', @mask_names]}";
                $reload_timer = AnyEvent->timer(after => 0.1, cb => \&reload_or_start_apache);
            }
        }
    );
}

sub reload_or_start_apache {
    $reload_timer = undef;
    my $pid_file = "$FindBin::Bin/../apache/run/ppc.yandex.ru.pid";
    my $pid = -f $pid_file && read_file($pid_file);
    if ($pid and kill(SIGUSR1, $pid) > 0) {
        say "[@{[dt()]}] Sent reload signal to apache";
    } else {
        # Похоже что апач мёртв
        say "[@{[dt()]}] The Apache is dead! long live the apache!";
        system("$FindBin::Bin/../apache/init.sh", "start");
    }
}

sub dt {
    strftime('%Y-%m-%d %H:%M:%S', localtime);
}

main();

