#!/usr/bin/perl

=head1 mysql_direct.pl

    connect to direct db via mysql client

    use:
        mysql_direct.pl db_name [-- --another-mysql-args]
        mysql_direct.pl --no-safe-updates --no-separate-history db_name [-- --another-mysql-args]
        mysql_direct.pl --dump --print-only db_name table_names...
        mysql_direct.pl production:db_name [-- --another-mysql-args]
        cat 'select * from users' | bin/mysql_direct.pl ppc -t
        bin/mysql_direct.pl ppc -- -t -e 'select * from campaigns limit 10'

    add completion for zsh:
        compctl -k "(--no-safe-updates --no-separate-history $(perl -Iprotected -MSettingsALL ./bin/db_names.pl))" mysql_direct.pl

    add completion for bash:
        complete -W "--no-safe-updates --no-separate-history $(perl -Iprotected -MSettingsALL ./bin/db_names.pl)" mysql_direct.pl

    $Id$

=cut

use warnings;
use strict;

use my_inc '..';

use Getopt::Long;

use Yandex::Shell;

use SettingsALL;

use Yandex::DBTools;

#-----------------------------------------------------------
sub main
{
    my $SEPARATE_HISTORY = 1;
    my $SAFE_UPDATES = 1;
    my $DUMP = 0;
    my $PRINT_ONLY = 0;
    GetOptions(
            "separate-history!" => \$SEPARATE_HISTORY,
            "safe-updates!" => \$SAFE_UPDATES,
            "dump" => \$DUMP,
            "print-only" => \$PRINT_ONLY,
            "help" => sub {system("pod2text-utf8", $0); exit 0;}
            ) || exit 0;

    my $db_name = shift @ARGV;

    die "not found db name\nuse: $0 db_name\n" unless defined $db_name;

    my $cfg = eval {get_db_config($db_name)};

    if ($@) {
        # для базы ppc умолчательный шард - 1
        if ($db_name =~ s/(^|:)ppc(:\D|$)/${1}ppc:1$2/) {
            $cfg = get_db_config($db_name);
        } else {
            die $@;
        }
    }

    # default args
    my $host = ref $cfg->{host} ? $cfg->{host}->[0] : $cfg->{host};
    $ENV{MYSQL_PWD} = $cfg->{pass};
    my @mysql_args = (($DUMP ? "-f" : "--prompt=$db_name> ")
                    , "--default-character-set=utf8"
                    , ($SAFE_UPDATES && !$DUMP ? ("--safe-updates") : ())
                    , "--user=$cfg->{user}"
                    , ($PRINT_ONLY ? "--password=$cfg->{pass}" : ())
                    , "--host=$host", "--port=$cfg->{port}"
                    , $cfg->{db}
                     );
    

    # set term title
    if (!$DUMP && -t STDOUT) {
        my $db_name_term_title = $db_name =~ /:/ ? $db_name : "$Settings::CONFIGURATION:$db_name";
        print "\033]0;mysql: $db_name_term_title\007";
    }

    # add user args
    push @mysql_args, @ARGV;

    if ($SEPARATE_HISTORY && !$DUMP) {
        (my $hist_name = $db_name) =~ s/^(prod|dev|test).*?://;
        $hist_name =~ s/:.*//g;
        $ENV{MYSQL_HISTFILE} = "/home/".[getpwuid($<)]->[0]."/.mysql_history.$hist_name";
    }

    my $cmd = $DUMP ? 'mysqldump' : 'mysql';
    
    if ($PRINT_ONLY) {
        print join(" ", $cmd, map {yash_quote($_)} @mysql_args), "\n";
    } else {
        exec $cmd, @mysql_args;
    }
}

#-----------------------------------------------------------
main();
