#!/usr/bin/perl

# $Id$

=head1 NAME

    svn_release.pl - Создание релизов в svn, мердж изменений

=head1 SYNOPSYS

    svn_release.pl PROJECT_SVN_URL RELEASE_REV MERGE_REV1,-MERGE_REV2,MERGE_REV3-MERGE_REV4

    svn mkdir https://svn.yandex.ru/direct/releases
    
    svn_release.pl https://svn.yandex.ru/direct 32768

    svn_release.pl https://svn.yandex.ru/direct 32768 65536,65539-65540

=head1 DESCRIPTION
    
    Скрипт позволяет создавать в SVN, в дирректории PROJECT_SVN_URL/releases создавать бранчи, соотвествующие тестируемым или выкладываемым
    релизам, а так же накладывать на релизы патчи из транка.

    1-ый параметр - SVN урл репозитория проекта. Предполагается, что у проекта стандартный конфиг - транк называется транком.

    2-ой параметр - базовая ревизия для создания релиза. Если бранча PROJECT_SVN_URL/releases/release-RELEASE_REV не существует - он создаётся.

    3-ий параметр - списки коммитов и диапазонов коммитов для накладывания на релиз патчей.

=cut

use warnings;
use strict;

use SVN::Client;
use Data::Dumper;
use File::Path qw/rmtree/;
use File::Temp;
use Date::Parse;
use Time::Local;
use POSIX qw/strftime/;
use Encode;

use Yandex::Interactive qw/prompt prompt_yn/;
use Yandex::Shell;

use utf8;
use open ':std' => ':utf8';

# разбираемся с аргументами командной строки
if (@ARGV < 2 || @ARGV > 3
    || $ARGV[0] !~ /^(https?|svn\+ssh|file):\/\//
    || $ARGV[1] !~ /^\d+$/
    || (@ARGV>2 && $ARGV[2] !~ /^(-?\d+|\d+-\d+)?(,-?\d+|\d+-\d+)*$/)
) {
    die "Usage: $0 SVN_URL RELEASE_REV MERGE_REV1,-MERGE_REV2,MERGE_REV3-MERGE_REV4";
}
my ($REPOSITORY_URL, $REV, $patches) = @ARGV;
my $WC_PATH = File::Temp::tempdir(undef, CLEANUP => 1);

my $svn = SVN::Client->new();

# проверяем существование бранча, если бранча нет - создаём
my $RELEASE_URL = "$REPOSITORY_URL/releases/release-$REV";
my $BASE_URL = "$REPOSITORY_URL/trunk";
if (!exists $svn->ls("$REPOSITORY_URL/releases", 'HEAD', 0)->{"release-$REV"}) {
    $svn->log_msg(sub {svn_ask("Create branch", "RELEASE: Created release-$REV", undef, @_);});
    $svn->copy($BASE_URL, $REV, $RELEASE_URL);
    print "Branch created: $RELEASE_URL\n";
}

# если патчи накладывать не нужно - выходим
if (!$patches) {
    exit;
}

# доводим working copy до нужной кондиции 
#my $want_checkout = 0;
#if (!-d $WC_PATH) {
#    $want_checkout = 1;
#} else {
#    $svn->status($WC_PATH, 'HEAD', sub {$want_checkout=1 if $_[1]->text_status() != $SVN::Wc::Status::external;}, 1, 0, 0, 0);
#    if ($want_checkout) {
#        exit if !prompt_yn("Delete dirty WC ($WC_PATH)? ");
#        rmtree($WC_PATH);
#    }
#}
#if ($want_checkout) {
    print "Start checkout to empty $WC_PATH...\n";
    $svn->checkout($RELEASE_URL, $WC_PATH, 'HEAD', 1);
#} else {
#    print "Start switch $WC_PATH...\n";
#    $svn->switch($WC_PATH, $RELEASE_URL, 'HEAD', 1);
#}

# мерджим изменения, собираем commit-message
my @messages;
for my $patch (split ',', $patches) {
    my ($r1, $r2);
    if ($patch =~ /^\d+$/) {
        ($r1, $r2) = ($patch-1, $patch);
    } elsif ($patch =~ /^-(\d+)$/) {
        ($r1, $r2) = ($1, $1-1);
    } elsif ($patch =~ /^(\d+)-(\d+)$/) {
        ($r1, $r2) = $1 <= $2 ? ($1-1, $2) : ($1, $2-1);
    } else {
        die "Incorrect patch definition: $patch";
    }
    my @log_rev = $r1 <= $r2 ? ($r1+1, $r2) : ($r1, $r2+1);
    $svn->log($BASE_URL, @log_rev, 0, 0,
              sub {
                  my (undef, $rev, $author, $date, $msg) = map {Encode::decode_utf8($_)} @_;
                  push @messages, "r$rev: ".format_date($date).", $author".($r1 > $r2 ? ' - ROLLBACK' : '')."\n$msg";
              }
        );
    print "Start merge $patch\n";
    $svn->merge($BASE_URL, $r1, $BASE_URL, $r2, $WC_PATH, 1, 0, 0, 0);
}

# проверяем, всё ли удачно смерджилось
my $want_commit = 0;
my @conflicts;
print "Changes:\n";
$svn->status($WC_PATH, 'HEAD',
             sub {
                 my $st = $_[1]->text_status();
                 print "  ".format_status($st)."\t$_[0]\n";
                 push @conflicts, $_[0] if $_[1]->text_status() == $SVN::Wc::Status::conflicted;
                 $want_commit = 1 if $_[1]->text_status() != $SVN::Wc::Status::external;
             }, 
             1, 0, 0, 0);

# если нужно - коммитим изменения
if ($want_commit) {
    my $commit_message = join "\n\n", "RELEASE: Merged $patches", @messages;
    $commit_message =~ s/\s+$//;
    $commit_message =~ s/(\s*\n\s*){3,}/\n\n/g;
    if (@conflicts) {
        print "Conflict occured during merge, resolve and commit manually.\n";
        if (prompt_yn("Keep directory $WC_PATH? ")) {
            $File::Temp::KEEP_ALL = 1;
            print STDERR "\nCommit message:\n";
            print STDERR $commit_message, "\n\n";
            print STDERR "Please, don't forget to remove directory after commit.\n";
        }
        exit(1);
    }
    if ($want_commit) {
        $svn->log_msg(
            sub {
                svn_ask("Commit changes?", $commit_message, yash_qx("svn", "diff", $WC_PATH), @_);
            }
            );
        my $commit = $svn->commit($WC_PATH, 0);
        print "Changes commited (revision ".$commit->revision().").\n";
    }
}

print "Release url: $RELEASE_URL\n";

# формат времени из gmt в human-readable
sub format_date {
    return strftime "%Y-%m-%d %H:%M", localtime( timegm( strptime $_[0]));
}

# текстовые обозначения статусов
sub format_status {
    my %STATES = (
        $SVN::Wc::Status::none => '?',
        $SVN::Wc::Status::unversioned => '?',
        $SVN::Wc::Status::normal => '',
        $SVN::Wc::Status::added => 'A',
        $SVN::Wc::Status::missing => '_',
        $SVN::Wc::Status::deleted => 'D',
        $SVN::Wc::Status::replaced => 'R',
        $SVN::Wc::Status::modified => 'M',
        $SVN::Wc::Status::merged => 'M',
        $SVN::Wc::Status::conflicted => 'C',
        $SVN::Wc::Status::ignored => 'I',
        $SVN::Wc::Status::obstructed => '?',
        $SVN::Wc::Status::external => 'X',
        $SVN::Wc::Status::incomplete => '?',
        );
    return exists $STATES{$_[0]} ? $STATES{$_[0]} : "?($_[0])";
}

# показываем пользователю умолчательное сообщение для коммита,
# предлагаем отредактировать
sub svn_ask {
    my ($question, $msg, $diff, $ret_ref) = @_;
    while(1) {
        print "Log message:\n";
        print "  $_\n" for split /\n/, $msg;
        my @opts = defined $diff ? ('y', 'e', 'd', 'q') : ('y', 'e', 'q');
        my %names = ('y' => 'yes', 'e' => 'edit message', 'd' => 'show diff', 'q' => 'quit');
        my $query = "$question (".join(" | ", map {"$_=$names{$_}"} @opts).")?";
        my $res = lc prompt("$query ", {"$query (Please enter ".join('|', @opts)."): " => \@opts});
        if ($res eq 'q') {
            exit 1;
        } elsif ($res eq 'd') {
            Yandex::Interactive::view($diff);
        } elsif ($res eq 'e') {
            $msg = Yandex::Interactive::edit($msg);
        } elsif ($res eq 'y') {
            last;
        } else {
            print "Incorrect answer: '$res'\n";
        }
    }
    $$ret_ref = $msg;
}

