(function($, Lego) {

    // in the case these components should be reused in any other place (what is unlikely)
    // they should be moved to `direct` namespace not to pollute global scope
    include("./PhraseRow.js");

    Lego.block['b-advanced-forecast__result-table'] = function(params) {

        var $this = $(this),
            $doc = $(document),
            model = params.model,

            // bulk actions controls
            toggleAllPhrasesCheckbox = $this.find(".js-toggle-all-phrases"),
            removeAllPhrasesButton = $this.find(".js-remove-all-phrases"),

            // separate bodies for phrases
            // and for bulk position editing operations as well
            table = $this.find('.js-result-table'),
            tableBody = table.find('.js-phrases-table-body'),
            bulkPositionEditingBody = table.find('.js-bulk-position-editing-table-body'),

            // summary info nodes
            summarySection = table.find('.js-summary-section').toggle(false),
            summaryRequestsCell = $this.find('.js-summary-requests'),
            summaryBudgetCell = $this.find('.js-summary-budget'),
            summaryShowsCell = $this.find('.js-summary-shows'),
            summaryClicksCell = $this.find('.js-summary-clicks'),
            summaryPeriodCell = $this.find('.js-summary-period'),
            convertedBudgetCell = $this.find('.js-converted-budget'),

            bulkPositionEditingList = bulkPositionEditingBody.find('.js-bulk-position-editing-list'),
            bulkPositionEditingLinks = (function(){
                var jsLinks = {};

                $.each(direct.forecast.positions, function(key, posName){
                    jsLinks[posName] = bulkPositionEditingList.find('.js-' + posName + '-pos');
                });

                return jsLinks;
            })(),

            // semi-transparent layer which covers the table when forecast calculation happens
            loadingLayer = $this.find('.js-loading-layer'),

            // bottom toolbar nodes
            bottomToolbar = table.find('.js-bottom-toolbar-section').toggle(false),
            bottomNoticesRow = bottomToolbar.find('.js-bottom-notices-row'),
            commonMinusWordsNotice = bottomToolbar.find('.js-common-minus-words-notice'),
            ungluedPhrasesNotice = bottomToolbar.find('.js-unglued-phrases-notice'),
            emptyForecastNotice = bottomToolbar.find('.js-empty-forecast-notice'),
            minusWordsPopupLink = bottomToolbar.find('.js-minus-words-popup-link'),
            showEmptyForecastNotice = false,

            // List of Phrases textarea nodes
            toggleTextareaLink = bottomToolbar.find(".js-toggle-textarea-link"),
            phrasesListTextareaRow = bottomToolbar.find(".js-phrases-textarea-row"),
            phrasesListTextarea = bottomToolbar.find(".js-toolbar-textarea"),
            openArrow = bottomToolbar.find(".open-arrow").text('►'),
            closeArrow = bottomToolbar.find(".close-arrow").text('▼'),

            exportToXLSLink = bottomToolbar.find(".js-export-to-xls"),

            nothingSelectedSection = table.find('.js-nothing-selected-section').toggle(false),
            resultTableJSLinks = table.find(".b-advanced-forecast__result-table__js-link"),

            minusWordsPopup = $('#' + params.minusWordsPopupID),

            templateRow = getTemplateString('.js-phrase-row-template'),

            firstChange = true, // флаг, чтобы фиксировать первичное построение таблицы ( чтобы при первом построении и только при нём, произвести автосортировку )

            // hash for storing instances of PhraseRow
            phraseRows = {},

            // hash used for building window.phrases array
            globalPhrases = {},


            // whether viewport's `scrollTop` property should be animated or not when forecast calculation starts
            shouldScrollToBlock = true;

            window.phrases = window.phrases || [];


        // initialization of block
        bindEvents();
        onForecastPeriodChange();
        onForecastParamsChange();

        /*------------------------------------------------------------------------------------------------------------*/

        function bindEvents() {

            // lego implementation of pseudo links haven't been used in this table
            // since it doesn't allow to render multiline ones
            // Hence, they were implemented using <A> tags and we should prevent default behaviour
            resultTableJSLinks.click(function(e){ e.preventDefault(); });


            initToolbarTextarea();

            // click on this js link should toggle block with Export to XLS params
            exportToXLSLink.click(function(){
                $doc.trigger('export2xls.toggle');
            });

            model
            // when summary info is updated
                .bind('change.budget', onSummaryBudgetChange)
                .bind('change.shows', onSummaryShowsChange)
                .bind('change.clicks', onSummaryClicksChange)
                .bind('change.requests', onSummaryRequestsChange)

                .bind('change.forecast_period', onForecastPeriodChange)

                .bind('change.unglued_keys', onUngluedKeysChange)
                .bind('change.popup_minus_words', onPopupMinusWordsChange)

            // when keywords are added or removed
            // these events are triggered after getting AJAX response containing raw data
            // essential for forecast calculating
                .bind('change.current_phrases', onPhrasesChange)

            // when slider is moved or forecast restrictions inputs are modified
                .bind('change.phrases', onPhrasesDetailsChange)

            // when forecast options are modified
                .bind('change.forecast-params', onForecastParamsChange);


            minusWordsPopupLink.click(onMinusWordsPopupLinkClick);

            toggleAllPhrasesCheckbox.bind('change', onToggleAllPhrasesCheckboxChange);
            removeAllPhrasesButton.bind('click', function(e) { e.preventDefault(); removeAllPhrases(); });

            // when forecast calculation starts -- triggered by b-advanced-forecast.js
            $doc
                .bind('forecast-data.calculating_started', function(){
                    closeAllPopupWindows();
                    showLoadingLayer();
                    toggleBulkPositionActionsBody();
                    if (shouldScrollToBlock) {
                        common.ui.scrollToBlock($this);
                    }
                })
            // when forecast calculation is stopped for any reason (whether a success or a failure)
                .bind('forecast-data.calculation_error forecast-data.calculating_finished', onForecastCalculationCompletion)
            // triggered by wordstat.html when some phrases are added in a corresponding popup window
                .bind('result-table.add-phrases', function(e, phrases){ addPhrases(phrases); })
            // when any phrase gets unchecked
                .bind("unchecked.phrase", onUncheckedPhrase)
            // when certain phrase checkbox is toggled
                .bind("toggled.phrase", function(e, enabled, key) { togglePhrase(enabled, key); } )
            // when certain phrase checkbox is clicked -- should be processed a little bit later than toggled.phrase
                .bind("clicked-checkbox.phrase", function(e, enabled, key){ onPhraseCheckboxClicked(enabled, key); })
            // when new position is set for a phrase in Budget By Positions mode
                .bind("new-position-event", function(e, position, key) { onChangePosition(position, key); } )
            // when phrase is changed -- see `PhraseRow.changePhrase` method
                .bind("changed.phrase", function(e, key) { onPhraseChange(key); } )
            // when phrase is deleted -- see `PhraseRow.removePhrase` method
                .bind("removed.phrase", function(e, key){ onPhraseRemove(key); })
            // when phrase is specified -- triggered from wordstat_minus.html popup window
                .bind("specified.phrase", function(e, key, newPhrase) {
                    if (phraseRows.hasOwnProperty(key)) {
                        phraseRows[key].changePhrase(newPhrase);
                    }
                })
            // happens when forecast calculation mode is changed
                .bind('forecast-type.budget-by-positions forecast-type.distributed-budget', function(){
                    toggleUIElementsForBudgetByPositions();
                });

            // initialization of the bulk position editing operations
            $.each(bulkPositionEditingLinks, function(positionName, jsLinks){
                jsLinks.bind('click',function(){
                    setPositionForAllPhrases(positionName);
                })
            })
        }

        /*------------------------------------------------------------------------------------------------------------*/

        function onForecastParamsChange(e) {
            if (!arguments.length) {
                $doc.trigger('calculate.forecast');
            }
        }

        function onForecastCalculationCompletion() {
            hideLoadingLayer();
            shouldScrollToBlock = true;
        }

        function setPositionForAllPhrases(positionName) {
            if (model.isDistributedBudgetMode()) return;
            $.each(phraseRows, function(){ this.setPositionForBudgetByPositions(positionName) });
        }

        function toggleBulkPositionActionsBody() {
            bulkPositionEditingBody.toggleClass("g-hidden", !!model.isDistributedBudgetMode());
        }

        function onPopupMinusWordsChange() {
            commonMinusWordsNotice.toggleClass("g-hidden", !model.hasPopupMinusWords());
            toggleBottomNoticeRow();
        }

        function onUngluedKeysChange() {
            ungluedPhrasesNotice.toggleClass("g-hidden", !model.hasUngluedPhrases());
            toggleBottomNoticeRow();
        }

        function toggleBottomNoticeRow() {
            var toBeShown =  showEmptyForecastNotice || model.hasUngluedPhrases() || model.hasPopupMinusWords();
            bottomNoticesRow.toggleClass("g-hidden", !toBeShown);
        }

        function closeAllPopupWindows() {
            $doc.trigger('b-window_close-all');
        }

        function onMinusWordsPopupLinkClick() {
            if (minusWordsPopup.data('api').visible()) {
                minusWordsPopup.data('api').close();
            } else {
                $doc.trigger('b-window_close-all');
                minusWordsPopup.removeClass('g-hidden').data('api').open();
            }
        }

        function initToolbarTextarea() {
            toggleTextareaLink.click((function(){
                var isTextareaVisible = false;
                return function() {
                    isTextareaVisible = !isTextareaVisible;
                    phrasesListTextareaRow.add(closeArrow).toggleClass('g-hidden', !isTextareaVisible);
                    openArrow.toggleClass('g-hidden', isTextareaVisible);
                }
            })());

            phrasesListTextarea.click(function(){ phrasesListTextarea.select(); });

            // hide 'select all phrases' checkbox when there's no phrases
            // or vice versa in the case  some phrases exist
            // update the list of active ('checked') phrases
            // in the bottom toolbar textarea
            model.bind('change.window_phrases', function(){
                var phrases = [];

                toggleAllPhrasesCheckbox.toggle(!!(model.window_phrases || []).length);

                $.each(model.window_phrases, function(index, item){
                    if (item.isActive) {
                        phrases.push(item.phrase);
                    }
                });
                phrasesListTextarea.val(phrases.sort().join(",\n"));
            });
        }

        // template for the certain type of phrase row is stored in hidden `<tr>` tag
        // all template rows are childs of `tbody.js-row-templates`
        function getTemplateString(templateNodeSelector) {
            var div = document.createElement('div');
            div.appendChild($this.find(templateNodeSelector).eq(0).clone()[0]);
            return div.innerHTML;
        }

        function addPhrases(phrases) {
            $.each(phrases, function(i, phrase){
                window.phrases.push({
                    enable: true,
                    phrase: $.trim(phrase)
                })
            });

            model.update({ "window_phrases": window.phrases });
            // getting new raw data for calculating forecast via AJAX
            $doc.trigger('recalculate.phrases');
        }

        function showLoadingLayer() {
            $this.toggleClass('b-advanced-forecast__nonempty-table', !!window.phrases.length);
            nothingSelectedSection.hide();
            loadingLayer.show();
        }

        function hideLoadingLayer() {
            bottomToolbar.add(nothingSelectedSection).toggle(!window.phrases.length);
            loadingLayer.hide();
        }

        function onSummaryClicksChange() {
            summaryClicksCell.text(formatValue(model.clicks));
        }

        function onSummaryRequestsChange() {
            summaryRequestsCell.text(formatValue(model.requests));
        }

        function onForecastPeriodChange() {
            // DIRECT-58932 костыль про неделю
            summaryPeriodCell.text(model.forecastPeriodType == 'week' ?
                iget('неделю') :
                model.getMessage('forecastPeriod'));
        }

        function onSummaryShowsChange() {
            summaryShowsCell.text(formatValue(model.shows));
            onSummaryBudgetChange();
        }

        function onSummaryBudgetChange() {
            summaryBudgetCell.text(common.number.format(model.sum / 1e6));

            var convertedBudget = '' + model.sum * model.currencyRate / 1e6;

            convertedBudgetCell.text(' (' + formatValue(+convertedBudget, 2) + ' ' + model.currencyName + ')');

            showEmptyForecastNotice = model.sum == 0 && !isTableEmpty();
            emptyForecastNotice.toggleClass('g-hidden', !showEmptyForecastNotice);
            toggleBottomNoticeRow();
        }

        function formatValue(value, precision) {
            return common.number.format(value, {precision: precision || 0});
        }

        function removeAllPhrases() {
            var confirmMessage = '', key;
            if (window.phrases.length) {
                confirmMessage = iget("Вы желаете удалить все фразы?");
            } else {
                return;
            }

            if (!confirm(confirmMessage)) {
                return;
            }

            for (key in phraseRows) {
                if (!phraseRows.hasOwnProperty(key)) continue;
                phraseRows[key].removePhrase(true);
            }

            globalPhrases = {};
            window.phrases = [];

            toggleAllPhrasesCheckbox[0].checked = false;

            model.setEmptyValues();
        }

        function onPhraseRemove(key) {
            if (globalPhrases[key]) delete globalPhrases[key];
            updateWindowPhrases(true);

            $doc.trigger('recalculate.phrases');
            $(window).trigger('other-phrases-change');
        }

        var prevSiblingOfChangedPhrase = [];
        function onPhraseChange(key) {
            if (!phraseRows[key]) return;

            // DIRECT-11419 we should not scroll to the beginning of the table when the phrase was modified
            shouldScrollToBlock = false;
            prevSiblingOfChangedPhrase = phraseRows[key].getJQueryNode().prev();
            // in the case changed phrase is the only one
            if (!prevSiblingOfChangedPhrase.length) {
                prevSiblingOfChangedPhrase = tableBody;
            }

            globalPhrases[key] = globalPhrases[key] || {};
            globalPhrases[key].phrase = phraseRows[key].phrase;
            updateWindowPhrases();

            $doc.trigger('recalculate.phrases');
        }

        function getObjectsHashByKey(key) {
            return model['phrases'] || {};
        }

        function onChangePosition(position, key) {
            var objectsHash = getObjectsHashByKey(key);
            objectsHash[key] = objectsHash[key] || {};
            objectsHash[key].position = position;
        }

        // we should wait for a little while since `toggled.phrase` should trigger at first
        var phraseCheckboxClickTimer = null;
        function onPhraseCheckboxClicked(enabled, key) {
            clearTimeout(phraseCheckboxClickTimer);
            phraseCheckboxClickTimer = setTimeout(function(){
                $doc.trigger('calculate.forecast', getForecastRestrictions("sum"));
            }, 5);
        }

        var prevRestrictionsData = null;
        function getForecastRestrictions(fieldName) {
            var unselectedAll = allUnselected();
            var data = null;
            var modelData = { key: fieldName, value: model[fieldName] };
            var notNullValue = !!(parseFloat(modelData.value, 10) > 0);

            // saving current restrictions when the last item is unselected in order to calculate forecast with the same restrictions the next time
            if (unselectedAll && notNullValue) {
                prevRestrictionsData = modelData;
            }

            data = notNullValue ? modelData : prevRestrictionsData;

            return unselectedAll ? null : data || prevRestrictionsData;
        }


        // makes visible ui elements which correspond to the current forecast mode
        function toggleUIElementsForBudgetByPositions() {
            var toBeShown = !model.isDistributedBudgetMode();
            toggleBulkPositionActionsBody();
            $.each(phraseRows, function(){ if (this.isVisible) this.toggleUIElementsForBudgetByPositions(toBeShown); });
        }

        var updateAllPhrasesCheckbox = function performUpdatingAllPhrasesCheckbox() {
            var keys = common.utils.keys;

            // default value depends on the number of phrases
            var selectedAll = keys(phraseRows) ? true : false;
            selectedAll = updateSelectedAllFlag(selectedAll, phraseRows);
            toggleAllPhrasesCheckbox[0].checked = selectedAll;

            function updateSelectedAllFlag(flagValue, rowObjectsHash) {
                var value = flagValue;
                for (var key in rowObjectsHash) {
                    if(rowObjectsHash.hasOwnProperty(key)) {
                        var rowObject = rowObjectsHash[key];
                        if (!rowObject.isChecked && rowObject.isVisible) {
                            value = false;
                        }
                    }
                }
                return value;
            }

        }

        var performPostPhraseTogglingSubroutine = common.func.debounce(postPhraseTogglingSubroutine, 10);
        function postPhraseTogglingSubroutine(enabled, key) {
            model.update({
                "window_phrases": window.phrases
            });

            updateAllPhrasesCheckbox();

            if (allUnselected()) {
                updateRecBudget();
            }
        }

        function togglePhrase(enabled, key) {
            if (!globalPhrases[key]) return;

            updateWindowArrays(enabled, key, window.phrases, "window_phrases", globalPhrases, phraseRows);

            performPostPhraseTogglingSubroutine(enabled, key);

            $(window).trigger('other-phrases-change');

        }

        function updateWindowArrays(enabled, key, array, fieldName, localHash, rowObjectsHash) {
            if (localHash[key]) {
                localHash[key].isActive = enabled;
                if (rowObjectsHash[key]) {
                    var index = rowObjectsHash[key].getWindowPhrasesIndex();
                    if (index > -1) {
                        array[index].isActive = enabled;
                        array[index].enable = enabled;
                        model[fieldName] = array;
                    }
                }
            }
        }

        function allUnselected() {
            return (common.utils.keys(model.getDisabledPhrases()) == (window.phrases.length));
        }

        function onUncheckedPhrase() {
            toggleAllPhrasesCheckbox[0].checked = false;
        }

        function onToggleAllPhrasesCheckboxChange(e) {
            var toBeChecked = toggleAllPhrasesCheckbox[0].checked, key;

            for (key in phraseRows) {
                if(phraseRows.hasOwnProperty(key)) {
                    phraseRows[key].toggleCheck(toBeChecked, undefined, true);
                }
            }

            $doc.trigger('calculate.forecast', getForecastRestrictions("sum"));

            if (!toBeChecked) {
                updateRecBudget();
            }

            table.find('tr:visible').hide().show(); // инициируем перерисовку - правка для DIRECT-12777
        }

        function updateRecBudget() {
            model.update({
                "restrictions": {
                    type: "sum",
                    value: model.rec_budget || 0
                }
            })
        }

        function onItemsDetailsChange(itemsType) {
            var modelItems = model[itemsType],
                getRowObject = getPhraseRowObject,
                callback = updateWindowPhrases;

            for (var key in modelItems) {
                if (modelItems.hasOwnProperty(key)) {
                    getRowObject(key).updateData(modelItems[key] || {});
                }
            }
            callback();
        }


        function onPhrasesDetailsChange(e) {
            onItemsDetailsChange('phrases');
        }


        // updates global array `window.phrases` on basis of `globalPhrases` hash
        var windowPhrasesTimer = null;
        function updateWindowPhrases(now) {
            clearTimeout(windowPhrasesTimer);
            now ?
                performUpdateWindowPhrases() :
                windowPhrasesTimer = setTimeout(function(){ performUpdateWindowPhrases() }, 100);
        }

        function performUpdateWindowPhrases() {
            var newPhrases = [],
                ungluedKeys = [];

            for (var key in globalPhrases) {
                if (!globalPhrases.hasOwnProperty(key)) continue;

                newPhrases.push(globalPhrases[key]);

                var phraseRow = phraseRows[key];

                phraseRow.toggleVisibility(false);

                // by default all phrases are not selected regardless of the calculated details for the given phrase
                // later on, if user toggled checkbox manually `isActive` property is updated
                phraseRow.toggleCheck(globalPhrases[key] ? globalPhrases[key].isActive : false);

                // in the case we don't have any details for the given phrase we set empty values
                if (!model.phrases || !model.phrases.hasOwnProperty(key)) phraseRow.setEmpty();

                phraseRow.toggleVisibility(true);
            }
            window.phrases = newPhrases;

            // remove from unglued_keys unused items
            if (model.unglued_keys) {
                ungluedKeys = $.map(model.unglued_keys, function(item){
                    if (!(item in globalPhrases)) return null;
                    return item;
                });
            }

            model.update({
                window_phrases: window.phrases,
                unglued_keys: ungluedKeys
            });

            var noPhrases = isTableEmpty();

            summarySection.add(bottomToolbar).toggle(!noPhrases);
            nothingSelectedSection.toggle(noPhrases);

            updateAllPhrasesCheckbox();
            toggleUIElementsForBudgetByPositions();
            $(window).trigger('other-phrases-change');
        }

        function isTableEmpty() {
            return (!window.phrases.length);
        }


        function onPhrasesChange() {
            var phraseAppended = 0,
                key,
                oldGlobalPhrases = globalPhrases;

            globalPhrases = {};

            var docFragment = document.createDocumentFragment();

            for (key in model.key2phrase) {
                if (!model.key2phrase.hasOwnProperty(key)) continue;

                var phraseRow = getPhraseRowObject(key),
                    isLowCtr = (!!($.inArray(key, model.low_ctr_keys) > -1) && !phraseRow.isActive());

                globalPhrases[key] = oldGlobalPhrases[key] || {
                    key: key,
                    phrase: model.key2phrase[key],
                    enable: !isLowCtr,
                    isActive: !isLowCtr
                };

                if (!phraseRow.isAppended) {
                    if (prevSiblingOfChangedPhrase && prevSiblingOfChangedPhrase.length) {
                        phraseRow.getJQueryNode()[prevSiblingOfChangedPhrase == tableBody ? 'appendTo' : 'insertAfter'](prevSiblingOfChangedPhrase);
                    } else {
                        docFragment.appendChild(phraseRow.node);
                    }
                    phraseRow.isAppended = true;
                    phraseRow.setLowCtr(isLowCtr);
                    phraseAppended++;
                }

                phraseRow.toggleCheck(!isLowCtr || model.isDistributedBudgetMode());

                // phrase maybe modified on server -- for instance, some minus words might be added automatically
                // but the key(md5) of phrase remains unchanged
                if (phraseRow.phrase != model.key2phrase[key]) {
                    phraseRow.setPhrase(model.key2phrase[key]);
                }
            }

            // hiding the phrase rows which were removed by user
            for (key in phraseRows) {
                if (!phraseRows.hasOwnProperty(key)) continue;
                var toBeShown = globalPhrases.hasOwnProperty(key);
                phraseRows[key].toggle(toBeShown);
            }

            tableBody.append(docFragment.childNodes);

            prevSiblingOfChangedPhrase = [];
            !firstChange && phraseAppended && table.data('api').clearHeaders(); // если добавили фразу -- сбрасываем сортировку
            firstChange && !(firstChange = false) && sortTable('phrase');
            updateWindowPhrases(true); // параметр true для правки бага DIRECT-13039
        }

        function sortTable(name) {
            // b-sortable-table api methods should be retrieved by jquery data method only
            // since table is not instance of b-sortable-table yet at the moment of initialization of the given block
            table.data('api').sortBy(name);
        }


        function getPhraseRowObject(key) {
            return phraseRows[key] || (phraseRows[key] = new PhraseRow(key, model.key2phrase[key], templateRow, model));
        }

    }
})(jQuery, window.Lego);
