/** @requires BEM */
/** @requires BEM.INTERNAL */

(function(BEM, $, undefined) {

    //кэш с найденными группами модедей
    var modelsGroupsCache = {};

    /**
     * @namespace
     * @name BEM.MODEL
     */
    var MODEL = BEM.MODEL = $.inherit($.observable, {

        __constructor: function(modelName, data) {
            this.name = modelName;
            this._initSchemes();

            this._initValidator();

            this.changed = [];
            this.childModelsHash = {};
            this.fireChange = $.debounce(this._fireChange, this.changesTimeout, this);

            this.debounceTrigger = $.debounce(function(name, data) {
                this.trigger(name, data);
            }, this.changesTimeout, false, this);

            this._initFields(data || {});

            return this;
        },

        modelType: '2.0',

        isModel: true,

        getUniqId: function() {
            return this._uniqId;
        },


        /**
         * Возвращает значения модели, установленные при инициализации
         * Значения также считаются установленными при инициализации, если на модели
         * был выполнен метод fixData
         * @protected
         * @returns {Object}
         * todo бывшее getDefault
         */
        getModelStartValue: function() {
            var defaults = {};

            $.each(this.fields, $.proxy(function(name, field){
                defaults[name] = field.getFieldStartValue();
            }, this));

            return defaults;
        },

        /**
         *  Возвращает хэш значений прописаных в схеме модели как default
         *  @protected
         *  @returns {Object}
         */
        getModelDefaults: function() {
            var defaults = {};

            $.each(this.fields, $.proxy(function(name, field){
                var defaultValue = field.getDefault();

                defaultValue && (defaults[name] = field.getDefault());
            }, this));

            return defaults;
        },

        /**
         * Возвращает дефолтное значение для поля name модели
         * @param {String} name имя поля
         * @protected
         * @returns {String|Boolean|Number}
         */
        getDefault: function(name) {
            if (this.fields[name]) return this.fields[name].getDefault();
        },

        /**
         * Устанавливает данные из data в качестве дефолтных значений
         * @protected
         * @param {Object} data
         * @returns {BEM}
         */
        setModelDefaults: function(data) {
            var _this = this;

            $.each(data, function(name, value){
                if (_this.fields[name]) _this.fields[name].setDefault(value);
            });

            return this;
        },

        /**
        * Проверяет, совпадают ли текущие значения модели с умолчальными
        * @protected
        * @returns {Boolean}
        */
        isValuesDefault: function() {
            var defaults = this.getModelDefaults(),
                _this = this, res = true;

            $.each(defaults, function(name, value) {
                if (_this.fields[name] && !_this.fields[name].isEqual(value)) {
                    res = false;
                    return false
                }
            });

            return res;
        },

        /**
         * Возвращает общий текст ошибки для модели
         * @return {String}
         */
        getGlobalErrorText: function() {
            return this.validator.getErrorsMessagesForModel().join(', ');
        },



        /**
         * Возвращает значение поля, установленное при инициализации
         * Значения также считаются установленными при инициализации, если на модели
         * был выполнен метод fixData
         * @protected
         * @param {String} name имя поля
         * @returns {String|Number|Object}  - возвращает значение поля в зависимости от типа
         * todo Бывшее getStartValue
         */
        getFieldStartValue: function(name) {
            var field = this.fields[name];

            if (field) return field.getFieldStartValue();
        },


        /**
         * Устанавливает валидатор, необходимый для каждой модели с validateRules
         * @private
         * @returns {BEM}
         */
        _initValidator: function() {
            this.validator = BEM.create('i-model-validator', { model: this });
            return this;
        },

        /**
         * Записывает в модель данные, полученные из её схемы
         * @private
         * @returns {BEM}
         */
        _initSchemes: function() {
            var schemes = MODEL.models[this.name].getScheme(),
                _this = this;

            if (!schemes) {
                throw new Error(iget('Модель с именем %s не определена', this.name));
            }

            this.fieldsSchemes = schemes.fields;
            this.changesTimeout = schemes.changesTimeout || 500;

            $.map(['validateRules', 'childModels'], function(name) {
                _this[name] = schemes[name]
            });

            this._parseFieldTypes();

            return this;
        },


        needToValidateField: function(fieldId) {
            if (!this.validator) return false;

            return this.validator.needToValidateField(fieldId);
        },


        /**
         * Разделяем названия полей по массивам, в соответствии с типо поля input|toServer|fromServer
         * у одного поля может быть одновременно несколько типов и оно окажется в нескольких массивах
         * @private
         * @returns {BEM}
         */
        _parseFieldTypes: function() {
            var fieldTypes = this.fieldTypes = {
                'input': [], //deprecated для совместимости со старым кодом
                'toServer': [],
                'fromServer': [],
                'hasDom': [],
                'hasDomInput': [],
                'hasDomOutput': [],
                'all': []
                },
                _this = this;

            $.each(this.fieldsSchemes, function(name, field) {
                field = _this.getFieldScheme(name);
                $.each(fieldTypes, function(fieldType) {
                    if (field[fieldType]) {
                        if (fieldType == 'input') {
                            fieldTypes.hasDom.push(name);
                            fieldTypes.hasDomInput.push(name);
                            fieldTypes.hasDomOutput.push(name);
                        }

                        if (fieldType == 'hasDom') {
                            fieldTypes.hasDomInput.push(name);
                            fieldTypes.hasDomOutput.push(name);
                        }

                        fieldTypes[fieldType].push(name);
                    }
                });

                fieldTypes.all.push(name);

            });

            return this;
        },

        /**
         * Возвращает массив полей заданного типа (input|toServer|fromServer)
         * @protected
         * @param {String} mode название типа (input|toServer|fromServer) //todo - посмотреть как правильно оформлять
         * @returns {Array}
         */
        getFieldsNames: function(mode) {
            return (this.fieldTypes[mode] || this.fieldTypes.all);
        },

        /**
         * Создаёт хэш с полями в соответствии со схемой модели и заполняет эти поля начальными данными если они есть
         * @private
         * @param data
         * @returns {BEM}
         */
        _initFields: function(data) {
            var _this = this;

            this.fields = {};

            $.each(this.fieldsSchemes, $.proxy(function(name, props) {
                props = _this.getFieldScheme(name);
                props.name = name;
                this.fields[name] =  BEM.MODEL.FIELD.init(props, props.type);
            }, this));

            !$.isEmptyObject(data) && this.initData(data);

            return this;
        },


        /**
         * Записывает/возвращает путь к модели
         * @protected
         * @param {String} path путь к модели
         * @returns {BEM | String}
         */
        path: function(path) {
            if (typeof path == 'undefined') return this._path;
            this._path = path;

            return this;
        },

        /**
         * Заполняет модель начальными данными
         * Если у модели есть дочерние модели - они тоже заполняются
         * @protected
         * @param {Object} data хэш с данными для модели, соответствующий схеме модели
         * @param {Object} source блок инициировавший инициализацию данных
         * @param {Boolean} [withChildren] апдейтитиь также дочерние модели
         * @returns {BEM}
         */
        initData: function(data, source, withChildren) {
            var _this = this;
            data = data || {};

            $.each(data, function(name, value){
                if (!_this.hasField(name)) return true;

                _this.fields[name].initData(value, _this);
                _this.calcChildren(name, 1, source);
            });

            withChildren && this.childModels && $.each(this.childModels, function(modelName, info) {
                var childData = info.source ? data[info.source] : data;
                if (!childData) { return true; }
                _this.getChildModel(modelName).initData(childData, source, withChildren);
            });

            this.dataInited = true;
            this.trigger('init-data');

            return this;
        },

        /**
         * Вешаем обработчик на инициализацию начальных данных модели
         * коллбэк выполняется хотя бы один раз в любом случае, даже если модель, на которую подписываемся
         * уже проинициализированна к моменту подписки
         * @protected
         * @param {Function} callback функция-обработчик
         * @param {Object} ctx контекст выполнения обработчика
         * @returns {BEM}
         */
        onInitData: function(callback, ctx) {

            if (this.dataInited) callback.call(ctx);

            //модель может инициализироваться в несколько этапов, так что всё равно подписываемся на события
            return this.on('init-data', callback, ctx);
        },


        /**
         *  Устанавливаем значение поля модели, имеющее флаг protected (т.е. это поле невозможно переписать классическим set, update ect)
         * @param {String} name имя поля
         * @param {String|Array|Object|Number|Boolean} value значение поля
         * @param {BEM} source BEM-блок, установивший это поле
         * @returns {BEM}
         */
        setProtected: function(name, value, source) {
            this.set(name, value, source, true);

            return this;
        },

        /**
         * фиксируем текущее состояние модели как начальное
         * @protected
         * @returns {BEM}
         */
        fixData: function() {
            $.each(this.fields, function(name, field){
                field.fixData()
            });

            return this.trigger('fix-data');
        },


        /**
         * Возвращаем имя события, которое выстреливается при измененеии поля
         * @protected
         * @param {String} fieldId имя поля
         * @returns {String}
         * todo вынести в DOM-реализацию
         */
        getUpdateEvent: function(fieldId) {
            var fieldsScheme = this.getFieldScheme(fieldId);

            return fieldsScheme ? fieldsScheme.event : '' || 'change';
        },


        /**
         * Устанавливаем значение поля модели
         * @protected
         * @param {String} name имя поля
         * @param {String|Array|Object|Number|Boolean} value значение поля
         * @param {BEM} source BEM-блок, установивший это поле
         * @param {string} [eventName] событие, по которому вызвалось set
         * @returns {BEM}
         */
        set: function(name, value, source, eventName) {
            var field = this.fields[name],
                fieldsScheme = this.getFieldScheme(name),
                type = field ? field.getType() : '';

            if (!field || !fieldsScheme) return this;

            if (!field.isEqual(value)) {
                //чистим сообщение об ощибках на данном поле
                this.validator && this.validator.clearField(name);
                field.set(value, source, eventName);
                type != 'const' && this.changed.push(name);
                this.calcChildren(name, false, source);
                this.clearChildModelErrors(name);
                //если изменилось несколько полей модели подряд - вызовем только одно событие change
                type != 'const' && this.fireChange(source);
            }

            return this;
        },


        /**
         * Если изменились поля, влияющие на валидацию дочерних моделей - очистиьт сообщения об ошибках
         * @param name - имя поля
         * @return {BEM}
         */
        clearChildModelErrors: function(name) {
            var fieldsScheme = this.getFieldScheme(name),
                _this = this;

            if (fieldsScheme && fieldsScheme.childModels) {
                $.each(fieldsScheme.childModels, function(i, childName) {
                    var model = _this.getChildModel(childName);
                    model.clearModelErrors();
                })
            }

            return this;
        },


        clearModelErrors: function() {
            if (!this.validator) return this;

            return this.validator.clearModelErrors();
        },


        /**
         * Откатывает модель на состояние в котором она проинициализировалась
         * @protected
         * @param {BEM} source BEM-блок, совершивший действие
         * @param {Array} expulsions имена полей, которые не надо откатывать
         * @returns {BEM}
         */
        rollback: function(source, expulsions) {
            $.each(this.fields, $.proxy(function(name){
                if (expulsions && $.inArray(name, expulsions) > -1) { return true; }
                this.set(name, this.fields[name].getFieldStartValue(), source, 'rollback')
            }, this));
            return this.trigger('rollback', {source: source});
        },


        /**
         * Поле без ошибок приведено к прописанному в его свойствах формату
         * @protected
         * @param {String} name имя поля
         * @returns {Boolean}
         * todo Бывшее isFormated
         */
        isFieldFormated: function(name) {
            return this.fields[name].isFormated();
        },


        /**
         * Проверяет на пустоту все поля данного типа
         * @protected
         * @param {String} mode название типа toServer|fromServer|input
         * @param {Boolean} [doChildren] нужно ли проверять на пустоту дочерние модели
         * @param {Array} [excludes] поля, которые не надо проверять на пустоту
         * @returns {Boolean}
         * todo Бывшее isFieldsEmpty
         */

        isModelEmpty: function(mode, doChildren, excludes) {
            var empty = true, _this = this;

            $.each(this.getFieldsNames(mode), function(i, name){
                if (excludes && $.inArray(name, excludes) != -1) return;
                empty = empty && _this.isFieldEmpty(name);
            });

            if (doChildren && this.childModels)  {
                $.each(this.childModels, function(name) {
                    var model = BEM.blocks['i-models-manager'].get(_this.path(), name);
                    empty = empty && model.isModelEmpty(mode, doChildren, excludes);
                })
            }
            return empty;
        },

        /**
         * Проверяет поле на пустоту
         * @protected
         * @param {String} name имя поля
         * @returns {Boolean}
         * todo Бывшее isEmpty
         */
        isFieldEmpty: function(name) {
            return this.fields[name].checkEmpty(this.get(name));
        },


        /**
         * Апдейтит модель присланными данными. В отличии от update, которое изменяет только присланные в data поля
         * reset изменяет ВСЕ поля. Если поле пришло в data - берёт значение из data,
         * если не пришло - ставит значение по умолчанию
         * @protected
         * @param {Object} data новые значения модели
         * @param {BEM} source блок, вызвавший метод
         * @param {Boolean} doChildren нужно ли применять метод к дочерним моделям
         * @returns {BEM}
         *
         */
        reset: function(data, source, doChildren) {
            var _this = this;
            data = data || {};

            $.each(this.fields, function(name){
                _this.set(name, data[name], source);
            });

            doChildren && this.childModels && $.each(this.childModels, function(modelName, info) {
                var childData = info.source ? data[info.source] : data;
                if (!childData) { return true; }
                _this.getChildModel(modelName).reset(childData, source, doChildren);
            });
            return this.trigger('reset', {source: source});
        },



        /**
         * Очищает поля модели
         * @protected
         * @param {BEM} source блок, вызвавший метод
         * @param {Boolean} doChildren нужно ли применять метод к дочерним моделям
         * @param {Array} [excludes] имена полей, которые не надо чистить
         * @returns {BEM}
         *
         */
         clear: function(source, doChildren, excludes) {
             excludes = excludes || [];

             $.each(this.fields, $.proxy(function(name){
                 if ($.inArray(name, excludes) != -1) return true;
                 this.set(name, '', source);

             }, this));

             doChildren && this.doChildrenModels('clear', [source, doChildren]);

             return this.trigger('clear', {source: source});
         },

        /**
         *
         * Навешивает события на модель
         * @param {Object|String} e - название события
         * @param {Object} [data] - дополнительные данные
         * @param {Function} fn - коллбэк
         * @param {Object} [ctx] - контекст выполнения коллбэка
         * @param {Object} [doChildren] - отслеживать изменения на дочерних моделях
         * @return {BEM}
         */
        on: function(e, data, fn, ctx, doChildren) {
            if (typeof ctx != 'object') {
                doChildren = ctx;
            }
            this.__base.apply(this, arguments);

            doChildren && this.doChildrenModels('on', [e, data, fn, ctx, doChildren]);

            return this;
        },


        /**
         * Возвращает значение поля модели
         * @protected
         * @param {String} name имя поля модели
         * @returns {String|Array|Object|Number|Boolean}
         *
         */
        get: function(name) {
            return this.hasField(name) && this.fields[name].get();
        },


        /**
         * Возвращает значение поля модели в том виде, в котором оно было введено пользователем/прислано сервером
         * т.е. без округлений и прочих преобразований
         * @protected
         * @param {String} name имя поля модели
         * @returns {String|Array|Object|Number|Boolean}
         *
         */
        raw: function(name) {
            return this.hasField(name) && this.fields[name].raw();
        },

        /**
         * Возвращает значение поля модели для вывода на печать - т.е. с пробелами и прочей красивостью
         * @protected
         * @param {String} name имя поля модели
         * @returns {String}
         *
         */
        view: function(name) {
            return this.hasField(name) && this.fields[name].view();
        },

        /**
         * Выдаёт по имени поле имя соответствующего ему DOM-элемента
         * @protected
         * @param {String} name имя поля модели
         * @returns {String}
         * @deprecated
         *
         */
        getElemName: function(name) {
            return this.hasField(name) && this.fields[name].getElemName();
        },

        /**
         * Проверка на существование поля с заданным именем
         * @protected
         * @param {String} name имя поля модели
         * @returns {Boolean}
         *
         */
        hasField: function(name) {
            return !!this.fields[name];
        },

        /**
         * Возвращает список значений для заданного массива полей
         * @protected
         * @param {Array} fieldsArray имя поля модели
         * @returns {Object}
         *
         */
        mementoFields: function(fieldsArray) {
            var state = {}, value;

            $.each(fieldsArray, $.proxy(function(i, fieldName){
                value = this.fields[fieldName].get();
                state[fieldName] = value;
            }, this));

            return state;
        },

        /**
         * Возвращает текущее значение полей модели
         * @protected
         * @param {Boolean} clearUndefined удалить из результата пустые поля модели
         * @param {Boolean} doChildren получить данные из дочерних моделей
         * @param {String} mode название типа toServer|fromServer|input
         * @returns {Object}
         *
         */
        memento: function(clearUndefined, doChildren, mode) {
            var state = {}, value, _this = this;

            if (typeof clearUndefined == 'string') {
                mode = clearUndefined;
                clearUndefined = false;
            }

            if (typeof doChildren == 'string') {
                mode = doChildren;
                doChildren = false;
            }

            $.each(this.getFieldsNames(mode), $.proxy(function(i, name){
                var field = _this.fields[name];
                value = field.get();
                if (!clearUndefined || !field.checkEmpty(value)) {
                    state[name] = value;
                }
            }, this));

            if (doChildren && this.childModels)  {
                $.each(this.childModels, function(name, info) {
                    var model = BEM.blocks['i-models-manager'].get(_this.path(), name),
                        modelData = model.memento(clearUndefined, doChildren, mode);

                    if (info.source) {
                        state[info.source] = modelData;
                    } else {
                        state = $.extend(state, modelData);
                    }
                })
            }

            return state;
        },


        /**
         * менялись ли значения с момента инициализации
         * @protected
         * @param {String} mode название типа toServer|fromServer|input
         * @returns {Boolean}
         */
        isChanged: function(mode) {
            var changed = false;

            $.each(this.getFieldsNames(mode), $.proxy(function(i, name){
                changed = this.fields[name].isChanged() || changed;
                if (changed) return false;
            }, this));

            return changed;
        },

        /**
         * Менялось ли поле модели с момента инициализации
         * @protected
         * @param {String} fieldId название поля
         * @param {String|Number|Boolean} value значение
         * @returns {Boolean}
         *
         */
        isFieldChanged: function(fieldId, value) {
            if (this.fields[fieldId]) {
                return value == undefined ? this.fields[fieldId].isChanged() : !this.fields[fieldId].isEqual(value);
            }  else {
                var _this = this, changed = false;
                if (this.childModels)  {
                    $.each(this.childModels, function(name, info) {
                        var model = _this.getChildModel(name);
                        if (model.isFieldChanged(fieldId, value)) {
                            changed = true;
                            return false;
                        }
                    })
                }
                return changed;
            }
            return false;
        },

        /**
         * Равно ли зачение поля модели заданному значению
         * @protected
         * @param {String} name название поля
         * @param {String|Boolean|Number} value значение для сравнения
         * @returns {Boolean}
         *
         */
        isFieldEqual: function(name, value) {
            return !this.fields[name] || this.fields[name].isEqual(value);
        },

        /**
         * Возвращает текущее значение полей модели
         * @protected
         * @param {String} mode название типа toServer|fromServer|input
         * @returns {Object}
         * todo бывшее getChanged
         *
         */
        getChangedData: function(mode) {
            var changed = {};

            $.each(this.getFieldsNames(mode), $.proxy(function(i, name){
                if (this.fields[name].isChanged())  {
                    changed[name] = this.get(name);
                }
            }, this));

            return changed;
        },


        /**
         * Изменились ли значения полей по сравнению с состоянием  stateData
         * @protected
         * @param {Object} stateData данные для сравнения
         * @param {String} compareMode название типа toServer|fromServer|input|data . Если выбран тип сравния data
         * то мы сравниваем только поля, имеющиеся в stateData, а остальные не трогаем
         * @returns {Boolean}
         *
         *
         */
        isDataChanged: function(stateData, compareMode) {
            var changed = false;

            compareMode = compareMode || 'data';
            if (compareMode == 'data') {
                $.each(stateData, $.proxy(function(name, value){
                    if (!this.isFieldEqual(name, value)) {
                        changed = true;
                        return false;
                    }
                }, this));
            } else {
                var fields = this.getFieldsNames(compareMode);
                fields && $.each(fields, $.proxy(function(i, name){
                    if (!this.isFieldEqual(name, stateData[name])) {
                        changed = true;
                        return false;
                    }
                }, this));
            }

            return changed;
        },


        /**
         * Апдейтит поля модели заданными данными
         * @protected
         * @param {Object} data данные для апдейта data = {name: value, ...}
         * @param {BEM} source BEM-блок, вызвавший метод
         * @param {Boolean} doChildren нужно ли апдейтить дочерние модели
         * @returns {BEM}
         *
         *
         */
        update: function(data, source, doChildren) {
            var _this = this;
            data = data || {};
            $.each(data, function(name, value){
                var field = _this.fields[name];

                if (!field) return;
                _this.set(name, value, source);
            });

            doChildren && this.childModels && $.each(this.childModels, function (modelName, info) {
                var childData = info.source ? data[info.source] : data;
                if (!childData) {
                    return true;
                }
                _this.getChildModel(modelName).update(childData, source, doChildren);
            });

            return this.trigger('quick-change', {source: source});
        },


        /**
         * Вычисляем значения полей, напрямую зависящих от значения поля name
         * @private
         * @param {String} name имя поля
         * @param {Boolean} isInit Находимся ли мы на стадии инициализации
         * @param {BEM} source BEM-блок, вызвавший метод
         * @param {String} [eventName] событие инициировавшее изменение
         * @returns {BEM}
         *
         *
         */
        calcChildren: function(name, isInit, source, eventName) {
            var fieldsScheme = this.getFieldScheme(name);

            if (fieldsScheme && fieldsScheme.children) {
                $.each(fieldsScheme.children, $.proxy(function(i, childName) {
                    this.fields[childName] &&
                    this[isInit ? 'initField' : 'set'](childName, this.getFieldScheme(childName).calcValue.call(this), source, eventName)
                },  this))
            }

            return this;
        },

        getFieldScheme: function(fieldName) {
            var fieldsScheme = this.fieldsSchemes[fieldName];

            return fieldsScheme && $.isFunction(fieldsScheme) ? fieldsScheme.call(this) : fieldsScheme;
        },

        /**
         * Инициализируем поле name со значением value
         * Т.е. присваиваем полю значение, считая что это состояние начальнок
         * @protected
         * @param {String} name имя поля
         * @param {String|Boolean|Number} value значение
         * @param {BEM} source BEM-блок, вызвавший метод
         * @param {String} [eventName] событие, инициировавшее изменение
         * @returns {BEM}
         */
        initField: function(name, value, source, eventName) {

            var field = this.fields[name];

            if (!field) return this;
            if (!field.isEqual(value)) {
                field.initData(value, source, eventName);
                this.calcChildren(name, true, source, eventName);
            }

            return this;
        },


        /**
         * Выполняем метод methodName на дочерних моделях
         * @private
         * @param {String} methodName имя метода
         * @param {Array} params значение
         * @returns {BEM}
         */
        doChildrenModels: function(methodName, params) {
            if (this.childModels)  {
                var _this = this;

                $.each(this.childModels, function(name) {
                    var model = _this.getChildModel(name);

                    model[methodName].apply(model, params);
                })
            }

            return this;
        },


        /*
         * @deprecated - для совместимости со старым интерфейсом, новый вариант - on
         */
        bind: function(e) {
            this.__self.on.apply(this, arguments);

            return this;
        },


        /**
         * Подписываемся на событие на конкретном поле модели
         * @protected
         * @param {String} name имя поля или полей в виде 'name1 name2 name3'
         * @param {Object|String} e событие или название события
         * @param {Object} data дополнительные данные
         * @param {Function} fn функция-обработчик
         * @param {Object} ctx контекст выполения события
         * @returns {BEM}
         */
        onField: function(name, e, data, fn, ctx) {
            var names = name.split(' '), field;

            for (var i = 0; i < names.length; i++) {
                field = this.fields[names[i]];
                field && field.on.apply(field, Array.prototype.slice.call(arguments, 1));
            }

            return this;
        },

        /**
         * Выстреливаем событием на данном поле/полях
         * @protected
         * @param {String} name имя поля или полей в виде 'name1 name2 name3'
         * @param {Object|String} e событие или название события
         * @param {Object} data дополнительные данные
         *
         * @returns {BEM}
         */
        triggerField: function(name, e, data) {
            var args = Array.prototype.slice.call(arguments, 1),
                field = this.fields[name];

            field && field.trigger.apply(field, args);

            return this;
        },


        /**
         * Отписываемсяс от события на конкретном поле модели
         * @protected
         * @param {String} name имя поля или полей в виде 'name1 name2 name3'
         * @param {Object|String} e событие или название события
         * @param {Function} fn функция-обработчик
         * @param {Object} ctx контекст выполения события
         * @returns {BEM}
         */
        unField: function(name, e, fn, ctx) {
            var names = name.split(' '),
                field;

            for (var i = 0; i < names.length; i++) {
                field = this.fields[names[i]];
                field && field.un.apply(field, [e, fn, ctx]);
            }

            return this;
        },


        _fireChange: function(source) {
            this.trigger('change', { changed: this.changed, source: source });
            this.changed = [];
        },

        getChildModel: function(name) {
            if (this.childModelsHash[name]) return this.childModelsHash[name];
            var info = this.childModels[name] || {};
            return this.childModelsHash[name] = BEM.blocks['i-models-manager'].get(info.path || this.path(), name);
        },

        //deprecated
        validate: function(doChildren) {
            return this.validateModel(doChildren)
        },

        /**
         * Валидирует модель в соответствии с заданной в декларации модели схемой
         * @param {Boolean} [doChildren] - нужно ли проверять дочерние модели
         * @returns {Boolean}
         */
        validateModel: function(doChildren) {
            var _this = this,
                isValid = this.validator ? this.validator.validateModel() : true;


            if (this.childModels)  {
                $.each(this.childModels, function(name, info) {

                    if (!info || !info.condition || info.condition.call(_this)) {
                        isValid = _this.getChildModel(name).validate(doChildren) && isValid;
                    }
                })
            }

            this.trigger('validate', {isValid: isValid});
            return isValid;
        },


        /**
         * Возвращяет массив с сообщениями об ошибке для данного поля
         * @param {String} fieldId - поле модели
         * @param {['errors'|'warnings']} [type]  что нужно проверить - ошибку/предупреждение
         * @returns {Array}
         */
        getErrorsMessagesForField: function(fieldId, type) {
            return this.validator.getErrorsMessagesForField(fieldId, type);
        },


        /**
         * Возвращяет массив с сообщениями об ошибке для всей модели
         * @param {['errors'|'warnings']} [type]  что нужно проверить - ошибку/предупреждение
         * @returns {Array}
         */
        getErrorsMessagesForModel: function(type) {
            return this.validator.getErrorsMessagesForModel(type);
        },

        //deprecated
        getErrorTextForModel: function(type) {
            return this.getErrorsMessagesForModel(type);

        },

        //deprecated
        getErrorTextForField: function(type, fieldId) {
            return this.getErrorsMessagesForField(type, fieldId);
        },


        getErrors: function(fieldId) {
            if (!this.validator) return {};

            return this.validator.get('errors', fieldId);
        },

        getWarnings: function(fieldId) {
            if (!this.validator) return {};

            return this.validator.get('warnings', fieldId);
        },


        hasErrors: function(fieldId) {
            if (!this.validator) return false;

            return this.validator.has('errors', fieldId);
        },

        hasWarnings: function(fieldId) {
            if (!this.validator) return false;

            return this.validator.has('warnings', fieldId);

        },

        clearErrors: function(fieldId) {
            if (!this.validator) return this;

            return this.validator.clear('errors', fieldId);

        },

        clearWarnings: function(fieldId) {
            if (!this.validator) return this;

            return this.validator.clear('warnings', fieldId);


        },

        validateField: function(fieldId) {
            if (!this.validator) return true;

            return this.validator.validateField(fieldId);
        },

        /**
         * Проверяет есть ли ошибки в модели
         * @param {Boolean} doChildren - нужно ли проверять дочерние модели на наличие ошибок
         * @returns {Boolean}
         */
        modelHasErrors: function(doChildren) {
            if (!this.validator) return false;

            var _this = this,
                hasErrors = this.validator.has('errors');

            this.childModels && doChildren && $.each(this.childModels, function(name, info) {
                hasErrors = _this.getChildModel(name).modelHasErrors(doChildren) || hasErrors;

            });

            return hasErrors;
        },


        getData: function() {
            return this.memento(0, 1);
        }

    }, /** @lends BEM.MODEL */{
        /**
         * Хранилище деклараций моделей (хэш по имени модели)
         * @static
         * @protected
         * @type Object
         */
        models : {},

        triggers : {
            'changeModel': {},
            'changeField': {},
            'deleteModel': {}
        },

        /**
         * Функция декларации модели
         * @class Базовый блок для создания моделей
         * @constructs
         * @private
         * @param {Object} decl полное описание модели
         */
        decl : function(decl) {

            decl.name && (decl.model = decl.name);

            if (decl.baseModel) {

                var baseModelDecl = this.models[decl.baseModel],
                    baseScheme = baseModelDecl ? baseModelDecl.getScheme() : false;

                if(!baseScheme)
                    throw('baseModel "' + decl.baseModel + '" for "' + decl.model + '" is undefined');

                // TODO: подумать про наследование
                decl.baseModel && (decl.scheme = $.extend(true, decl.scheme, baseScheme));
            }

            this.models[decl.model] = new ($.inherit(
                $.observable, {
                    storage: {},
                    getScheme: function() {
                        return decl.scheme;
                    }
                },
                {}));
        },


        /**
         * Создаёт модель
         * @param {String} modelName имя модели
         * @param {String} modelPath - путь к модели
         * @param {Object} [data] данные для модели
         * @returns {Object}
         */
        create : function(modelName, modelPath, data) {
            var model = new MODEL(modelName, data);

            MODEL._addModel(model.path(modelPath), modelName, modelPath);
            return model;
        },


        /**
         * Добавляет модель в хранилище
         * @private
         * @param {Object} model модель
         * @param {String} name имя модели
         * @param {String} path путь к модели
         * @returns {Object}
         */
        _addModel: function(model, name, path) {
            var _this = this,
                currTriggers = this.triggers.changeModel[name],
                currTriggersField = this.triggers.changeField[name],
                currDeleteTriggers = this.triggers.deleteModel[name],
                regExp;

            MODEL.models[name].storage[path] = model;

            if (!currTriggers && !currTriggersField && !currDeleteTriggers) return this;

            currTriggers && currTriggers.length > 0 &&
                $.each(currTriggers, function(i, event) {
                    regExp = new RegExp(_this._getPathRegexp(event.path), 'g');
                    if (regExp.test(path)) {
                        model.on(event.name, { model: model }, event.callback)
                    }
                });

            currTriggersField && currTriggersField.length > 0 &&
                $.each(currTriggersField, function(i, event) {
                    regExp = new RegExp(_this._getPathRegexp(event.path), 'g');
                    if (regExp.test(path)) {
                        model.onField(event.fieldName, event.name, { model: model }, event.callback);
                    }
                });

            currDeleteTriggers && currDeleteTriggers.length > 0 &&
                $.each(currDeleteTriggers, function(i, event) {
                    regExp = new RegExp(_this._getPathRegexp(event.path), 'g');
                    if (regExp.test(path)) {
                        model.on('delete', event.name, event.callback);
                    }
                });

            return this;
        },


        /**
         * Удаляет модель из общей коллекции
         * @protected
         * @param {String}  path путь к модели
         * @param {String}  name имя модели
         * @returns {Object}
         */
        deleteModel: function(path, name) {
            MODEL.models[name].storage[path].trigger('delete');
            delete MODEL.models[name].storage[path];
            delete modelsGroupsCache[name];

            return this;
        },


        /**
         * Возвращает модели с заданным именем, лежащие по заданному пути
         * @protected
         * @param {String}  path путь к модели
         * @param {String}  modelName имя модели
         * @param {Boolean} [dropCash] не учитывать текущий кэш
         * @returns {Array}
         */
        getGroup: function(path, modelName, dropCash) {

            if (!path) return [];

            dropCash === undefined && (dropCash = true);

            if (!dropCash && modelsGroupsCache[modelName] && modelsGroupsCache[modelName][path]) return modelsGroupsCache[modelName][path];

            var modelsArray = [],
                pathArr = path.split(','),
                regexp;

            for (var i = 0; i < pathArr.length; i++) {
                regexp = this._getPathRegexp(pathArr[i]);

                $.each(BEM.MODEL.models[modelName].storage, function(currPath, model) {

                    if ((new RegExp(regexp, 'g')).test(currPath)) {
                        modelsArray.push(model);
                    }

                })
            }
            modelsGroupsCache[modelName] = modelsGroupsCache[modelName] || {};
            modelsGroupsCache[modelName][path] = modelsArray;
            return modelsArray;
        },


        /**
         * Возвращает строку для формирования RegExp-а для поиска модели
         * @private
         * @returns {String}
         */
        _getPathRegexp: function(path) {
            return path.replace(/\*/g, '([^\&:]*)') + '$';
        },


        /**
         * Возвращает модель с заданным именем, лежащую по заданному пути. Если такой модели не существует,
         * создаст модель с пустыми данными согласно задекларированной для name схемы
         * @protected
         * @param {String} path путь к модели
         * @param {String} name имя модели
         * @returns {Object}
         */
        get: function(path, name) {
            if (!MODEL.models[name])
                throw('model "' + name + '" is not declared');
            return MODEL.models[name].storage[path] || MODEL.create(name, path, {});
        },


        /**
         * Вешает обработчик события, онисанного в event на каждую из моделей с именем name, лежащую по заданному пути
         * если позже добавится модель с тем же путём - на неё событие тоже навесится
         * @protected
         * @param {String} path путь к модели
         * @param {String} name имя модели
         * @param {Object} event описание события вида {name: name1, callback: function() {...}}
         * @returns {Object}
         */
        onGroup: function(path, name, event) {
            BEM.MODEL.eachModel(function(id, elem) {
                elem.on(event.name, { model: elem }, event.callback);
            }, path, name);

            if (!this.triggers.changeModel[name]) this.triggers.changeModel[name] = [];

            event.path = path;
            this.triggers.changeModel[name].push(event);

            return this;
        },

        /**
         * Навешивает обработчик на событие удаления модели из коллекции
         * @protected
         * @param {String}  path путь к модели
         * @param {String}  modelName имя модели
         * @param {Function}  callback колбэк
         * @param {Object}  context контекст выполнения колбэка
         * @returns {Object}
         */
        onDelete: function(path, modelName, callback, context) {
            BEM.MODEL.eachModel(function(id, elem) {
                elem.on('delete', callback, context)
            }, path, modelName);

            if (!this.triggers.deleteModel[modelName]) this.triggers.deleteModel[modelName] = [];

            this.triggers.deleteModel[modelName].push({
                name: modelName,
                callback: callback,
                context: context,
                path: path
            });

            return this;
        },


        // TODO: добавить метод un

        /**
         * Вешает обработчик события, онисанного в event на поле каждой из моделей с именем name, лежащую по заданному пути
         * @protected
         * @param {String} path путь к модели
         * @param {String} modelName имя модели
         * @param {Object} event описание события вида {name: name1, fieldName: fieldName, callback: function() {...}}
         * @returns {Object}
         */
        onGroupField: function(path, modelName, event) {
            BEM.MODEL.eachModel(function(id, elem) {
                elem.onField(event.fieldName, event.name, { model: elem }, event.callback)
            }, path, modelName);

            if (!this.triggers.changeField[modelName]) this.triggers.changeField[modelName] = [];

            event.path = path;
            this.triggers.changeField[modelName].push(event);

            return this;
        },


        /**
         * Удаляет обработчик события, онисанного в event на поле каждой из моделей с именем name, лежащую по заданному пути
         * @protected
         * @param {String} path путь к модели
         * @param {String} modelName имя модели
         * @param {Object} event описание события вида {name: name1, fieldName: fieldName, callback: function() {...}}
         * @returns {Object}
         */
        unField: function(path, modelName, event) {

            return BEM.MODEL.eachModel(function(id, elem) {
                elem.unField(event.fieldName, event.name)
            }, path, modelName);

        },


        /**
         * Тригерит событие 'change.<имя поля>' на каждой из моделей
         * @deprecated
         * @param {Object} source
         * @param {String} path путь к модели
         * @param {String} name имя модели
         * @returns {Object}
         */
        triggerGroup: function(source, path, name)  {
            this.trigger('change.' + name, source);

            return BEM.MODEL.eachModel(function(id, model) {
                model.trigger('change.' + name, source);
            }, path, name);
        },


        /**
         * Апдейтит все модели соответствующие заданному имени/пути
         * @deprecated
         * @param {Object} data данные для аплейта
         * @param {Object} source
         * @param {String} path путь к модели
         * @param {String} name имя модели
         * @returns {Object}
         */
        update: function(data, source, path, name) {
            BEM.MODEL.eachModel($.proxy(function(id, model) {
                model.update(data[id], source, false);

            }, this), path, name);

            return this.triggerGroup.call(this, source, path, name);
        },


        /**
         * Применяет callback ко всем моделям, соответствующим заданному имени и пути
         * @protected
         * @param {Function} callback функция-обработчик
         * @param {String} path путь к модели
         * @param {String} modelName имя модели
         * @param {Boolean} [dropCash] сбросить текущий кэш моделей
         * @returns {Object}
         */
        eachModel: function(callback, path, modelName, dropCash) {
            var modelsByPath = BEM.MODEL.getGroup(path, modelName, dropCash);
            modelsByPath && modelsByPath.length > 0 &&
                $.each(modelsByPath, function (id, model) {
                    return callback(id, model)
                });

            return this;
        }


    });

    //todo - для совместимости со старым кодом
    BEM.blocks['i-models-manager'] = BEM.MODEL;

})(BEM, jQuery);
