y5.require('Utils', function () {

var Utils = y5.Utils,
    regexp3   = /^#([0-9a-fA-F]{3})$/,
    regexp6   = /^#([0-9a-fA-F]{6})$/,
    regexpRGB = /^rgb\s*\(\s*(\-?\d+%?)\s*,\s*(\-?\d+%?)\s*,\s*(\-?\d+%?)/,
    emptyFunction = y5.VOID;

function getColorHex(c) {
    return Utils.hex2dec(c + c)
}

function getColorValue(c) {
    var num = parseInt(c, 10);
    var value = String(c).indexOf('%') == -1 ? num : Math.round(num * 2.55);

    if (value > 255) value = 255;
    if (value < 0)   value = 0;

    return value;
}

function parse3(color) {
    var match = color.match(regexp3);

    if (match) {
        var tokens = match[1].split('');
        return [
            getColorHex(tokens[0]),
            getColorHex(tokens[1]),
            getColorHex(tokens[2])
        ];
    }

    return null;
}

function parse6(color) {
    var match = color.match(regexp6);

    if (match) {
        var tokens = match[1].match(/../g);
        return [
            Utils.hex2dec(tokens[0]),
            Utils.hex2dec(tokens[1]),
            Utils.hex2dec(tokens[2])
        ];
    }

    return null;
}

function parseRGB(color) {
    var match = color.match(regexpRGB);

    if (match) {
        return [
            match[1],
            match[2],
            match[3]
        ];
    }

    return null;
}

function strToColor(color) {
    return typeof color == 'string' ? new Color(color) : color;
}

/**
 * Преобразует значение цвета в пространстве RGB в представление HSL.
 * http://en.wikipedia.org/wiki/HSL_color_space.
 *
 * @param {Number} r Красный (0..255)
 * @param {Number} g Зеленый (0..255)
 * @param {Number} b Синий (0..255)
 * @returns {Array} Представление HSL
 * @private
 */
function rgbToHsl(r, g, b) {
    r /= 255;
    g /= 255;
    b /= 255;

    var max = Math.max(r, g, b),
        min = Math.min(r, g, b),
        h, s,
        l = (max + min) * 50,  // (/2*100);
        d = max - min;

    if (max == min) {
        h = s = 0; // ахроматический
    } else {
        s = l > 50 ? d / (2 - max - min) : d / (max + min);
        switch(max) {
            case r:
                h = (g - b) / d * 60 + 360;
                break;
                
            case g:
                h = (b - r) / d * 60 + 120;
                break;
                
            case b:
                h = (r - g) / d * 60 + 240;
                break;
        }
        h = h % 360;
    }
    return [Math.round(h), Math.round(s*100), Math.round(l)];
}

/**
 * Преобразует значение цвета в пространстве HSL в представление RGB.
 * http://en.wikipedia.org/wiki/HSL_color_space.
 *
 * @param {Number} h Тон (0..360)
 * @param {Number} s Насыщенность (0..100)
 * @param {Number} v Яркость (0..100)
 * @returns {Array} Представление RGB
 * @private
 */
function hslToRgb (h, s, l) {
    if (s == 0) {
        l /= 100;
        return [l * 255, l * 255, l * 255]; // ахроматический
    }
    
    function hue2rgb(t) {
        if(t < 0) t += 1;
        if(t > 1) t -= 1;
            
        if(t < 1/6) return p + (q - p) * 6 * t;
        if(t < 1/2) return q;
        if(t < 2/3) return p + (q - p) * (2/3 - t) * 6;
        return p;
    };
    
    h = h/360;
    s /= 100;
    l /= 100;
        
    var r, g, b,
        q = l < 0.5 ? l * (1 + s) : l + s - l * s,
        p = 2 * l - q;
    
    r = hue2rgb(h + 1/3);
    g = hue2rgb(h);
    b = hue2rgb(h - 1/3);

    return [Math.round(r * 255), Math.round(g * 255), Math.round(b * 255)];
}

var ColorParsers = [parse6, parseRGB, parse3];

/**
 * Класс для работы с цветом: равенство, разность, контраст, изменения цвета в пространстве HSL, RGB &lt;-&gt; HSL
 *
 * @see <xref href="http://www.wat-c.org/tools/CCA/1.1/#aert"/>.
 *
 * @class Содержит функции для работы с цветом.
 * @name y5.Color
 * @param {String} color Цвет в формате RGB (<xref href="http://www.w3.org/TR/CSS21/syndata.html#value-def-color"/>)
 * 
 * @example
 * new y5.Color().setRGB(51,51,255).alterLightness(20).getRGB();
 * // -> rgb(153,153,255) 
 */
y5.Color = function(color) {
    if (color) {
        var rgb = null,
            i = 0,
            l = ColorParsers.length;
        for (; i < l; i++) {
            if ((rgb = ColorParsers[i](color))) {
                this.setRGB(rgb[0], rgb[1], rgb[2]);
                break;
            }
        }
    }
};

y5.Color.prototype = {
    
    hex: '#000000',
    rgb: 'rgb(0,0,0)',
    red: 0,
    green: 0,
    blue: 0,
    
    hsl: 'hsl(0,0,0)',
    hue: 0,
    saturation: 0,
    lightness: 0,
    
    
    /**
     * Создает новый объект y5.Color из данного.
     * @name y5.Color.clone
     * @memberOf y5.Color
     * @function
     * @returns {y5.Color} Копия объекта
     */
    clone: function() {
        return new Color(this.hex);
    },

    /**
     * Возвращает строковое представление цвета #rrggbb.
     * @name y5.Color.toString
     * @memberOf y5.Color
     * @function
     * @returns {String} Цвет
     */
    toString: function() {
        return this.hex;
    },

    /**
     * Проверяет два цвета на равенство.
     * @name y5.Color.equals
     * @memberOf y5.Color
     * @function
     *
     * @param {y5.Color | String} color Цвет для сравнения
     * @returns {Boolean} Цвет совпадает/не совпадает
     *
     * @example
     * new y5.Color('#fff').equals(new y5.Color('rgb(255,255,255)'));
     * // -> true
     */
    equals: function(color) {
        return this.hex == strToColor(color).hex;
    },

    /**
     * Возвращает абсолютное значение разности между яркостью двух цветов.
     * @name y5.Color.brightnessDiff
     * @memberOf y5.Color
     * @function
     *
     * @param {y5.Color | String} color Цвет для сравнения
     * @returns {Number} Разность яркости цветов
     *
     * @example
     * new y5.Color('#000').brightnessDiff(new y5.Color('#00f'));
     * // -> 29
     */
    brightnessDiff: function(color) {
        return Math.abs(this.brightness() - strToColor(color).brightness());
    },

    /**
     * Возвращает значение разности двух цветов.
     * @name y5.Color.difference
     * @memberOf y5.Color
     * @function
     *
     * @param {y5.Color | String} color Цвет для сравнения
     * @returns {Number} Разность цвета
     *
     * @example
     * new y5.Color('#fff').difference(new y5.Color('#00f'))
     * // -> 510
     */
    difference: function(color) {
        color = strToColor(color);

        return (
            (Math.max(this.red, color.red) - Math.min(this.red, color.red)) +
            (Math.max(this.green, color.green) - Math.min(this.green, color.green)) +
            (Math.max(this.blue, color.blue) - Math.min(this.blue, color.blue))
        );
    },

    /**
     * Проверяет два цвета на контрастность.
     * @name y5.Color.contrast
     * @memberOf y5.Color
     * @function
     *
     * @param {y5.Color | String} color Цвет для сравнения
     * @returns {Boolean} Контраст/не контраст
     *
     * @example
     * new y5.Color('#fff').contrast(new y5.Color('#000'));
     * // -> true
     */
    contrast: function(color) {
        color = strToColor(color);

        return (this.brightnessDiff(color) > 125) && (this.difference(color) > 400);
    },

    /**
     * Возвращает значение яркости цвета.
     * @name y5.Color.brightness
     * @memberOf y5.Color
     * @function
     *
     * @returns {Number} Значение яркости
     *
     * @example
     * new y5.Color('#000').brightness();
     * // -> 0
     * new y5.Color('#fff').brightness();
     * // -> 255
     * new y5.Color('#7f7f7f').brightness();
     * // -> 127
     */
    brightness: function() {
        return Math.round((.299 * this.red) + (.587 * this.green) + (.114 * this.blue));
    },

    /**
     * Возвращает серый цвет для данного.
     * @name y5.Color.gray
     * @memberOf y5.Color
     * @function
     *
     * @returns {y5.Color} Новый цвет
     *
     * @example
     * new y5.Color('#7f7f00').gray().getRGB();
     * // -> rgb(113,113,113)
     */
    gray: function() {
        var luma = this.brightness();
        return new Color().setRGB(luma, luma, luma);
    },

    /**
     * Возвращает противоположный цвет для данного.
     * @name y5.Color.invert
     * @memberOf y5.Color
     * @function
     *
     * @returns {y5.Color} Новый цвет
     *
     * @example
     * new y5.Color('#7f7f00').invert().getRGB();
     * // -> rgb(128,128,255)
     */
    invert: function() {
        return new Color().setRGB(255 - this.red, 255 - this.green, 255 - this.blue);
    },
    
    /**
     * Устанавливает значение цвета в hex-формате
     * @name y5.Color.setHex
     * @memberOf y5.Color
     * @function
     * @param {String} hex Цвет в hex-формате
     * @returns {Object} y5.Color
     */
    setHex: function (hex) {
        var rgb;
        if ((rgb = parse6(hex)) || (rgb = parse3(hex))) {
            this.setRGB(rgb[0], rgb[1], rgb[2]);
        }
        return this;
    },
    
    /**
     * Возвращает значение цвета в hex-формате
     * @returns {String} Цвет в hex-формате
     */
    getHex: function () {
        return this.hex;
    },
    
    _setRGB: function(red, green, blue) {
        this.red = getColorValue(red);
        this.green = getColorValue(green);
        this.blue = getColorValue(blue);
    },
    
    _setHSL: function(h, s, l) {
        this.hue = h;
        this.saturation = s;
        this.lightness = l;
    },
    
    /**
     * Обновляет три состояния (RGB, HSL, HSV)
     * @private
     */
    _updateState: function(state) {
        var color;
        
        if (state == "rgb" || state == "red" || state == "green" || state == "blue") {
            color = rgbToHsl(this.red, this.green, this.blue);
            this._setHSL(color[0], color[1], color[2]);
            
        } else {
            color = hslToRgb(this.hue, this.saturation, this.lightness);
            this._setRGB(color[0], color[1], color[2]);
        }
        
        this.rgb = 'rgb(' + this.red + ',' + this.green + ',' + this.blue + ')';
        this.hex = '#' + Utils.dec2hex(this.red) + Utils.dec2hex(this.green) + Utils.dec2hex(this.blue);
        this.hsl = 'hsl(' + this.hue + ',' + this.saturation + ',' + this.lightness + ')';
    },
    
/// RGB    
    /**
     * Устанавливает значения цветов RGB.
     * @name y5.Color.setRGB
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} red Красный (0..255)
     * @param {Number} blue Синий (0..255)
     * @param {Number} green Зеленый (0..255)
     * @returns {y5.Color}
     * 
     * @example
     * new y5.Color().setRGB(255,255,0).getHex();
     * // -> #FFFF00
     */
    setRGB: emptyFunction,
     
    /**
     * Возвращает значение цвета в формате rgb(R,G,B).
     * @name y5.Color.getRGB
     * @memberOf y5.Color
     * @function
     *
     * @returns {String} rgb(R,G,B)
     *
     * @example
     * new y5.Color('#7f7f00').getRGB();
     * // -> rgb(127,127,0)
     */
    getRGB: emptyFunction,
     
    /**
     * Изменяет значение каждого канала RGB.
     * @name y5.Color.alterRGB
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} red Величина, на которую надо изменить параметр
     * @param {Number} blue Величина, на которую надо изменить параметр
     * @param {Number} green Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     * 
     * @example
     * new y5.Color().setRGB(255,255,0).alterRGB(-55,-55,10).getRGB();
     * // -> rgb(200,200,10)
     */
    alterRGB: emptyFunction,
         
/// HSL
    /**
     * Возвращает значение цвета в формате hsl(H,S,L).
     * @name y5.Color.getHSL
     * @memberOf y5.Color
     * @function
     *
     * @returns {String} hsl(H,S,L)
     *
     * @example
     * new y5.Color('#7f7f00').getHSL();
     * // -> hsl(60, 100, 25)
     */
    getHSL: emptyFunction,
     
    /**
     * Устанавливает значения цветов HSL.
     * @name y5.Color.setHSL
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} hue Тон (0..360)
     * @param {Number} saturation Насыщенность (0..100)
     * @param {Number} lightness Светлота (0..100)
     * @returns {y5.Color}
     * 
     * @example
     * new y5.Color().setHSL(240,100,60).getRGB();
     * // -> rgb(51,51,255)
     */
    setHSL: emptyFunction,
     
    /**
     * Изменяет значения цветов HSL.
     * @name y5.Color.alterHSL
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} hue Величина, на которую надо изменить параметр
     * @param {Number} saturation Величина, на которую надо изменить параметр
     * @param {Number} lightness Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     * 
     * @example
     * new y5.Color('rgb(51,51,255)').alterHSL(0,0,20).getRGB();
     * // -> rgb(153,153,255) 
     */
    alterHSL: emptyFunction,
     
// Hue
    /**
     * Возвращает значение тона.
     * @name y5.Color.getHue
     * @memberOf y5.Color
     * @function
     *
     * @returns {Number} Тон
     *
     * @example
     * new y5.Color('#7f7f00').getHue();
     * // -> 60
     */
    getHue: emptyFunction,

    /**
     * Устанавливает значение тона.
     * @name y5.Color.setHue
     * @memberOf y5.Color
     * @function
     * 
     * @param {Number} hue Тон (0..360)
     * @returns {y5.Color}
     * 
     * @example
     * new y5.Color('#7f7f00').setHue(140).getBlue();
     * // -> 43
     */
    setHue: emptyFunction,
     
    /**
     * Изменяет значение тона.
     * @name y5.Color.alterHue
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} diff Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     * 
     * @example
     * new y5.Color('#7f7f00').alterHue(200).getHSL();
     * // -> hsl(260,100,25)
     */
    alterHue: emptyFunction,
     
// Saturation
    /**
     * Возвращает значение насыщенности.
     * @name y5.Color.getSaturation
     * @memberOf y5.Color
     * @function
     * 
     * @returns {Number} Насыщенность
     *
     * @example
     * new y5.Color('#7f7f00').getSaturation();
     * // -> 100
     */
    getSaturation: emptyFunction,

    /**
     * Устанавливает значение насыщенности.
     * @name y5.Color.setSaturaton
     * @memberOf y5.Color
     * @function
     * @name y5.Color.getRGB
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} saturation Насыщенность (0..100)
     * @returns {Object} y5.Color
     */
    setSaturation: emptyFunction,
     
    /**
     * Изменяет значение насыщенности.
     * @name y5.Color.alterSaturation
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} diff Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     */
    alterSaturation: emptyFunction,
     
// Lightness
    /**
     * Возвращает значение яркости.
     * @name y5.Color.getLightness
     * @memberOf y5.Color
     * @function
     *
     * @returns {Number} Яркость
     *
     * @example
     * new y5.Color('#7f7f00').getSaturation();
     * // -> 0.24901960784313726
     */
    getLightness: emptyFunction,

    /**
     * Устанавливает значение яркости.
     * @name y5.Color.setLightness
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} lightness Яркость (0..100)
     * @returns {y5.Color}
     */
    setLightness: emptyFunction,
     
    /**
     * Изменяет значение яркости.
     * @name y5.Color.alterLightness
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} diff Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     */
    alterLightness: emptyFunction,
     
// Red     
    /**
     * Возвращает значение красного канала.
     * @name y5.Color.getRed
     * @memberOf y5.Color
     * @function
     *
     * @returns {Number} red
     */
    getRed: emptyFunction,

    /**
     * Устанавливает значение красного канала.
     * @name y5.Color.setRed
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} red Красный (0..255)
     * @returns {y5.Color}
     */
    setRed: emptyFunction,
     
    /**
     * Изменяет значение красного канала.
     * @name y5.Color.alterRed
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} diff Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     */
    alterRed: emptyFunction,
     
// Green
     /**
     * Возвращает значение зеленого канала.
     * @name y5.Color.getGreen
     * @memberOf y5.Color
     * @function
     *
     * @returns {Number} green
     */
    getGreen: emptyFunction,

    /**
     * Устанавливает значение зеленого канала.
     * @name y5.Color.setGreen
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} diff Зеленый (0..255)
     * @returns {y5.Color}
     */
    setGreen: emptyFunction,
     
    /**
     * Изменяет значение зеленого канала.
     * @name y5.Color.alterGreen
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} green Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     */
    alterGreen: emptyFunction,
     
// Blue
    /**
     * Возвращает значение синего канала.
     * @name y5.Color.getBlue
     * @memberOf y5.Color
     * @function
     *
     * @returns {Number} blue
     */
    getBlue: emptyFunction,

    /**
     * Устанавливает значение синего канала.
     * @name y5.Color.setBlue
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} blue Синий (0..255)
     * @returns {y5.Color}
     */
    setBlue: emptyFunction,
     
    /**
     * Изменяет значение синего канала.
     * @name y5.Color.alterBlue
     * @memberOf y5.Color
     * @function
     *
     * @param {Number} diff Величина, на которую надо изменить параметр
     * @returns {y5.Color}
     */
    alterBlue: emptyFunction
};

// Создаем getters/setters
var Color = y5.Color,
    params = ['Hue', 'Saturation', 'Lightness', 'Red', 'Green', 'Blue'],
    length = params.length,
    i = 0;

for (; i<length; i++) {
    //getter
    Color.prototype['get'+params[i]] = (function(property) {
        return function() {
            return this[property];
        };
    })(params[i].toLowerCase());
    
    //setter
    Color.prototype['set'+params[i]] = (function(property) {
        return function(value) {
            this[property] = value;
            this._updateState(property);
            return this;
        };
    })(params[i].toLowerCase());
    
    //alter
    Color.prototype['alter'+params[i]] = (function(property) {
        return function(value) {
            this[property] = this[property]+value;
            this._updateState(property);
            return this;
        };
    })(params[i].toLowerCase());
}

// get.HSL(), setHSL(), alterHSL()
params = ['HSL', 'RGB'];
var axis = [
    ['hue', 'saturation', 'lightness'],
    ['red', 'green', 'blue']
];
length = params.length;
for (i=0; i<length; i++) {
    Color.prototype['set'+params[i]] = (function(colorSpace){
        return function(p1, p2, p3) {
            this['_set'+colorSpace](p1, p2, p3);
            this._updateState(colorSpace.toLowerCase());
            return this;
        }
    })(params[i]);
    
    Color.prototype['alter'+params[i]] = (function(colorSpace, axis){
        return function(p1, p2, p3) {
            this['set'+colorSpace](this[axis[0]]+p1, this[axis[1]]+p2, this[axis[2]]+p3);
            return this;
        }
    })(params[i], axis[i]);
    
    Color.prototype['get'+params[i]] = (function(colorSpace){
        return function() {
            return this[colorSpace];
        }
    })(params[i].toLowerCase());
}

/**
 * Проверяет, что данный цвет является корректным
 * @param {String} color Цвет в формате #hhh или #hhhhhh
 * @returns {Boolean} Результат проверки
 * @memberOf y5.Color
 * @name y5.Color.isValid
 * @function
 * @static
 */
Color.isValid = function(color) {
    return Color.isValid3(color) || Color.isValid6(color) || Color.isValidRGB(color);
};

/**
 * Проверяет, что данный цвет является 3-значным и корректным
 * @param {String} color Цвет в формате #hhh
 * @returns {Boolean} Результат проверки
 * @memberOf y5.Color
 * @name y5.Color.isValid3
 * @function
 * @static
 */
Color.isValid3 = function(color) {
    return regexp3.test(color);
};

/**
 * Проверяет, что данный цвет является 6-значным и корректным
 * @param {String} color Цвет в формате #hhhhhh
 * @returns {Boolean} Результат проверки
 * @memberOf y5.Color
 * @name y5.Color.isValid6
 * @function
 * @static
 */
Color.isValid6 = function(color) {
    return regexp6.test(color);
};

/**
 * Проверяет, что данный цвет является RBG и корректным
 * @param {String} color цвет в формате rgb(R, G, B)
 * @returns {Boolean} результат проверки
 * @memberOf y5.Color
 * @name y5.Color.isValidRGB
 * @function
 * @static
 */
Color.isValidRGB = function(color) {
    return regexpRGB.test(color);
};

y5.loaded('Color');

});