/**
 * Интерфейсный элемент "ползунок", позволяет задавать количество делений и ориентацию.
 * Позволяет перемещать ползунок мышкой, клавишами:
 *    - кликом по шкале;
 *    - колесиком мыши.
 *
 * <strong>Параметры</strong>:
 *    orientation - vertical или horizontal, по умолчанию vertical.
 *    value  - значение при инициализации, по умолчанию 0, если его нет то смотрится value у hidden поля.
 *    scrollSize - величина, на которую изменяется значение при скролле. По умолчанию - 1. Если 0, то испольуется значение event.scrollDetail
 *  
 * 	  Значения задаются:
 *    values - массив значений для ползунка [1,2,3,4,5,6]
 * 		или
 *    range  - диапазон значений для ползунка [-10, 20].
 *    step   - шаг при заданном range, по умолчанию 1.
 * 
 *    если значения не заданы, то они задаются размерами шкалы
 *
 * <strong>Инициализирует события</strong>:
 *    y5:Slider:move - при любом перемещении ползунка возвращает новое значение.
 *    y5:Slider:change - при изменении значения ползунка возвращает новое значение.
 * 
 * <strong>Слушает события:</strong>:
 *    y5:Slider:set - установить новое значение.
 *
 * <a href="../tests/components/Slider.html">Пример работы</a>
 *
 * @alias y5.Components.Slider
 */

y5.Components.Slider = function(slider, params){

    this.slider = slider;
    this.isVertical = (params.orientation != 'horizontal');
    this.scrollSize = parseInt(params.scrollSize);
    if (isNaN(this.scrollSize)) {
    	this.scrollSize = 1;
    }
    this.startIndex = 0;
    this.index = null;
    this.coefficient = 1;
    this.values = this.getValuesFromParams(params);

    this.button = y5.Dom.getElementByClass('y5-Slider-Button', this.slider) || null;
	this.scale = y5.Dom.getElementByClass('y5-Slider-Scale', this.slider);
	this.grab = y5.Dom.getElementByClass('y5-Slider-Grab', this.slider);
	this.hidden = y5.Dom.getElementByClass('y5-Slider-Hidden', this.slider);
	
	this.plus = y5.Dom.getElementByClass('y5-Slider-Plus', this.slider) || null;
	this.minus = y5.Dom.getElementByClass('y5-Slider-Minus', this.slider) || null;
	this.cachePlusMinus();
        
	this.setMouseListeners();
	
	function setNewValue (value) {
		this.setIndex(this.getIndexFromValue(value));
	}
	y5.Events.observe('y5:Slider:set', setNewValue, this.slider, true, this);

    this.setIndex(this.getIndexFromValue(params.value));

    this.grab.style.visibility = 'visible';
};

y5.Components.Slider.prototype = new function () {
	
	/**
	 * Инициализация массива допустимых значений
	 */
    this.getValuesFromParams = function(params){
        if (params.values){
            return params.values;
        }
        if (params.range){
            var step = params.step || 1;
            while(Math.abs(step) < 1) {
                this.coefficient *= 10;
                step *= 10;
            }
            var curVal = params.range[0] * this.coefficient;
            var end = params.range[1] * this.coefficient;
            var array = [];
            while(curVal < end) {
                array[array.length] = curVal;
                curVal += step;
            }
            array[array.length] = end;
            return array;
        };
        return null;
    };
	
	/**
	 * Кэшируем изображения кнопок "плюс" и "минус"
	 */
    this.cachePlusMinus = function(plus, minus){
        if (this.plus == null || this.minus == null){
            return;
        }
        this.processPlusMinusIcons('plus');
        this.processPlusMinusIcons('minus');
    };
	
	/**
	 * Обработка иконок plus, minus
	 */
	this.processPlusMinusIcons = function (name){
		function restoreImgSrc (e, img) {
			img.src = this[name + 'Src'];
		}
		y5.Events.observe('error', restoreImgSrc, this[name], true, this);

        this[name + 'Src'] = this[name].src;
        this[name + 'SrcDisabled'] = this[name].src.replace(/\.([a-zA-Z]*)$/, '-disabled.$1');
        
        // пытаемся загрузить disabled иконку
        var test_img = document.createElement('img');
        function removeDisabledIcon (e, img) {
        	this[name + 'SrcDisabled'] = this[name + 'Src'];
        }
        y5.Events.observe('error', removeDisabledIcon, test_img, true, this);
		test_img.src = this[name + 'SrcDisabled'];
	};

	this.countSliderParams = function () {
		if (this.isVertical){
            this.grabSize = this.grab.offsetHeight;
            this.scaleSize = this.scale.offsetHeight;
            this.scaleOffset = y5.Dom.offsetTop(this.scale);
        } else {
            this.grabSize = this.grab.offsetWidth;
            this.scaleSize = this.scale.offsetWidth;
            this.scaleOffset = y5.Dom.offsetLeft(this.scale);
        }
        
        this.maxIndex = this.values ? this.values.length - 1 : this.scaleSize;
        
        this.step = this.scaleSize / this.maxIndex;
	}
	
    this.setMouseListeners = function () {
        var _this = this;
		
		function _start (obj, e) {
			_this.start(e);
		}
		
		function _move (obj, e) {
			_this.move(e);
		}
		
		function _up (obj, e) {
			_this.stop(e);
		}
		
		new y5.ObjectMove(this.grab, this.scale, {start: _start, move: _move, up: _up}, y5.Events.observe('click', this.click, this.scale, true, this));
		
        
        if (this.plus){
            function plus(e){
                e.preventDefault();
                this.setIndex(_this.index + 1);
            }
            y5.Events.observe('click', plus, this.plus, true, this);
        }

        if (this.minus){
            function minus(e){
                e.preventDefault();
                this.setIndex(_this.index - 1);
            }
            y5.Events.observe('click', minus, this.minus, true, this);
        }

        y5.Events.observe('DOMMouseScroll', this.scroll, this.scale, true, this);

    };

    this.cropIndex = function(index){
        return Math.max(0, Math.min(this.maxIndex, index));
    };

    this.getValueFromIndex = function (index){
        return this.values ? this.values[index] / this.coefficient : index;
    };

    this.saveValueToHidden = function(index){
       	if (this.hidden) {
       		this.hidden.value = this.getValueFromIndex(index);
       	}
    };

    this.getIndexFromValue = function (value){
        if(typeof(value) == y5.UNDEF || value === null) {
            value = this.hidden.value;
        }
        value *= this.coefficient;
        var index = this.values ? this.values.indexOf(value) : value;
        return index != -1 ? index : 0;
    };

    this.setValueFromIndex = function(index){
        index = this.cropIndex(index);
        if (this.index == index){
            return false;
        }
        this.enableDisableIMG('plus', (index == this.maxIndex));
        this.enableDisableIMG('minus', (index == 0));
        if (this.isVertical){
            // invert because zero on the bottom, but opera 8.5 doesn't allow to set style.bottom;
            this.grab.style.top = (this.scaleSize - this.grabSize) - this.getOffsetFromIndex(index) + 'px';
        }else{
            this.grab.style.left = this.getOffsetFromIndex(index) + 'px';
        }
        this.index = index;
        this.saveValueToHidden(this.index);
        return true;
    };

    this.getOffsetFromIndex = function(index){
        return ((index != this.maxIndex ?  Math.round(index * this.step) : this.scaleSize) - this.grabSize / 2);
    };

    this.countIndexFromOffset = function(offset){
        return Math.round(offset / this.step);
    };

    this.getOffset = function(e){
        if (this.isVertical){
            // Y zero is on top but max value is on top too
            return this.scaleSize - e.pageY + this.scaleOffset;
        }
        // Y zero is on left
        return e.pageX - this.scaleOffset;
    };

    this.moveToIndex = function(value){
        if (this.setValueFromIndex( value )){
            y5.Events.notify('y5:Slider:move', this.slider, true, this.getValueFromIndex(this.index));
        }
    };

    this.setIndex = function(index){
        this.start();
        this.moveToIndex(index);
        this.stop()
    };

    this.move = function(e){
        var index = this.countIndexFromOffset(this.getOffset(e));
        this.moveToIndex(index);
    };

    this.start = function(){
		this.countSliderParams();
        this.startIndex = this.index;
    };

    this.stop = function(){
        if (this.startIndex != this.index){
            y5.Events.notify('y5:Slider:change', this.slider, true, this.getValueFromIndex(this.index));
            this.startIndex = this.index;
        }
    };

    this.click = function(e){
        this.move(e);
        this.stop();
    };

    this.scroll = function(e){
        if (e.scrollDetail == 0){
            return;
        }
        e.preventDefault();
        
        // если размер скролла 0, то используем системное значение
        if (this.scrollSize == 0) {
        	this.setIndex(this.index - e.scrollDetail);
        } else {
        	// вверх
        	if (e.scrollDetail < 0) {
        		this.setIndex(this.index + this.scrollSize);
        		
            // вниз
        	} else {
        		this.setIndex(this.index - this.scrollSize);
        	}
        }
        
    };

    this.enableDisableIMG = function(name, disable){
        if (!this[name]){
            return;
        }
        this[name].src = disable ? this[name + 'SrcDisabled'] : this[name + 'Src'];
    };
};

y5.require(['Dom', 'Events', 'ObjectMove'], function(){y5.loaded('Components.Slider')});