y5.require('Strings', function() {

var Strings = y5.Strings,
    Types = y5.Types,
    cssTextSplitRegexp = /\s*;\s*/g,
    styleSplitRegexp = /\s*:\s*/,
    isHTML = /[<>\s]/,
    pxTest = /\d+px/,
    pxExcept = /z-?index|font-?weight|opacity|zoom|line-?height/i;

/**
 * Устанавливает значение CSS-свойства элемента DOM.
 * @function
 * @private
 * @param {Element} element Элемент
 * @param {String} propname Свойство, например 'margin-left'
 * @param {String} value Значение
 * @returns {Element}
 */
function setPropertyValue (element, property, value) {
    element.style[Strings.camelize(property)] = value;
    return element;
}

/**
 * Возвращает вычисленное значение CSS-свойства элемента DOM.
 * @function
 * @private
 * @param {Element} element Элемент
 * @param {String} propname Свойство, например 'margin-left'
 * @returns {String} Значение свойства, например '5px'
 */
function getPropertyValue (element, property) {
    return y5.Elements.getStyle(element).getPropertyValue(Strings.dasherize(property));
}

/**
 * @class Модуль для создания и работы с элементами DOM
 * @static
 * @name y5.Elements
 */
y5.Elements = {
    /**
     * Создает элемент DOM в соответствии с заданными параметрами
     * @name y5.Elements.create
     * @memberOf y5.Elements
     * @function
     * @param {String} tagName Имя элемента или html-разметка
     * @param {Object} [attributes] Хэш атрибутов вида {name:'...', ...}
     * @param {String} [innerHTML] HTML-код
     * @returns {Element} Элемент
     */
    create: function(tagName, attributes, innerHTML) {
        var element;
        // deprecated: tagName - объект вида {tagName:'...', attributes: {name:'...', ...}}
        if (!Types.string(tagName)) {
            attributes = tagName.attributes;
            tagName = tagName.tagName;
        }

        // если имя тэга
        if (!isHTML.test(tagName)) {
            if (Strings.compare(tagName, 'style', true) == 0) {
                // да, этот код не содержит workaround для создания в IE тега STYLE с атрибутом NAME
                element = document.createElement('div');
                element.innerHTML = '<p>x<\/p><style>' + (innerHTML || attributes.innerHTML || '') + '<\/style>';
                element = element.childNodes[1];
                innerHTML = undefined;
                delete attributes.innerHTML;

            } else {
                // для избежания ошибок от создания неправильных тэгов
                try {
                    if (y5.is_ie && attributes && attributes.name) {
                        element = document.createElement('<' + tagName + ' name="' + attributes.name + '"/>');
                        delete attributes.name;
                    } else {
                        element = document.createElement(tagName);
                    }
                } catch (e) {}
            }
        }

        // если первый аргумент - html-разметка
        if (!element) {
            element = document.createElement('div');

            // если создаем stylе, то для IE6/IE7 надо добавить в начало тэг <p>
            if (Strings.startsWith(tagName, '<style', true)) {
                element.innerHTML = '<p>x<\/p>' + tagName;
                element = element.childNodes[1];

            } else {
                element.innerHTML = tagName;
                element = element.firstChild;
            }
        }

        if (Types.object(attributes)) {
            this.setAttributes(element, attributes);
        }

        if (Types.def(innerHTML)) {
            this.setHTML(element, innerHTML);
        }

        return element;
    },

    /**
     * Устанавливает набор атрибутов элементу DOM.
     * @name y5.Elements.setAttributes
     * @memberOf y5.Elements
     * @function
     * @param {Element} element Элемент DOM
     * @param {Object} [attributes] Список атрибутов, вида {name:'test', ...}
     */
    setAttributes: function(element, attributes) {
        if (!attributes) return;

        var name,
            value,
            styles,
            stylesLength,
            style,
            i;


        for (name in attributes) {
            value = attributes[name];

            switch (name) {
                case 'style':
                case 'cssText':
                    if (element.style.cssText && !(value.indexOf('opacity') != -1 && y5.is_ie)) {
                        element.style.cssText = value;
                    } else {
                        styles = value.split(cssTextSplitRegexp);
                        stylesLength = styles.length;
                        for (i = 0; i < stylesLength; i++) {
                            style = styles[i].split(styleSplitRegexp);
                            setPropertyValue(element, style[0], style[1]);
                        }
                    }
                    break;

                case 'class':
                case 'className':
                    element.className = value;
                    break;

                case 'innerHTML':
                    element.innerHTML = value;
                    break;

                default:
                    element.setAttribute(name, value);
            }
        }
    },

    /**
     * Формирует содержимое элемента DOM из строки HTML-разметки.
     * @name y5.Elements.setHTML
     * @memberOf y5.Elements
     * @function
     * @param {Element} element Элемент DOM
     * @param {String} html HTML-код
     * @returns {Element} Элемент
     */
    setHTML: function(element, html) {
        element.innerHTML = html;
        return element;
    },

    /**
     * <p>Метод позволяет работать с CSS-свойствами DOM-элемента.</p>
     * <p>Имена свойств можно задавать как в CSS-стиле (например margin-top), так и в JavaScript-стиле (marginTop).</p>
     * <p>При установке свойств к значениям имеющим тип number, автоматически прибавляется постфикс "px".</p>
     * <p>При чтении свойств значения имеющие постфикс "px" приводятся к типу number.</p>
     * <p>Логика работы зависит от переданных аргументов:</p>
     * <p>y5.Elements.css(element, propertyName) - возвращает значение CSS-свойства.</p>
     * <p>y5.Elements.css(element, propertyName, propertyValue) - устанавливает значение CSS-свойства.</p>
     * <p>y5.Elements.css(element, priopertyHash) - устанавливает CSS-свойства перечисленные в хеше.</p>
     * @function
     * @name y5.Elements.css
     * @memberOf y5.Elements
     * @param {HTMLElement} element DOM-элемент, для которого осуществляется работа с CSS-свойствами.
     * @param {String | Object} [property] Имя свойства или хеш значений.
     * @param {String | Number} [propertyValue] Значение.
     * @returns {String | Number} Значение CSS-свойства.
     * @emample
     * // Установка нескольких CSS-свойств.
     * y5.Elements.css(element, {
     *      top: 10,
     *      left: "10%",
     *      'margin-left': "1em",
     *      zIndex: 100
     * });
     * // Установка одного CSS-свойства.
     * y5.Elements.css(element, "left", 10);
     * // Получение значения CSS-свойства.
     * var left = y5.Elements.css(element, "left"); -> 10
     */
    css: function (element, property, value) {
        var propertyHash = property,
            prop;

        if (Types.string(property)) {
            // если нет value, то возвращаем значение свойства элемента
            if (Types.undef(value)) {
                value = getPropertyValue(element, property);
                // opera fix
                if (property == 'opacity') {
                    return parseFloat(value, 10);
                }
                return pxTest.test(value) ? parseInt(value, 10) : value;
            } else {
                propertyHash = {};
                propertyHash[property] = value;
            }
        }

        for (prop in propertyHash) {
            setPropertyValue(
                element,
                prop,
                Types.number(value = propertyHash[prop]) && !pxExcept.test(prop) ? value + "px" : value
            );
        }
    },

    /**
     * Возвращает вычисленные стили элемента DOM.
     * @name y5.Elements.getStyle
     * @memberOf y5.Elements
     * @function
     * @param {Element} element Элемент
     * @returns {Object} Стили
     */
    getStyle: function(element) {
        return document.defaultView.getComputedStyle(element, null);
    }
};

var Elements = y5.Elements;

/**
 * @name createElement
 * @memberOf y5.Elements
 * @function
 * @deprecated y5.Elements.create
 */
Elements.createElement = Elements.create;

/**
 * @name setElementAttributes
 * @memberOf y5.Elements
 * @function
 * @deprecated y5.Elements.setAttributes
 */
Elements.setElementAttributes = Elements.setAttributes;

/**
 * @name createElementWithName
 * @memberOf y5.Elements
 * @function
 * @deprecated y5.Elements.createWithName
 */
Elements.createElementWithName = Elements.createWithName;

/**
 * @name createElementFromHTML
 * @memberOf y5.Elements
 * @function
 * @deprecated y5.Elements.createFromHTML
 */
Elements.createElementFromHTML = Elements.create;

/**
 * Создает элемент DOM с заданным параметром name.
 * @name y5.Elements.createWithName
 * @memberOf y5.Elements
 * @function
 * @param {String} tag Имя элемента (имя тега)
 * @param {String} name Атрибут name элемента
 * @returns {Element} Элемент
 * @deprecated y5.Elements.create
 */
Elements.createWithName = function(tag, name) {
    return Elements.create(tag, {name: name});
};

/**
 * Создает элемент DOM из строки HTML-разметки.
 * @name y5.Elements.createFromHTML
 * @memberOf y5.Elements
 * @function
 * @param {String} html Текст HTML
 * @returns {Element} Первый элемент из списка созданных
 * @deprecated y5.Elements.create
 */
Elements.createFromHTML = Elements.create;

/**
 * Возвращает вычисленное значение CSS-свойства элемента DOM.
 * @name y5.Elements.getPropertyValue
 * @memberOf y5.Elements
 * @function
 * @param {Element} element Элемент
 * @param {String} propname Свойство, например 'margin-left'
 * @returns {String} Значение свойства, например '5px'
 * @deprecated y5.Elements.css
 */
Elements.getPropertyValue = Elements.css;

/**
 * Возвращает вычисленное целое значение CSS-свойства элемента.
 * @name y5.Elements.getPropertyValuePx
 * @memberOf y5.Elements
 * @function
 * @param {Element} element Элемент
 * @param {String} propname Свойство, например 'margin-left'
 * @returns {String} Значение свойства, например 5
 * @deprecated y5.Elements.css
 */
Elements.getPropertyValuePx = Elements.css;

/**
 * Возвращает вычисленное числовое значение CSS-свойства элемента DOM.
 * @name y5.Elements.getPropertyValueFloat
 * @memberOf y5.Elements
 * @function
 * @param {Element} element Элемент
 * @param {String} propname Свойство, например 'opacity'
 * @returns {String} Значение свойства, например 0.5
 * @deprecated y5.Elements.css
 */
Elements.getPropertyValueFloat = Elements.css;

/**
 * Устанавливает значение CSS-свойства элемента DOM.
 * @name y5.Elements.setPropertyValue
 * @memberOf y5.Elements
 * @function
 * @param {Element} element Элемент
 * @param {String} propname Свойство, например 'margin-left'
 * @param {String} value Значение
 * @deprecated y5.Elements.css
 */
Elements.setPropertyValue = Elements.css;

/**
 * Устанавливает значение CSS-свойства элемента DOM, предварительно добавив к нему суффикс 'px'.
 * @name y5.Elements.setPropertyValuePx
 * @memberOf y5.Elements
 * @function
 * @param {Element} element Элемент
 * @param {String} propname Свойство, например 'margin-left'
 * @param {Number} value Значение
 * @deprecated y5.Elements.css
 */
Elements.setPropertyValuePx = Elements.css;

// IE
if (Types.undef(document.defaultView)) {

    Elements.getStyle = function(element) {
        return element.currentStyle || element.runtimeStyle;
    };

   getPropertyValue = function(element, propname) {
        var name = Strings.camelize(propname),
            value;
        switch (name) {
            case 'opacity':
                value = 100;
                try {
                    value = element.filters['DXImageTransform.Microsoft.Alpha'].opacity;
                } catch (e) {
                    try {
                        value = element.filters('alpha').opacity;
                    } catch(e) { }
                }
                return (value / 100).toString();

            case 'float':
                name = 'styleFloat';
                break;

            // возвращаем пиксельные значения размеров
            case 'width':
            case 'height':
            case 'top':
            case 'right':
            case 'bottom':
            case 'left':
                var set = false;
                if (!element.style[name]) {
                    element.style[name] = Elements.getStyle(element)[name];
                    set = true;
                }

                value = element.style['pixel' + Strings.capitalize(name)];

                if (set) {
                    element.style[name] = null;
                }

                return value;
        }

        return Elements.getStyle(element)[name];
    };

    function cutAlphaFilter(filter){
        return filter.replace(/alpha\s*\([^\)]*\)/ig, '');
    }

    var setPropertyValueOld = setPropertyValue;

    setPropertyValue = function(element, propname, value) {
        switch (propname) {
            case 'opacity':
                var filter = getPropertyValue(element, 'filter');
                var style = element.style;
                if (value == 1) {
                    filter = cutAlphaFilter(filter);
                    if (filter) {
                        style.filter = filter;
                    } else {
                        style.removeAttribute('filter')
                    }
                    return element;
                } else if (value < 0.00001) {
                    value = 0;
                }
                // fix opacity bug
                if (!style.zoom) {
                    style.zoom = 1;
                }
                style.filter = cutAlphaFilter(filter) + 'alpha(opacity=' + (value * 100) + ')';
                break;

            default:
                setPropertyValueOld(element, propname, value);
        }

        return element;
    };
}

y5.loaded('Elements');

});