y5.require('Utils', 'Dom', 'Strings', 'Color', 'Elements', 'EventsExt', function() {

var Utils = y5.Utils,
    Dom = y5.Dom,
    Strings = y5.Strings,
    Color = y5.Color,
    Elements = y5.Elements,
    VOID = y5.VOID;

/// Fx
/////////////////////////////////////////////////////////////////////

var Fx = y5.Fx = function(element, effect, params) {
    return new FxProxy(element, effect, params);
};


/// Fx Base
/////////////////////////////////////////////////////////////////////

var FxProxy = function(element, effect, params) {
    this.delay = 0;
    this.element = element;
    this.effects = [];
    this.blend(effect, params);
};

FxProxy.prototype = {
    blend: function(effect, params) {
        params = params || {};
        params.delay = this.delay + (params.delay || Effect.prototype.defaultParams.delay);
        this.effects.push(effect(this.element, params));

        return this;
    },

    after: function(effect, params) {
        params = params || {};
        this.delay += (params.duration || Effect.prototype.defaultParams.duration);
        this.blend(effect, params);

        return this;
    },

    start: function() {
        this.delay = 0;
        this.effects.forEach(function(item) { item.start(); });
    }
};


/// Transitions
/////////////////////////////////////////////////////////////////////

var Transitions = Fx.Transitions = {
    _BACK_OVERSHOOT: 1.70158,

    none: function() {
        return 0;
    },

    fill: function() {
        return 1;
    },

    linear: function(value) {
        return value;
    },

    wobble: function(value) {
        return (-Math.cos(value * Math.PI * (9 * value)) / 2) + 0.5;
    },

    reverse: function(value) {
        return 1 - value;
    },

    random: function(value) {
        return Math.random();
    },

    back: function(value, params) {
        if (value == 0 || value == 1) {
            return value;
        }

        params = params || {}
        var overshoot = params.overshoot || Transitions._BACK_OVERSHOOT;
        return (overshoot * value + value - overshoot) * (value * value);
    },

    bounce: function(value) {
        value = 1 - value;

        var result = 0, c1 = 7.5625, c2 = 2.75;

        if (value < 1 / c2) {
            result = c1 * value * value;
        } else if (value < 2 / c2) {
            result = c1 * (value -= 1.5 / c2) * value + .75;
        } else if (value < 2.5 / c2) {
            result = c1 * (value -= 2.25 / c2) * value + .9375;
        } else {
            result = c1 * (value -= 2.625 / c2) * value + .984375;
        }

        return 1 - result;
    },

    elastic: function(value, params) {
        if (value == 0 || value == 1) {
            return value;
        }

        params = params || {}
        return Math.pow(2, 6 * --value) * Math.cos(10 * value * Math.PI * (params.amplitude || 1) / 3);
    },

    quad: function(value) {
        return value * value;
    },

    cubic: function(value) {
        return value * value * value;
    },

    pow: function(value, params) {
        params = params || {}
        return Math.pow(value, params.pow || 4);
    },

    expo: function(value) {
        return Math.pow(2, 10 * (value - 1));
    },

    sin: function(value) {
        return Math.sin(value * Math.PI / 2);
    },

    circ: function(value) {
        return 1 - Math.sqrt(1 - value * value);
    }
};

['back', 'bounce', 'elastic', 'quad', 'cubic', 'pow', 'expo', 'sin', 'circ'].forEach(
    function(name) {
        var transition = Transitions[name];

        Transitions[name + 'In'] = transition;

        Transitions[name + 'Out'] = function(value, params) {
            return 1 - transition(1 - value);
        };

        Transitions[name] = function(value, params) {
            if (value <= 0.5) {
                return transition(value * 2) / 2;
            }
            return (2 - transition(2 * (1 - value))) / 2;
        };
    }
);


/// Bezier
var Bezier = Fx.Bezier = {

    position: function(start, end, points, time) {
        var i, j, p,
            time_1 = 1 - time,
            length = points.length,
            result = new Array(length + 2);

        // copy points
        result[0] = [start[0], start[1]];
        for (i = 0; i < length; i++) {
            p = points[i];
            result[i + 1] = [p[0], p[1]];
        }
        result[i + 1] = [end[0], end[1]];
        length += 2;

        for (j = 1; j < length; j++) {
            for (i = 0; i < length - j; i++) {
                var p_i = result[i], p_k = result[i + 1];
                result[i][0] = time_1 * p_i[0] + time * p_k[0];
                result[i][1] = time_1 * p_i[1] + time * p_k[1];
            }
        }

        p = result[0];
        return [p[0], p[1]];
    }
};

/// Effect Base
/////////////////////////////////////////////////////////////////////

var UNDEF_TYPE = false,
    LENGTH_TYPE = 1,
    FLOAT_TYPE = 2,
    COLOR_TYPE = 3;

var LENGTH_RE = /left|top|width|height|size|right|bottom|spacing|indent|padding|align/i,
    FLOAT_RE = /opacity|weight/i,
    COLOR_RE = /color/i;

function testType(name) {
     if (COLOR_RE.test(name)) {
        return COLOR_TYPE;
    } else if (LENGTH_RE.test(name)) {
        return LENGTH_TYPE;
    } else if (FLOAT_RE.test(name)) {
        return FLOAT_TYPE;
    }

    return UNDEF_TYPE;
}

function getLength(element, name, value) {
    if (value == 'auto') {
        switch (name) {
            case 'width':
                return [element.scrollWidth, 'px'];

            case 'height':
                return [element.scrollHeight, 'px'];

            default:
                return [0, 'px'];
        }
    }

    if (value == 'normal') {
        switch (name) {
            case 'lineHeight':
                return [1.2, 'em'];

            default:
                return [0, 'px'];
        }
    }

    var num = parseFloat(value), type;

    try {
        type = value.match(/[+-]?\d*.?\d+(.*)/)[1];
    } catch (e) {
        type = 'px';
    }

    if (type == '') {
        type = 'px';
    }

    return [num, type];
}

/**
 * Создает базовый экземпляр класса для реализации на его основе анимационных эффектов.
 *
 * @name y5.Effect
 * @class Класс для создания анимационных эффектов. Набор базовых методов для реализации на их основе различных анимационных эффектов. Механизм создания эффектов требует указать элемент DOM, к которому будет применяться эффект, стили, которые будут изменяться, и некоторые дополнительные параметры - длительность, задержка, число повторов, обработчики состояний и др.
 * @constructor
 * @param {Element} element Элемент, к которому применяется эффект
 * @param {Object} styles Стили, которые изменяет эффект
 * @param {Object} params Параметры эффекта
 */
var Effect = y5.Effect = function(element, styles, params) {
    this.element = element;
    this.initCSS = element.style.cssText;

    // стили
    this.styles = Utils.objectCopy(
        Utils.objectCopy({}, this.defaultStyles),
        styles
    );

    // параметры
    this.params = Utils.objectCopy(
        Utils.objectCopy({}, this.defaultParams),
        params
    );

    this.forward = true;
    this.transition = this.params.transition;
    this.ticks = Math.round(this.params.duration * this.params.fps);
    if (this.ticks == 0) {
        this.ticks = 1;
    }
    this.timer = new y5.TimerObserver(this.tween, 1 / this.params.fps, false, this);

    this.init();

    if (this.params.start) {
        this.start();
    }
};

Effect.prototype = {
    /**
     * @private
     */
    init: VOID,

    /**
     * @private
     */
    defaultStyles: {
    },

    /**
     * @private
     */
    defaultParams: {
        // запускать немедленно
        start: false,

        // время выполнения
        duration: 1.0,

        // задержка перед выполнением
        delay: .0,

        // частота обновления
        fps: 40
    },

    /**
     * @private
     */
    animate: function(delay) {
        delay = typeof delay == 'number' ? delay : this.params.delay;
        Utils.setTimeout(this.timerStart, delay * 1000, this);
    },

    /**
     * Запускает анимацию.
     * @name y5.Effect.start
     * @memberOf y5.Effect
     * @function
     * @param {Number} [delay] Время, через которое анимация будет запущена
     */
    start: function(delay) {
        this.forward = true;
        this.animate(delay);
    },

    /**
     * Запускает анимацию в обратном порядке.
     * @name y5.Effect.rewind
     * @memberOf y5.Effect
     * @function
     * @param {Number} [delay] Время, через которое анимация будет запущена
     */
    rewind: function(delay) {
        this.forward = false;
        this.animate(delay);
    },

    /**
     * Прекращает анимацию.
     * @name cancel
     * @memberOf y5.Effect
     * @function
     */
    cancel: function() {
        this.timer.remove(true);

        if (this.forward) {
            this.runCallback('complete');
        } else {
            this.runCallback('completeRewind');
        }
    },

    /**
     * Сбрасывает стили элемента, т.е. возвращает их в исходное состояние (в котором они находились до начала анимации).
     * @name reset
     * @memberOf y5.Effect
     * @function
     */
    reset: function() {
        this.element.style.cssText = this.initCSS;
    },

    /**
     * @private
     */
    tween: function(timer, tick, skip) {
        this.tick = tick;
        var ticks = this.ticks;
        var forward = this.forward;

        if (!forward) {
            tick = ticks - tick;
        }

        var last = (forward ? ticks : 0) == tick;

        if (last || !skip) {
            var value = this.transition(tick / ticks, this.params);
            this.update(value, tick, ticks);
            this.runCallback('tween', value);
        }

        if (last) {
            this.cancel();
        }
    },

    /**
     * @private
     */
    timerStart: function() {
        this.timer.add();
        this.runCallback('start');
    },

    /**
     * @private
     */
    runCallback: function(name, value) {
        var func = this.params['on' + name];
        if (typeof func == 'function') {
            func.call(this, value);
        }
        y5.Notify('fx:' + name, this.element, value);
    }
};



// преобразование по умолчанию
Effect.prototype.defaultParams.transition = Transitions.cubic;


/// Morph
/////////////////////////////////////////////////////////////////////

/**
 * Эффект превращения. Преобразует стили объекта от текущего состояния до заданного.
 *
 * @class Содержит функции для создания эффекта превращения.
 * @name y5.Effect.Morph
 * @extends y5.Effect
 */
var Morph = Effect.Morph = function(element, styles, params) {
    this.morph = {
        Type: {},
        Start: {},
        End: {}
    };

    this.Effect(element, styles, params);
};

Morph.prototype = {
    init: function() {
        var styles = this.styles;
        var element = this.element;

        for (var name in styles) {
            var type = testType(name);

            this.morph.Type[name] = type;
            this.initProp(name, null,         type, 'Start');
            this.initProp(name, styles[name], type, 'End');

            // приведение единиц измерения
            switch (type) {
                case LENGTH_TYPE:
                    var start = this.morph.Start[name];
                    var startValue = start[0];
                    var startUnit = start[1];
                    var endUnit = this.morph.End[name][1];

                    if (startUnit != endUnit) {
                        var parent = element.parentNode;
                        var px = Dom.unit2px(startValue, parent, startUnit);
                        var unit = Dom.px2unit(px, parent, endUnit);
                        this.morph.Start[name] = [unit, endUnit];
                    }

                    break;
            }
        }

        // this.Effect.prototype.init.apply(this);
    },

    initProp: function(name, value, type, pos) {
        if (value === null) {
            value = Elements.css(this.element, name);
        }

        switch (type) {
            case LENGTH_TYPE:
                value = getLength(this.element, name, value);
                break;

            case FLOAT_TYPE:
                value = parseFloat(value || 0);
                break;

            case COLOR_TYPE:
                if (['rgba(0, 0, 0, 0)', 'transparent'].indexOf(value) != -1) {
                    value = '#fff';
                }
                value = new Color(value);
                break;
        }

        this.morph[pos][name] = value;
    },

    update: function(value, counter, ticks) {
        var morph = this.morph;

        for (var name in morph.Start) {
            this.updateProp(name, value, counter, ticks);
        }
    },

    updateProp: function(name, value) {
        var morph = this.morph;
        var type = morph.Type[name];
        var start = morph.Start[name];
        var end = morph.End[name];
        var styleValue = null;

        switch (type) {
            case LENGTH_TYPE:
                styleValue = (start[0] + (end[0] - start[0]) * value) + end[1];
                break;

            case FLOAT_TYPE:
                styleValue = start + (end - start) * value;
                break;

            case COLOR_TYPE:
                styleValue = this.colorMorph(value, start, end);
                break;
        }

        if (styleValue !== null) {
            Elements.css(this.element, name, styleValue);
        }
    },

    colorMorph: function(value, start, end) {
        var red = start.red + (end.red - start.red) * value;
        var green = start.green + (end.green - start.green) * value;
        var blue = start.blue + (end.blue - start.blue) * value;

        return new Color().setRGB(red, green, blue).getRGB();
    }
};

Utils.objectExtends(Morph, Effect, 'Effect');


/// Scale
/////////////////////////////////////////////////////////////////////

/**
 * Эффект увеличения.
 *
 * @class Содержит функции для создания эффекта увеличения.
 * @name y5.Effect.Scale
 * @extends y5.Effect.Morph
 */
var Scale = Effect.Scale = function(element, styles, params) {
    this.scaleStyles = [
        'width',
        'height',
        'fontSize',
        'paddingTop',
        'paddingRight',
        'paddingBottom',
        'paddingLeft',
        'borderTopWidth',
        'borderRightWidth',
        'borderBottomWidth',
        'borderLeftWidth'
    ];
    this.Morph(element, styles, params);
};

Scale.prototype = {
    /**
     * Параметры по умолчанию.
     * @memberOf y5.Effect.Scale
     * @name y5.Effect.Scale.params
     * @type Object
     *
     * @example
     * // параметры по умолчанию
     * {
     *     // коэффицент увеличения
     *     scale: 1.5
     * }
     */
    defaultParams: {
        scale: 1.5
    },

    init: function() {
        this.scaleStyles.forEach(function(name) { this.styles[name] = null; }, this);

        this.Morph.prototype.init.apply(this);
    },

    initProp: function(name, value, type, pos) {
        if (pos == 'End' && this.scaleStyles.indexOf(name) != -1) {
            var item = this.morph.Start[name];
            this.morph.End[name] = [item[0] * this.params.scale, item[1]];
        } else {
            this.Morph.prototype.initProp.call(this, name, value, type, pos);
        }
    }
};

Utils.objectExtends(Scale, Morph, 'Morph');


/// Gray
/////////////////////////////////////////////////////////////////////

/**
 * Изменяет цвета элемента от текущего до серого.
 *
 * @class Содержит функцию для плавного превращения цвета элемента в серый.
 * @name y5.Effect.Gray
 * @extends y5.Effect.Morph
 */
var Gray = Effect.Gray = function(element, styles, params) {
    this.colorStyles = ['color', 'backgroundColor', 'borderTopColor', 'borderRightColor', 'borderBottomColor', 'borderLeftColor'];
    this.Morph(element, styles, params);
};

Gray.prototype = {
    init: function() {
        this.colorStyles.forEach(function(name) { this.styles[name] = null; }, this);

        this.Morph.prototype.init.apply(this);
    },

    initProp: function(name, value, type, pos) {
        if (pos == 'End' && this.colorStyles.indexOf(name) != -1) {
            this.morph.End[name] = this.morph.Start[name].gray();
        } else {
            this.Morph.prototype.initProp.call(this, name, value, type, pos);
        }
    }
};

Utils.objectExtends(Gray, Morph, 'Morph');


/// Invert
/////////////////////////////////////////////////////////////////////

/**
 * Инвертирует цвета объекта.
 *
 * @class Содержит функции для инвертирования цветов объекта.
 * @name y5.Effect.Invert
 * @extends y5.Effect.Morph
 */
var Invert = Effect.Invert = function(element, styles, params) {
    this.Gray(element, styles, params);
};

Invert.prototype = {
    initProp: function(name, value, type, pos) {
        if (pos == 'End' && this.colorStyles.indexOf(name) != -1) {
            this.morph.End[name] = this.morph.Start[name].invert();
        } else {
            this.Morph.prototype.initProp.call(this, name, value, type, pos);
        }
    }
};

Utils.objectExtends(Invert, Gray, 'Gray');


/// Pulsate
/////////////////////////////////////////////////////////////////////

/**
 * Эффект пульсации.
 *
 * @class Содержит функции для создания эффекта пульсации.
 * @name y5.Effect.Pulsate
 * @extends y5.Effect.Morph
 */
var Pulsate = Effect.Pulsate = function(element, styles, params) {
    this.Morph(element, styles, params);
};

Pulsate.prototype = {
    /**
     * Параметры по умолчанию.
     * @memberOf y5.Effect.Pulsate
     * @name y5.Effect.Pulsate.params
     * @type Object
     *
     * @example
     * // параметры по умолчанию
     * {
     *     // число пульсов
     *     pulses: 5
     * }
     */
    defaultParams: {
        pulses: 5
    },

    init: function() {
        this.transition = function() {
            var params = this.params,
                ticks = this.ticks,
                counter = this.tick;

            ticks /= params.pulses;
            counter %= ticks;

            if (counter * 2 > ticks) {
                counter = ticks - counter;
            }

            return params.transition(2 * counter / ticks, params);
        };

        this.Morph.prototype.init.apply(this);
    }
};

Utils.objectExtends(Pulsate, Morph, 'Morph');


/// Highlight
/////////////////////////////////////////////////////////////////////

/**
 * Эффект подсветки.
 *
 * @class Содержит функции для создания эффекта подсветки.
 * @name y5.Effect.Highlight
 * @extends y5.Effect.Pulsate
 */
var Highlight = Effect.Highlight = function(element, styles, params) {
    this.Pulsate(element, styles, params);
};

Highlight.prototype = {
    /**
     * Параметры по умолчанию.
     * @memberOf y5.Effect.Highlight
     * @name y5.Effect.Highlight.params
     * @type Object
     *
     * @example
     * // параметры по умолчанию
     * {
     *     // число пульсов
     *     pulses: 1,
     *
     *     // длительность
     *     duration: 2
     * }
     */
    defaultParams: {
        pulses: 1,
        duration: 2
    },

    /**
     * Стили по умолчанию.
     * @memberOf y5.Effect.Highlight
     * @name styles
     * @type Object
     *
     * @example
     * // стили по умолчанию
     * {
     *     // цвет фона
     *     backgroundColor: '#ffcc00',
     *
     *     // цвет текста
     *     color: '#000'
     * }
     */
    defaultStyles: {
        backgroundColor: '#ffcc00',
        color: '#000'
    }
};

Utils.objectExtends(Highlight, Pulsate, 'Pulsate');


/// Move
/////////////////////////////////////////////////////////////////////

/**
 * Эффект движения.
 *
 * @class Содержит функции для создания эффекта движения.
 * @name y5.Effect.Move
 * @extends y5.Effect.Morph
 */
var Move = Effect.Move = function(element, styles, params) {
    this.Morph(element, styles, params);

    var morph = this.morph;
    var start = morph.Start;
    var end = morph.End;

    // poins for Bezier
    this.startPoint = [start.left[0], start.top[0]];
    this.endPoint = [end.left[0], end.top[0]];
    this.controlPoints = this.params.controlPoints || [];
    this.leftUnit = end.left[1];
    this.topUnit = end.top[1];
    this.currentPoint = [0, 0];
};

Move.prototype = {
    defaultStyles: {
        left: 0,
        top: 0
    },

    update: function(value, counter, ticks) {
        this.currentPoint = Bezier.position(this.startPoint, this.endPoint, this.controlPoints, value);
        this.Morph.prototype.update.call(this, value, counter, ticks);
    },

    updateProp: function(name, value, counter, ticks) {
        switch (name) {
            case 'left':
                Elements.css(this.element, 'left', this.currentPoint[0] + this.leftUnit);
                break;

            case 'top':
                Elements.css(this.element, 'top', this.currentPoint[1] + this.topUnit);
                break;

            default:
                this.Morph.prototype.updateProp.call(this, name, value, counter, ticks);
        }
    }
};

Utils.objectExtends(Move, Morph, 'Morph');


/// Shake
/////////////////////////////////////////////////////////////////////

/**
 * Эффект тряски.
 *
 * @class Содержит функции для создания эффекта тряски.
 * @name y5.Effect.Shake
 * @extends y5.Effect.Morph
 */
var Shake = Effect.Shake = function(element, styles, params) {
    this.Morph(element, styles, params);
};

Shake.prototype = {
    defaultParams: {
        offsetX: 5,
        offsetY: 0,
        fps: 15
    },

    defaultStyles: {
        left: null,
        top: null
    },

    updateProp: function(name, value, counter, ticks) {
        var params = this.params,
            morph = this.morph,
            last = counter == ticks;

        switch (name) {
            case 'left':
                params.offsetX = last ? 0 : params.offsetX * -1;
                Elements.css(this.element, 'left', morph.Start.left[0] + params.offsetX * value);
                break;

            case 'top':
                params.offsetY = last ? 0 : params.offsetY * -1;
                Elements.css(this.element, 'top', morph.Start.top[0] + params.offsetY * value);
                break;

            default:
                this.Morph.prototype.updateProp.call(this, name, value, counter, ticks);
        }
    }
};

Utils.objectExtends(Shake, Morph, 'Morph');

for (var name in Effect) {
    Fx[name] = (function(name) {
        return function(element, styles, params) {
            return new Effect[name](element, styles, params);
        }
    })(name);
}

y5.loaded('Fx');

});