/**
 * @name ObjectMove
 * @fileOverview
 * Класс, реализующий drag'n'drop.
 * @require
 */

/**
 * Класс, реализующий drag'n'drop.
 * 
 * @constructor
 * @class Класс, реализующий drag'n'drop.
 * @param {Object} object Перетаскиваемый объект
 * @param {Object} region Область, в которой перетаскивается объект
 * @param {Object} callbacks Callback-функции {start, move, up}
 * @param {Object} clickListener Обработчик клика (экземпляр y5.AEventListener)
 * 
 * @example
 * // пример перемещения ползунка в слайдере
 * new y5.ObjectMove(
 *   grab,
 *   scale,
 *   {
 *     start: onStart,
 *     move: onMove,
 *     up: onUp
 *   },
 *   y5.Events.observe('click', onClick, scale, true, this)
 * );
 * 
 * В каждый callback приходит два аргумента, по аналогии с y5.Events.observe
 */
y5.ObjectMove = function(object, region, callbacks, clickListener, context) {
    this.object = object;
    this.region = region;
    this.callbacks = callbacks;
    this.clickListener = clickListener;
    this.context = context || this;

    this.initCallbacks();

    this.moved = false;
    this.startPoint = null;
    this.clickPoint = null;
    this.oldSelectStart = null;
    this.oldMouseDown = null;

    this.aClickListener = (this.clickListener instanceof y5.AEventListener);
    this.downListener = new y5.AEventListener('mousedown', this.down, this.object, false, this);
    this.moveListener = new y5.AEventListener('mousemove', this.move, document, false, this);
    this.upListener = new y5.AEventListener('mouseup', this.up, document, false, this);

    this.start();
};

y5.ObjectMove.prototype = {
    start: function() {
        this.downListener.add();
    },

    stop: function() {
        this.downListener.remove();
        this.moveListener.remove();
        this.upListener.remove();
    },

    // Нажатие мыши
    down: function(e) {
        // check left mouse button
        if (!e.buttonL) return true;

        this.moveListener.add();
        this.upListener.add();

        this.disableEvents(e);

        // Запоминаем:
        // 1) текущую позицию объекта: startPoint
        this.startPoint = y5.Dom.getOffset(this.object);

        // 2) позицию клика мыши: clickPoint
        this.clickPoint = [e.pageX, e.pageY];

        this.callbacks.down.call(this.context, this.object, e, this.startPoint, this.clickPoint);

        return false;
    },

    // Движение мыши
    move: function(e) {
        // Новая позиция
        // startPoint + (позицияКурсора - clickPoint)
        var newPoint = [
            this.startPoint[0] + (e.pageX - this.clickPoint[0]),
            this.startPoint[1] + (e.pageY - this.clickPoint[1])
        ];

        if (!this.moved) {
            this.callbacks.start.call(this.context, this.object, e, newPoint, this.startPoint, this.clickPoint);
        }

        this.moved = true;

        this.callbacks.move.call(this.context, this.object, e, newPoint, this.startPoint, this.clickPoint);

        return false;
    },

    up: function(e) {
        this.enableEvents(e);

        this.moved = false;

        this.moveListener.remove();
        this.upListener.remove();

        this.callbacks.up.call(this.context, this.object, e, y5.Dom.getOffset(this.object));

        return false;
    },

    // init callbacks
    CALLBACKS: ['down', 'start', 'move', 'up'],

    initCallbacks: function() {
        if (typeof(this.callbacks) !== 'object') {
            this.callbacks = {};
        }

        for (var i = 0, l = this.CALLBACKS.length; i < l; i++) {
            var name = this.CALLBACKS[i];
            if (typeof(this.callbacks[name]) !== 'function') {
                this.callbacks[name] = y5.FALSE;
            }
        }
    },

    // disable/enable select text on page
    disableEvents: function(e) {
        this.oldSelectStart = document.onselectstart;
        document.onselectstart = y5.FALSE;

        this.oldMouseDown = document.onmousedown;
        document.onmousedown = y5.FALSE;

        this.old_click = this.object.onclick;
        this.object.onclick = y5.FALSE;

        if (this.aClickListener) {
            this.clickListener.remove();
        }
    },

    enableEvents: function(e) {
        document.onselectstart = this.oldSelectStart;
        this.oldSelectStart = null;

        document.onmousedown = this.oldMouseDown;
        this.oldMouseDown = null;

        var _this = this;

        function restore()
        {
            _this.object.onclick = _this.old_click;
            if (_this.aClickListener) {
                _this.clickListener.add();
            }
        }

        if (!this.moved) {
            restore();
        } else {
            window.setTimeout(restore, 100);
        }
    }
};

y5.require(['Events', 'Dom'], function() {
    y5.loaded('ObjectMove');
});