y5.require('Events', 'Strings', function() {

/**
 * Создает экземпляр класса для работы с выделенной областью заданного элемента ввода.
 * @class Класс для работы с выделенной областью элементов ввода.
 * @name y5.Range
 * @param {Element} element Элемент ввода
 *
 * @example
 * var range = new y5.Range(form.text);
 *
 * // выделяем первое слово в элементе ввода
 * range.selectPattern(/^\w+$/, 0, true);
 *
 * // переводим символы выделенного слова в верхний регистр
 * range.replace(range.extract().toUpperCase());
 */
y5.Range = function(element) {
    this.element = element;
    this.init();
};

y5.Range.prototype = {
    init: y5.VOID,

    /**
     * Создает выделение. Выделяет текст, заданный начальной и конечной позицией.
     * @name y5.Range.select
     * @memberOf y5.Range
     * @function
     * @param {Number} start Начальная позиция
     * @param {Number} [end] Конечная позиция (по умолчанию совпадает со start)
     *
     * @example
     * range.select(1, 8);
     */
    select: function(start, end) {
        this.element.setSelectionRange(start, end || start);
    },

    /**
     * Снимает выделение и ставит курсор в начало или конец выделения.
     * @name y5.Range.collapse
     * @memberOf y5.Range
     * @function
     * @param {Boolean} [toStart] Позиция курсора (true - в начало выделения, иначе в конец)
     *
     * @example
     * range.collapse();
     */
    collapse: function(toStart) {
        this.select(toStart ? this.getStart() : this.getEnd());
    },

    /**
     * Заменяет текст в выделенном фрагменте заданной строкой символов
     * @name y5.Range.replace
     * @memberOf y5.Range
     * @function
     * @param {String} snippet Текст для замены
     * @param {Boolean} [select] Выделить вставленный кусок текста
     *
     * @example
     * // выделение после вставки снимается
     * range.replace('foo');
     *
     * // выделяем вставленный текст
     * range.replace('foo', true);
     */
    replace: function(snippet, select) {
        var selectionStart = this.getStart(),
            snippetEnd = selectionStart + snippet.length,
            element = this.element,
            value = element.value;

        element.value =
            value.substring(0, selectionStart) + // before
            snippet +
            value.substring(this.getEnd(), value.length); // after

        element.focus();

        // если выделение не требуется, то начальную позицию выделения совмещаем с конечной
        if (!select) {
            selectionStart = snippetEnd;
        }

        // выделяем или помещаем курсор после вставленного фрагмента
        this.select(selectionStart, snippetEnd);
    },

    /**
     * Заменяет содержимое выделенного фрагмента заданным символом.
     * @name y5.Range.fill
     * @memberOf y5.Range
     * @function
     * @param {String} chr Символ, которым заполняется выделенная область
     *
     * @example
     * range.fill('_');
     */
    fill: function(chr) {
        this.replace(y5.Strings.repeat(chr, this.getEnd() - this.getStart()), true);
    },

    /**
     * Вычисляет выделенную область элемента ввода. Возвращает содержимое выделенного фрагмента.
     * @name y5.Range.extract
     * @memberOf y5.Range
     * @function
     * @returns {String} Выделенный фрагмент
     *
     * @example
     * // element.value = '1234567890'
     * range.select(1, 8);
     * range.extract(); // -> '2345678'
     */
    extract: function() {
        return this.element.value.slice(this.getStart(), this.getEnd());
    },

    /**
     * Производит поиск фрагмента текста по шаблону и, если параметр select равен true, выделяет найденный фрагмент. Возвращает содержимое выделенного фрагмента.
     * @name y5.Range.selectPattern
     * @memberOf y5.Range
     * @function
     * @param {RegExp} pattern Шаблон для выбора (должен включать в себя начало и конец строки)
     * @param {Number} [start] Позиция, с которой начинать поиск соответствия
     * @param {Boolean} [select] Выделить выбранный кусок текста
     * @returns {String} Выбранный фрагмент
     *
     * @example
     * // element.value = '123abc'
     * range.selectPattern(/^\d+$/, 0, true);
     * range.extract(); // -> '123'
     */
    selectPattern: function(pattern, start, select) {
        var value = this.element.value,
            length = value.length,
            selectionStart = typeof start == 'number' ? start : this.getStart();

        if (selectionStart > length) {
            selectionStart = length;
        }

        var selectionEnd = selectionStart;

        // функция для проверки фрагмента на соответствие шаблону
        function snippet() {
            var snip = value.slice(selectionStart, selectionEnd);
            return (
                pattern.test(snip) &&
                snip.lastIndexOf("\n") == -1 // для сафари ( /^\w+$/.test("a\n") == true )
            ) ? snip : null;
        }

        // движемся к началу строки
        do {
            selectionStart--;
        } while (selectionStart >= 0 && snippet());

        selectionStart++;

        // движемся к концу строки
        do {
            selectionEnd++;
        } while (selectionEnd <= length && snippet());

        selectionEnd--;

        var snip = snippet();
        if (snip) {
            if (select) {
                this.select(selectionStart, selectionEnd);
            }
            return snip;
        }

        return null;
    },

    /**
     * Возвращает символ перед курсором или выделенный фрагмент, если курсор находится внутри выделения. Для получения выделенного фрагмента используется extract.
     * @name y5.Range.charAtCaret
     * @memberOf y5.Range
     * @function
     * Если есть выделение, то вызывает extract.
     * @returns {String} Выбранный символ или фрагмент
     */
    charAtCaret: function() {
        var start = this.getStart(),
            end = this.getEnd();

        if (start == end) {
            return this.element.value.slice(start, end + 1);
        }

        return this.extract();
    },

    /**
     * Устанавливает курсор внутри элемента ввода в позицию, заданную параметром offset.
     * @name y5.Range.setCaret
     * @memberOf y5.Range
     * @function
     * @param {Number} offset Новая позиция
     */
    setCaret: function(offset) {
        this.select(offset);
    },

    /**
     * Возвращает начало выделения.
     * @name y5.Range.getStart
     * @memberOf y5.Range
     * @function
     * @returns {Number} Смещение
     */
    getStart: function() {
        return this.element.selectionStart;
    },

    /**
     * Возвращает конец выделения.
     * @name y5.Range.getEnd
     * @memberOf y5.Range
     * @function
     * @returns {Number} Смещение
     */
    getEnd: function() {
        return this.element.selectionEnd;
    },

    /**
     * Проверяет, выделен ли фрагмент в элементе ввода. Возвращает true, если выделение существует, и false, если элемент ввода не имеет выделенных фрагментов.
     * @name y5.Range.isSelected
     * @memberOf y5.Range
     * @function
     * @returns {Boolean} Выделение
     */
    isSelected: function() {
        return this.getStart() != this.getEnd();
    },

    /**
     * Устанавливает функцию, которая отслеживает изменение позиции курсора в элементе ввода и обрабатывает это событие заданным образом.
     * Функция вызывается в контексте экземпляра класса y5.Range.
     * @name y5.Range.handleCaretPosition
     * @memberOf y5.Range
     * @function
     * @param {Function} callback Callback-функция на возникновение события
     *
     * @example
     * // заполняем элемент pattern словом, на котором находится курсор
     * range.handleCaretPosition(function() {
     *     // здесь this == range
     *     y5.$('pattern').innerHTML = this.selectPattern(/^\w+$/);
     * });
     */
    handleCaretPosition: function(callback) {
        return y5.Events.observe(['click', 'keydown', 'keypress', 'keyup'], callback, this.element, true, this);
    },

    /**
     * Устанавливает функцию, которая отслеживает клик в элементе ввода и обрабатывает это событие заданным образом.
     * Функция вызывается в контексте экземпляра класса y5.Range.
     * @name y5.Range.handleClick
     * @memberOf y5.Range
     * @function
     * @param {Function} callback Callback-функция на возникновение события
     *
     * @example
     * // выделяем слово при клике
     * range.handleClick(function() {
     *     // здесь this == range
     *     this.selectPattern(/^\w+$/, null, true);
     * });
     */
    handleClick: function(callback) {
        return y5.Events.observe('click', callback, this.element, true, this);
    }
};

var RangeProto = y5.Range.prototype;

if (!window.getSelection) {

    RangeProto.init = function() {
        this.isTextarea = this.element.tagName.toLowerCase() == 'textarea';
    };

    RangeProto.getRange = function(selection) {
        var range,
            element = this.element;

        if (this.isTextarea) {
            range = selection.duplicate();
            range.moveToElementText(element);
        } else {
            range = element.createTextRange()
        }

        return range;
    };

    RangeProto.select = function(start, end) {
        var element = this.element,
            value = element.value,
            range = element.createTextRange();

        if (start < 0) {
            start = 0;
        }

        if (end > value.length) {
            end = value.length;
        }

        end = end || start;
        end -= start + value.slice(start + 1, end).split("\n").length - 1;
        start -= value.slice(0, start).split("\n").length - 1;
        range.move('character', start);
        range.moveEnd('character', end);
        range.select();
    };

    RangeProto.getStart = function() {
        var selection = document.selection.createRange(),
            range = this.getRange(selection);

        range.setEndPoint('EndToStart', selection);
        return range.text.length;
    };

    RangeProto.getEnd = function() {
        var selection = document.selection.createRange(),
            range = this.getRange(selection);

        range.setEndPoint('EndToStart', selection);
        return range.text.length + selection.text.length;
    };

}

y5.loaded('Range');

});