y5.require('Utils', 'Dom', 'Styles', 'Template', function() {

/**
 * @class Модуль для организации механизма управления виджетами — примитивами графического интерфейса пользователя (виджет-менеджер).
 * Виджеты:
 *    Dialog - модальные/немодальные диалоговые окна, окна подтверждения (аналог window.confirm),
 *    Tooltip - подсказки, появляющиеся при mouseover и пропадающие при mouseout или любом действии мышью/клавиатурой,
 *    Popup - попапы,
 *    Menu - меню,
 *    Button - кнопки-переключатели (например, ColorPicker), когда поведение кнопки влияет на другой виджет
 * @name y5.Widget
 * @static
 * @example
 *
 * // модальное диалоговое окно типа standard при клике на кнопку button
 *
 * // содержимое диалога
 * var dialogContent = y5.$('widget_photoLoader');
 *
 * // устанавливаем callback на кнопки диалога
 * y5.Events.observe('y5:ok', function() { alert('Нажата кнопка "Ok"'); }, dialogContent, true);
 * y5.Events.observe('y5:cancel', function() { alert('Нажата кнопка "Отмена"'); }, dialogContent, true);
 *
 * // создаем диалог по клику на кнопку button
 * y5.Events.observe('click', function() { y5.Widget.make(dialogContent, 'Widget.Dialog.Standard', {modal: true}); }, button, true);
 */
y5.Widget = {

    /* список виджетов */
    widgets: {},
    widgetsQuery: {},

    /**
     * Cоздает виджет — инициализирует и показывает на странице.
     * @name y5.Widget.make
     * @memberOf y5.Widget
     * @function
     * @param {Object} object Ключ виджета: уникальная строка-идентификатор или элемент DOM
     * @param {String} type Тип виджета, например, 'Widget.Dialog.Standard'
     * @param {Object} params Опции виджета, например, модальность диалога {modal: true}
     */
    make: function(obj, type, params) {
        var id = this.getId(obj);
        var widget = this.getById(id);

        if (widget) {
            widget.make(params);
        } else {
            if (this.widgetsQuery[id]) {
                return obj;
            }
            this.widgetsQuery[id] = true;

            var content;
            if (y5.Types.element(obj)) {
                // содержимое окна, удаляем из DOM
                content = y5.Dom.getChildren(obj);
                y5.Dom.clearNode(obj);
            }
            this.add(obj, content, type, params);
        }

        return obj;
    },

    /**
     * Уничтожает виджет — убирает со страницы и освобождает память.
     * @name y5.Widget.kill
     * @memberOf y5.Widget
     * @function
     * @param {Object} object Ключ виджета: уникальная строка-идентификатор или элемент DOM
     */
    kill: function(obj) {
        var widget = this.get(obj);
        if (widget) {
            widget.kill();
        } else {
            y5.Console.warn('Unknown widget: ' + obj, ['y5.Widget']);
        }
    },

    /**
     * Возвращает виджет по его ключу (уникальной строке-идентификатору).
     * @name y5.Widget.get
     * @memberOf y5.Widget
     * @function
     * @param {Object} object Ключ виджета: уникальная строка-идентификатор или элемент DOM
     */
    get: function(obj) {
        return this.getById(this.getId(obj));
    },

    /**
     * Возвращает уникальный ID виджета.
     * @param {Object} object Ключ виджета: уникальная строка-идентификатор или элемент DOM
     * @private
     */
    getId: function(obj) {
        return y5.Types.string(obj) ? obj : y5.Utils.getUniqueId(obj);
    },

    /**
     * Возвращает виджет по уникальному ID виджета.
     * @param {String} id ID виджета
     * @private
     */
    getById: function(id) {
        return this.widgets[id] || null;
    },

    /**
     * Добавляет виджет в список виджетов.
     * @param {Object} DOM-узел
     * @param {String} Тип виджета
     * @param {Object} Опции виджета
     * @private
     */
    add: function(obj, content, type, params) {
        var id = this.getId(obj),
            _this = this;

        y5.require(type, function() {
            var widget = new (y5.moduleObject(type))(obj, content, params);
            y5.GC.collect(widget);
            _this.widgets[id] = widget;
        });
    }
};

y5.Widget.Templates = {
    templates: {},

    add: function(module, name, template) {
        if (!this.templates[module]) {
            this.templates[module] = {};
        }

        this.templates[module][name] = template;
    },

    get: function(module, name) {
        try {
            return this.templates[module][name];
        } catch (e) {
            return null;
        }
    }
};

y5.Widget.Template = function(module, parent, name) {
    name = name || 'default';

    this.frame = '';
    this.body = '';
    this.header = '';
    this.footer = '';

    if (parent) {
        parent = y5.Widget.Templates.get(parent, name);

        this.frame = parent.frame;
        this.body = parent.body;
        this.header = parent.header;
        this.footer = parent.footer;
    }

    y5.Widget.Templates.add(module, name, this);
};

y5.Widget.Template.prototype = {

    loadCSS: function(href) {
        return y5.Styles.createStyle(href);
    },

    loadCSSModule: function(module) {
        return y5.Styles.loadModule(module);
    },

    setFrame: function(html) {
        this.setPart(html, 'frame');
    },

    setBody: function(html) {
        this.setPart(html, 'body');
    },

    setHeader: function(html) {
        this.setPart(html, 'header');
    },

    setFooter: function(html) {
        this.setPart(html, 'footer');
    },

    setPart: function(html, type) {
        if (y5.Types.element(html)) {
            html = html.innerHTML;
        }
        this[type] = html;
    },

    getHTML: function() {
        return y5.T(this.frame, {content: this.header + this.body + this.footer});
    }
};

y5.loaded('Widget');

});