y5.require(['Utils', 'Classes', 'Events', 'Elements', 'ShortCuts', 'Widget.Popup', 'Dom'], function() {
var Dom = y5.Dom,
    Utils = y5.Utils,
    Classes = y5.Classes,
    Observer = y5.Observer,
    ShortCut = y5.ShortCut,
    Elements = y5.Elements,
    WidgetWindowName = 'Widget.Menu',
    AEventListener = y5.AEventListener;

/**
 * y5.Widget.Menu
 */
y5.Widget.Menu = function(element, content, params) {
    this.initMenu(element, [], params);
};

y5.Widget.Menu.prototype = {
    className: 'y5-w-menu',
    classNameItem: 'y5-w-menu-item',

    /**
     * Опции по умолчанию
     */
    defaultParams: {
        saveState: true
    },

    toString: function() {
        return WidgetWindowName;
    },

    initMenu: function(element, content, params) {
        this.uid = Utils.generateUniqueId();
        this.ids = {};
        this.elements = {};
        this.clear();

        this.MenuBase(element, content, params);
    },

    make: function(params) {
        this.MenuBase.prototype.make.apply(this, [params]);
        this.refresh();
    },

    clear: function() {
        this.items = [];
        this.content = [];
        this.counter = 0;
        this.activeItem = null;
        this.activeLast = null;

        for (var id in this.elements) {
            delete this.elements[id];
        }

        for (var id in this.ids) {
            delete this.ids[id];
        }

        if (this.eventsOver) {
            this.eventsOver.forEach(function(evt) { evt.cleanup() });
        }
        this.eventsOver = [];

        if (this.container) {
            Dom.clearNode(this.container);
        }
    },

    roll: function(direction) {
    },

    refresh: function() {
        // деактивируем последний активный
        if (this.activeLast) {
            var element = this.getElement(this.activeLast);
            if (element) {
                Classes.remove(element, 'y5-w-menu-item-active');
            }
        }

        // активируем текущий
        if (this.activeItem) {
            var element = this.getElement(this.activeItem);
            Classes.add(element, 'y5-w-menu-item-active');
        }
    },

    getUid: function() {
        return this.uid + '_' + this.counter++;
    },

    setActiveItem: function(id) {
        this.activeLast = this.activeItem;

        var item = this.getItem(id);
        if (!item || item.disabled) {
            this.activeItem = this.getNextId(id);
        } else {
            this.activeItem = id;
        }
    },

    /*
        Navigation
    */
    getFirstId: function() {
        for (var i = 0, l = this.items.length; i < l; i++) {
            if (!this.items[i].disabled) {
                return this.items[i].id;
            }
        }

        return null;
    },

    getLastId: function() {
        for (var i = this.items.length - 1; i >= 0; i--) {
            if (!this.items[i].disabled) {
                return this.items[i].id;
            }
        }

        return null;
    },

    getPrevId: function(id) {
        var i, found = false;

        for (i = this.items.length - 1; i >= 0; i--) {
            if (found) {
                if (!this.items[i].disabled) {
                    return this.items[i].id;
                }
                continue;
            }
            if (this.items[i].id == id) {
                found = true;
            }
        }

        if (!found) {
            return null;
        }

        for (i = this.items.length - 1; i >= 0; i--) {
            if (!this.items[i].disabled) {
                return this.items[i].id;
            }
        }

        return null;
    },

    getNextId: function(id) {
        if (id == null) {
            return this.getFirstId();
        }

        var i, l, found = false;

        for (i = 0, l = this.items.length; i < l; i++) {
            if (found) {
                if (!this.items[i].disabled) {
                    return this.items[i].id;
                }
                continue;
            }
            if (this.items[i].id == id) {
                found = true;
            }
        }

        if (!found) {
            return null;
        }

        for (i = 0, l = this.items.length; i < l; i++) {
            if (!this.items[i].disabled) {
                return this.items[i].id;
            }
        }

        return null;
    },

    getItem: function(id) {
        return this.items[this.elements[id]];
    },

    getElement: function(id) {
        var item = this.getItem(id);
        if (item) {
            return y5.$(item.uid);
        }
        return null;
    },

    /*
        Create
    */
    newItem: function(data) {
        // уникальный id
        var uid = this.getUid();
        data.uid = uid;

        // id
        var id = data.id || uid;
        data.id = id;

        if (this.elements[id]) {
            return null;
        }

        // элемент
        var block = Elements.create('div', {id: uid, className: this.classNameItem});

        // собственный класс элемента меню
        if (data.className) {
            Classes.add(block, data.className);
        }

        // ссылка
        var link = Elements.create('a', {href: data.href || ''});

        // содержимое ссылки
        if (data.html) {
            link.innerHTML = data.html;
        } else {
            link.appendChild(document.createTextNode(data.label));
        }

        // добавляем ссылку в блок
        block.appendChild(link);

        // сохраняем данные
        var length = this.items.push(data);

        this.elements[id] = length - 1;
        this.ids[uid] = id;

        // disable
        if (data.disabled) {
            Classes.add(block, 'y5-w-menu-item-disabled');
        }

        return block;
    },

    newSeparator: function(id) {
        var uid = this.getUid();
        var block = Elements.create('div', {id: uid, className: 'y5-w-menu-separator'});

        this.elements[id || uid] = 0;

        return block;
    },

    setItemContent: function(block, over) {
        if (block) {
            var node = this.container.appendChild(block);
            this.content.push(node);

            // over event
            if (over) {
                this.eventsOver.push(
                    new AEventListener('mouseover', function(e) {
                        e.preventDefault();
                        this.selectItem(this.ids[block.id]);
                    }, node, true, this)
                );
            }
        }
    },

    /*
        Actions
    */
    setItems: function(data) {
        this.clear();

        data.map(function(item) { return this.newItem(item) }, this)
            .forEach(function(item) { this.setItemContent(item, true) }, this);
    },

    addItem: function(data) {
        this.setItemContent(this.newItem(data), true);
    },

    addSeparator: function(id) {
        this.setItemContent(this.newSeparator(id));
    },

    actionItem: function(id, action) {
        for (var i = 0, l = this.items.length; i < l; i++) {
            var item = this.items[i];
            if (item.id == id) {
                action.apply(this, [item, i]);
                break;
            }
        }
        this.refresh();
    },

    removeItem: function(id) {
        function action(item, i) {
            // если удаляемый элемент сам, то переводим
            // фокус на следующий за ним
            if (this.activeItem == item.id) {
                this.setActiveItem(this.getNextId(item.id));
            }

            this.items.splice(i, 1);
            this.content.splice(i, 1);

            for (var id in this.elements) {
                if (this.elements[id] > i) {
                    this.elements[id]--;
                }
            }
        }

        this.container.removeChild(this.getElement(id));

        var uid = this.elements[id];
        delete this.ids[uid];
        delete this.elements[id];

        this.actionItem(id, action);
    },

    disableItem: function(id) {
        function action(item) {
            // если отключаемый элемент сам, то переводим
            // фокус на следующий за ним
            if (this.activeItem == item.id) {
                this.setActiveItem(this.getNextId(item.id));
            }

            Classes.add(this.getElement(item.id), 'y5-w-menu-item-disabled');
            item.disabled = true;
        }

        this.actionItem(id, action);
    },

    enableItem: function(id) {
        function action(item) {
            Classes.remove(this.getElement(item.id), 'y5-w-menu-item-disabled');
            item.disabled = false;
        }

        this.actionItem(id, action);
    },

    selectItem: function(id) {
        this.setActiveItem(id);
        this.refresh();
    },

    selectFirstItem: function() {
        this.selectItem(this.getFirstId());
    },

    /*
        Callbacks
    */
    setCallBacksOnce: function() {
        ['addItem', 'addSeparator', 'removeItem', 'disableItem', 'enableItem', 'selectItem', 'setItems', 'selectFirstItem', 'clear'].forEach(
            function(n) {
                this.callbacks[n] = new Observer('y5:' + n, this[n], this.element, true, this);
            },
            this
        );

        this.Window.prototype.setCallBacksOnce.apply(this);
    },

    /*
        Events
    */
    setEventsOnce: function() {
        this.events.click = new AEventListener('click', this.clickItem, this.window, true, this);

        this.MenuBase.prototype.setEventsOnce.apply(this);
    },

    clickItem: function(e) {
        this.goItem(e, this.findIdByElement(e.target));
    },

    goItem: function(e, id) {
        e.stopPropagation();
        e.preventDefault();

        var item = this.getItem(id);
        if (item) {
            var listener = item.listener;
            if (listener) {
                if (listener(this, e)) {
                    this.kill();
                }
            } else {
                this.kill();
                window.location.href = item.href;
            }
        }
    },

    findIdByElement: function(element) {
        var parent = Dom.getAncestorOrSelf(element, '*', this.classNameItem);
        if (parent) {
            return this.ids[parent.id];
        }
        return null;
    },

    /*
        Shortcuts
    */
    setShortcutsOnce: function() {
        var container = document,
            params = {checkTarget: false, context: this};

        this.shortcuts.enter = ShortCut.down('enter', this.keyItemEnter, container, params);
        this.shortcuts.home = ShortCut.down('home', this.keyItemHome, container, params);
        this.shortcuts.end = ShortCut.down('end', this.keyItemEnd, container, params);
        this.shortcuts.down = ShortCut.press(['down', 'tab'], this.keyItemDown, container, params);
        this.shortcuts.up = ShortCut.press(['up', 'shift+tab'], this.keyItemUp, container, params);

        this.MenuBase.prototype.setShortcutsOnce.apply(this);
    },

    keyItem: function(e, id) {
        e.preventDefault();
        e.stopPropagation();

        this.selectItem(id);
    },

    keyItemEnter: function(e) {
        this.goItem(e, this.activeItem);
    },

    keyItemDown: function(e) {
        this.keyItem(e, this.getNextId(this.activeItem));
    },

    keyItemUp: function(e) {
        this.keyItem(e, this.getPrevId(this.activeItem));
    },

    keyItemHome: function(e) {
        try {
            this.keyItem(e, this.getFirstId());
        } catch (e) {}
    },

    keyItemEnd: function(e) {
        try {
            this.keyItem(e, this.getLastId());
        } catch (e) {}
    }
};

Utils.objectExtends(y5.Widget.Menu, y5.Widget.Popup, 'MenuBase');

if (!y5.Widget.Templates.get(WidgetWindowName, 'default')) {
    var template = new y5.Widget.Template(WidgetWindowName, 'Widget.Window', 'default');
    template.loadCSSModule(WidgetWindowName);
}

y5.loaded(WidgetWindowName);

});
