

include('../models.js');
include('../observable.js');
include('../number.js');
include('../../direct/validators.js');

/**
 * Абстрактный класс для создания модели
 */
(function(){


    var DEFAULT = {
        precision: 2
    };

    common.models.abstractModel = $.extend({}, common.Observable, new function() {
        var
            getters = {
                'string': function(proto, name) {
                    return proto.strings[name];
                },
                'flag': function(proto, name) {
                    return proto.flags[name];
                },
                'number': function(proto, name, view) {
                    return proto.numbers[view ? 'views' : 'values'][name];
                },
                'CONST': function(proto, name) {
                    return proto.consts[name];
                }
            },
            setters = {
                'string': function(proto, name, value) {
                    var defaultValue = (proto.fieldSchemes[name]['default'] !== undefined) ? proto.fieldSchemes[name]['default'] : '';
                    proto.strings[name] = value || defaultValue;
                    return proto;
                },
                'flag': function(proto, name, value) {

                    var defaultValue = (proto.fieldSchemes[name]['default'] !== undefined) ? proto.fieldSchemes[name]['default'] : 0;
                    //строка '0' должна восприниматься как false

                    proto.flags[name] =  value && value != 0 ? !!value : defaultValue;
                    return proto;
                },
                'number': function(proto, name, value) {
                    if (value) {

                         value = clearNumber(proto, name, value);
                         proto.numbers.values[name] =  value;
                         setNumberView(proto, name, value);
                    } else {
                        var defaultValue = (proto.fieldSchemes[name]['default'] !== undefined) ? proto.fieldSchemes[name]['default'] : 0,
                            defaultViewValue = (proto.fieldSchemes[name]['defaultView'] !== undefined) ? proto.fieldSchemes[name]['defaultView'] : 0;
                        proto.numbers.values[name] =  defaultValue;
                        proto.numbers.raw[name] =  defaultValue;
                        proto.numbers.views[name] =  defaultViewValue;
                    }

                    return proto;
                },
                'CONST': function(proto, name, value) {
                    if ($.isFunction(proto.fieldSchemes[name].preprocess)) {
                        proto.consts[name] = proto.fieldSchemes[name].preprocess.call(this, value);
                    } else {
                        proto.consts[name] = value;
                    }
                }
            };



        function clearNumber(proto, name, value) {
            var failValue = proto.fieldSchemes[name].fail !== undefined  ? proto.fieldSchemes[name].fail : value;
            value = common.number.clear(value, {fail: failValue});
            proto.numbers.fails[name] = isNaN(value);
            if (!proto.numbers.fails[name]) {
                var precision = proto.fieldSchemes[name].precision !== undefined ? proto.fieldSchemes[name].precision : DEFAULT.precision;
                proto.numbers.raw[name] = value;
                value = common.number.round(value, {precision: precision});
            } else {
                proto.numbers.raw[name] = value;
            }
            return value;
        }

        function setNumberView(proto, name, value) {
            if (!proto.numbers.fails[name]) {
                var precision = proto.fieldSchemes[name].precision !== undefined ? proto.fieldSchemes[name].precision : DEFAULT.precision;
                proto.numbers.views[name] = common.number.format(value, {precision: precision})
            } else {
                proto.numbers.views[name] = proto.fieldSchemes[name].failView !== undefined  ? proto.fieldSchemes[name].failView : value;
            }
        }

        //выдаёт предопределённые на момент инициализации значения. К дефолтным значениям отнощения не имеет
        function getStartValues() {
            var values = {};
            for (var name in this.fieldSchemes) {
                if (!this.fieldSchemes.hasOwnProperty(name)) continue;
                if (this.fieldSchemes[name]['value']) {
                    values[name] = this.fieldSchemes[name]['value'];
                }
            }
            return values;
        }



        function setSchemes(modelSchemes) {
            this.globalTrigger =  modelSchemes.globalTrigger;
            this.fieldSchemes = modelSchemes.fields;
            this.controllersNames = modelSchemes.controllers;
            this.controllers = {};
            this.validateSchemes = modelSchemes.validateRules;
            this.errorsMessages = modelSchemes.errorsMessages;
            this.warningsMessages = modelSchemes.warningsMessages;
        }

        function setController(name, controller) {
            if ($.inArray(name, this.controllersNames) == -1) { return false }
            this.controllers[name] = controller;

        }

        function getController(name) {
            return this.controllers[name];
        }

        function set(field, value) {
            var scheme = this.fieldSchemes[field];
            if (!scheme) {
                return this;
            }
            setters[scheme.type](this, field, value);
            if (this.fieldSchemes[field].children) {
                $.each(this.fieldSchemes[field].children, $.proxy(function(i, name) {
                    set.call(this, name, this.fieldSchemes[name].calcValue.call(this))
                }, this))
            }
            return this;
        }

        //deprecated - перевести на что-то более внятное
        function getAllErrorsMessages() {
            var messages = $.isFunction(this.errorsMessages) ? this.errorsMessages() : this.errorsMessages,
                errorMessage = '',
                errors = this.getErrors();


            if (errors) {
                for (var fieldName in errors) {
                    for (var i = 0; i < errors[fieldName].length; i++) {
                        errorMessage += messages[fieldName][errors[fieldName][i]] + '\n';
                    }
                }
            }
            return errorMessage;
        }

        function getWarningsForField(params, type, fieldId, codes) {
            return this.warningsConstructor.getWarningsForField(params, type, fieldId, codes);
        }

        function clearErrors(fieldId) {
            this.validator.clearErrors(fieldId)

        }

        function clearWarnings(fieldId) {
            this.validator.clearWarnings(fieldId)

        }


        function getWarnings() {
            return this.validator.getWarnings();
        }

        function getErrors() {
            return this.validator.getErrors();
        }

        function get(field, view) {
            var scheme = this.fieldSchemes[field];
            if (!scheme) {
                return this;
            }
            return getters[scheme.type](this, field, view);
        }



        function raw(field) {
            var scheme = this.fieldSchemes[field];
            if (!scheme) {
                return this;
            }

            return scheme.type == 'number' ? this.numbers.raw[field] : getters[scheme.type](this, field);
        }

        function has(field) {
            return this.fieldSchemes.hasOwnProperty(field);
        }

        function reset(data, source) {
            var result = {};
            for (var name in this.fieldSchemes) {
                if (!this.fieldSchemes.hasOwnProperty(name)) continue;
                this.set(name, data[name]);
                result[name] = data[name];
            }
            this.trigger('reset', {source: source, changed: result, model: this})
        }

        function triggerChange(e) {
            $.each(e.changes, $.proxy(function(name, changes){
                if (!this.fieldSchemes.hasOwnProperty(name)) return;
                this.trigger('change.' + name, e);
            }, this));

            if (this.globalTrigger) {
                this.trigger('change.global.' + this.globalTrigger, e);
            }
        }



        function update(data, source, noevent) {
            var changes = {}, fields = [];
            for (var name in data) {
                if (!data.hasOwnProperty(name)) continue;
                if (!this.fieldSchemes[name]) continue;

                var prevValue = this.raw(name),
                    value = data[name];

                if (prevValue !== value) {
                    changes[name] = [prevValue, value];
                    fields.push(name);
                    this.set(name, value);
                }
            }

            if (!noevent) {
                triggerChange.call(this, {changes: changes, fields: fields, source: source, model: this});
            }
            return this;
        }

        function isDataChanged(stateData) {
            for (var name in this.fieldSchemes) {
                if (!this.fieldSchemes.hasOwnProperty(name)) continue;
                if (stateData[name] != this.get(name)) {
                    return true;
                }
            }
            return false;
        }

        function memento(clearUndefined) {
            var state = {}, value;
            for (var name in this.fieldSchemes) {
                if (!this.fieldSchemes.hasOwnProperty(name)) continue;
                value = this.get(name);
                if (!clearUndefined || value !== undefined) {
                    state[name] = value;
                }
            }
            return state;
        }


        function getUniqId() {
            return this.uniqId;
        }

        function init(data, schemes) {
            this.uniqId = $.identify();
            this.validator = BEM.create('i-model-validator', { model: this });
            this.numbers = {};
            this.numbers.values = {};
            this.numbers.views = {};
            this.numbers.fails = {};
            this.numbers.raw = {};
            this.strings = {};
            this.flags = {};
            this.consts = {};
            setSchemes.call(this, schemes);

            resetWithDefaults.call(this, data);
        }

        function resetWithDefaults(data) {
            reset.call(this, $.extend(getStartValues.call(this), data));
        }

        function validate() {
            return this.validator.validate();
        }

        function isEmpty(fieldId) {
            return direct.utils.isEmpty(this.get(fieldId))
        }

        function hasErrors(fieldId) {
            return this.validator.hasErrors(fieldId);

        }

        function getElementName(name) {
            return name.replace(/_/g, "-");
        }

        function getFieldName(elementName) {
            return elementName.replace(/-/g, "_");
        }

        function hasWarnings(fieldId) {
            return this.validator.hasWarnings(fieldId)

        }

        //внешнее АПИ
        return {
            getAllErrorsMessages: getAllErrorsMessages,
            set: set,
            get: get,
            init: init,
            getWarningsForField: getWarningsForField,
            validate: validate,
            getUniqId: getUniqId,
            update: update,
            memento: memento,
            reset: reset,
            resetWithDefaults: resetWithDefaults,
            isDataChanged: isDataChanged,
            clearErrors: clearErrors,
            clearWarnings: clearWarnings,
            hasErrors: hasErrors,
            getElementName: getElementName,
            hasWarnings: hasWarnings,

            getWarnings: getWarnings,
            getErrors: getErrors,
            raw: raw,
            has: has,
            setController: setController,
            getController: getController,
            isEmpty: isEmpty

        };

    });
})();
