include('../direct.js');
include('../direct/utils.js');

direct.autobroker = new function() {

    /**
     * Функция вычисляющая позицию и прочее, аналог перловой функции calc_price из AutoBroker.pm
     *
     * @param {number} price 0.XX в долларах
     * @param {Array} guarantee '10000,20000,30000' в 0000 центах, отсортированно
     * @param {Array} premium '40000,50000,60000' в 0000 центах
     * @param {string} larr '1000:1000,2000:2000|1000,2000 в 0000 центах
     * @param min_price
     * @param {number} camp_rest 0.XX остаток средств на кампании
     * @param {number} day_budget 0.XX сумма дневного ограничения бюджета (0, если ограничение не установлено)
     * @param spent_today
     * @param {Object} strategy
     * @param autobudget
     * @param autobudget_bid
     * @param timetarget_coef
     * @returns {
     *     {
     *         price: number,
     *         coverage: (*|number),
     *         truncated: number,
     *         place_name: string,
     *         place_name_without_coef: (*|string)
     *     }
     * }
     */
    this.calcPrice = function(price, guarantee, premium, larr, min_price, camp_rest, day_budget, spent_today, strategy, autobudget, autobudget_bid, timetarget_coef) {
        var data = {
                price: price,
                guarantee: guarantee,
                premium: premium,
                larr: larr,
                min_price: min_price,
                camp_rest: camp_rest,
                day_budget: day_budget,
                spent_today: spent_today,
                strategy: strategy,
                autobudget: autobudget,
                autobudget_bid: autobudget_bid,
                timetarget_coef: timetarget_coef
            },
            currency = campCurrency,
            MIN_PRICE = get_currency(currency).MIN_PRICE,
            PRICE_PLACES = window.PRICE_PLACES,
            ENTRY_PLACES = window.ENTRY_PLACES,
            PLACE_NAMES = Object.keys(PRICE_PLACES).reduce(function(obj, key) {
                    obj[PRICE_PLACES[key]] = key;

                    return obj;
                }, {}),
            larrParts = (data.larr + '').split('|'), // dima117a@todo: DIRECT-47264: Схемы для ajax-ответов, в которых приходит hash
            price = data.price,
            places = {
                // значения bid_price amnesty_price в массивах premium и guarantee приходят в виде val * 1e6
                // для точности вычислений
                premium: data.premium.map(function(item) {
                    return {
                        bid_price: +item.bid_price,
                        amnesty_price: +item.amnesty_price
                    };
                }),
                guarantee: data.guarantee.map(function(item) {
                    return {
                        bid_price: +item.bid_price,
                        amnesty_price: +item.amnesty_price
                    };
                }),
            },
            premiumEntryPrices = getPricesByPlaceName(places, PLACE_NAMES[ENTRY_PLACES.PREMIUM]),
            guaranteeEntryPrices = getPricesByPlaceName(places, PLACE_NAMES[ENTRY_PLACES.GUARANTEE]),
            bottom = larrParts[0].split(',').reduce(function(obj, larr) {
                var parts = larr.split(':'),
                    index = obj.prices.length;

                parts[1] && (obj.probs[index] = +parts[1]);

                obj.prices[index] = +parts[0];

                return obj;
            }, { prices: [], probs: [] }),
            minPrice = Math.floor(1e6 * (data.min_price || 0) + 0.5),
            campRest = Math.floor(1e6 * (data.camp_rest || 0) + 0.5),
            dayBudget = Math.floor(1e6 * data.day_budget + 0.5),
            spentToday = Math.floor(1e6 * data.spent_today + 0.5),
            timeTargetCoef = data.timetarget_coef,
            strategy = data.strategy || { name: '', search: { name: 'default' }, net: { name: 'default' } },
            searchStrategy = strategy.search,
            brokerPriceZero = { bid_price: 0, amnesty_price: 0 },
            effMinPrice = Math.min(premiumEntryPrices.bid_price, Math.max(1e6 * MIN_PRICE, bottom.prices[0] || 0)),
            autobudget = data.autobudget,
            autobudgetBid = Math.floor(1e6 * (data.autobudget_bid || 0) + 0.5),
            truncated = 0,
            coverage = 0,
            dayBudgetRest,
            effPrice,
            brokerPrice,
            place,
            possiblePrices,
            availablePrices,
            priceDataWithoutCoef;

        if (dayBudget > 0) {
            dayBudgetRest = dayBudget - spentToday;

            if (dayBudgetRest > 0) {
                // для кампаний совсем без денег с включённым дневным бюджетом рассчитываем ставки исходя из того, что денег будет больше, чем сумма дневного бюджета
                // с появлением денег начнём рассчитывать из min(денги, дневной бюджет)
                campRest = campRest > 0 ? Math.min(campRest, dayBudgetRest) : dayBudgetRest;
            } else {
                // уже перетратили дневной бюджет
                campRest = 0;
            }
        }

        if (autobudget != 'Yes' && 0 < timeTargetCoef && timeTargetCoef < 100) {
            // расчёт без учёта параметра timetarget_coef
            priceDataWithoutCoef = this.calcPrice(
                data.price,
                data.guarantee,
                data.premium,
                data.larr,
                data.min_price,
                data.camp_rest,
                data.day_budget,
                data.spent_today,
                data.strategy,
                data.autobudget,
                data.autobudget_bid);

            price *= timeTargetCoef / 100;
        }

        // цена не может быть меньше константы MIN_PRICE
        price = Math.floor(1e6 * Math.max(price, MIN_PRICE) + 0.5);

        // при автобюджете с указанной максимальной ставкой цена не может быть выше этой ставки
        autobudget == 'Yes' && autobudgetBid && (price = Math.min(price, autobudgetBid));

        effPrice = campRest > 0 ? Math.min(campRest, price) : price;

        availablePrices = []
            .concat(
                // в стратегии "показывать под результатами поиска" запретить рассматривать СР
                searchStrategy.name !== 'no_premium' ?
                    [
                        // 1-ое место в спецразмещение
                        { price: getPricesByPlaceName(places, 'PREMIUM1'), place: PRICE_PLACES.PREMIUM1 },
                        // 2-ое место в спецразмещение
                        // @todo coffeeich kabzon убрать проверку параметра при открытии на всех
                        window.showVCGAuction ? { price: getPricesByPlaceName(places, 'PREMIUM2'), place: PRICE_PLACES.PREMIUM2 } : null,
                        // вход в спецразмещение
                        { price: premiumEntryPrices, place: ENTRY_PLACES.PREMIUM }
                    ] :
                    [],
                // 1-ое место в гарантированных показах
                { price: getPricesByPlaceName(places, 'GUARANTEE1'), place: PRICE_PLACES.GUARANTEE1 },
                // вход в гарантированные показы
                { price: guaranteeEntryPrices, place: ENTRY_PLACES.GUARANTEE })
            // @todo coffeeich kabzon убрать фильтрацию при открытии на всех
            .filter(function(item) { return !!item; });

        // вычисляем цену
        // стратегия: наивысшая доступная позиция
        availablePrices.some(function(item) {
            var bidPrice = item.price.bid_price;

            if (effPrice < bidPrice && bidPrice <= price) {
                truncated = 1;
            } else if (bidPrice <= effPrice) {
                brokerPrice = item.price;
                coverage = 1;

                return true;
            }

            return false;
        });

        if (brokerPrice && brokerPrice.bid_price) {
            if (searchStrategy.name == 'min_price') {
                // вход в спецразмещение
                if (/^(premium|both)$/.test(searchStrategy.place) && premiumEntryPrices.bid_price <= brokerPrice.bid_price) {
                    brokerPrice = premiumEntryPrices;

                    truncated = 0;

                    // т.к. в этом случае положение, куда попадает баннер, считается
                    // не по принципу "максимум возможного для выбранной ставки",
                    // а иначе, мы сразу определяем это положение здесь, а потом
                    // уже считать его не будем.
                    place = PRICE_PLACES.PREMIUM3;
                }

                // вход в гарантированные показы
                if (/^(both)$/.test(searchStrategy.place) && guaranteeEntryPrices.bid_price <= brokerPrice.bid_price && brokerPrice.bid_price < premiumEntryPrices.bid_price) {
                    brokerPrice = guaranteeEntryPrices;

                    truncated = 0;

                    // см. комментарий выше
                    place = PRICE_PLACES.GUARANTEE4;
                }
            }

            if (searchStrategy.name == 'no_premium') {
                possiblePrices = [getPricesByPlaceName(places, 'GUARANTEE1'), guaranteeEntryPrices].filter(function(item) { return item.bid_price <= price; });

                if (searchStrategy.place == 'highest_place') {
                    brokerPrice = possiblePrices[0];
                } else if (searchStrategy.place == 'min_price') {
                    brokerPrice = possiblePrices[possiblePrices.length - 1];
                }

                if (brokerPrice && brokerPrice.bid_price > effPrice) {
                    brokerPrice = brokerPriceZero;
                } else {
                    if (searchStrategy.place == 'min_price') {
                        place = PRICE_PLACES.GUARANTEE4;
                    }

                    truncated = 0;
                }
            }
        }

        // если не нашлось место в гарантии - считаем покрытие
        if (!(brokerPrice && brokerPrice.bid_price)) {
            coverage = this.calcCoverage(effPrice, bottom.prices, bottom.probs) / 1e6;

            brokerPrice = brokerPriceZero;

            if (coverage) {
                truncated = +(price > effPrice);

                // вычисляем цену автоброкера
                brokerPrice.bid_price = brokerPrice.amnesty_price = larrParts[1] && larrParts[1]
                        .split(',')
                        .map(Number)
                        .filter(function(val) { return val <= effPrice; })
                        .sort(function(v1, v2) { return v2 - v1; })[0] || effPrice;
            }
        }

        // учитываем автобюджет и обнуляем ставку на поиске, если она меньше минимальной
        // "Нижний" минимум - до 8-го места
        if (brokerPrice.bid_price < Math.max(minPrice, effMinPrice)) {
            brokerPrice = brokerPriceZero;
        }

        if (autobudget == 'Yes' && minPrice && (autobudgetBid && minPrice < autobudgetBid || price < minPrice)) {
            brokerPrice = brokerPriceZero;
        }

        // вычисляем место
        if (searchStrategy.name == 'no_premium' && brokerPrice.bid_price > 0) {
            effPrice = Math.min(effPrice, brokerPrice.bid_price);
        }

        // имя блока, куда попадаем, могло быть определено ранее
        // в результате влияния стратегии "в блок по минимальной цене"
        isNaN(place) && (place = (function() {
            var i = 0,
                len = availablePrices.length,
                item;

            for (; i < len; i++) {
                item = availablePrices[i];

                if (item.price && effPrice >= item.price.bid_price) return item.place;
            }

            return PRICE_PLACES.ROTATION;
        })());

        return {
            price: round_price_to_currency_step(brokerPrice.amnesty_price / 1e6, currency, 'up'),
            coverage: coverage,
            truncated: truncated,
            place_name: place,
            place_name_without_coef: (priceDataWithoutCoef || {}).place_name || place
        };
    };

    this.calcCoverage = function( eff_price, prices, probs ) {
        if (prices.length < 2 || prices.length != probs.length || eff_price < prices[0]) return 0;

        // находим, между какими элементами попадает цена
        var low_i = direct.utils.binarySearch(eff_price, prices),
            i, low_price, high_price, k;

        // не учитываем последний элемент
        low_i = Math.min(prices.length - 1, low_i);
        // индекс предыдущего элемента (если указатель не на первом)
        low_i = Math.max(0, low_i - 1);

        low_price = prices[low_i];
        high_price = prices[low_i+1] >= eff_price ? prices[low_i+1] : eff_price;
        k = (eff_price - low_price) / ((high_price - low_price)||1);
        return probs[low_i] + k * ( probs[low_i+1] - probs[low_i] );
    };

    /*
     * Охват аудитории на тематических площадках
     */
    this.calcContextCoverage = function(price, pokazometer_data) {
        if (!pokazometer_data || !pokazometer_data.shows_list || pokazometer_data.shows_list.length < 1) { return 0; }

        var mass = prepareMass(pokazometer_data.shows_list, 'cost', 'cnt');

        //если все показы были с ценой выше нашей, то покрытие 0%
        if (price < mass[0].x) {
            return 0;
        }

        return interpolateLinear(price, mass)/(pokazometer_data.shows_cnt || 1)

    };

    this.calcRetargCoverage = function(price, hits_by_cost) {
        var length = hits_by_cost.length;
        //если все показы были с ценой выше нашей, то покрытие 0%
        if (price < hits_by_cost[0].price) {
            return 0;
        } else if (price >= hits_by_cost[length - 1].price ) {
            return 1;
        }
        var clicks = 0;


        for (var i = 0; i < length; i++) {
            if (price < hits_by_cost[i].price) {
                clicks = hits_by_cost[i - 1].shows;
                break;
            }
        }
        return  clicks/((hits_by_cost[length - 1].shows))
    };

    this.calcContextPriceByCoverage = function(cov, pokazometer_data) {
        if (!pokazometer_data || !pokazometer_data.shows_list || pokazometer_data.shows_list.length < 1) { return 0; }
        var cnt = pokazometer_data.shows_cnt * (cov / 100),
            mass = prepareMass(pokazometer_data.shows_list, 'cnt', 'cost');
        return interpolateLinear(cnt, mass);
    };

    function interpolateLinear(val, mass) {
        var prev = mass[0], current, result;
        for (var i = 0; i < mass.length; i++) {
            current = mass[i];
            if (val > current.x * 1) {
                prev = current;
            } else {
                return (prev.x * 1 - current.x * 1) != 0 ?
                    (prev.y * 1 - current.y * 1) * (val - current.x * 1) / (prev.x * 1 - current.x * 1) +current.y * 1 :
                        prev.y * 1;
            }
        }
        return prev.y;
    }

    function prepareMass(mass, x, y) {
        var result = [], data;
        for (var i = 0; i < mass.length; i++) {
            result.push({x: mass[i][x], y: mass[i][y]})
        }
        result = result.sort(function(a, b){
            if (a.x * 1 > b.x * 1) {
                return 1;
            } else if (a.x * 1 < b.x * 1) {
                return -1;
            } else {
                return 0;
            }
        });
        return result;

    }

    var placeRules = {
        PREMIUM1: { type: 'premium', index: 0 },
        PREMIUM2: { type: 'premium', index: 1 },
        PREMIUM3: { type: 'premium', index: 2 },
        PREMIUM4: { type: 'premium', index: 3 },
        GUARANTEE1: { type: 'guarantee', index: 0 },
        GUARANTEE2: { type: 'guarantee', index: 1 },
        GUARANTEE3: { type: 'guarantee', index: 2 },
        GUARANTEE4: { type: 'guarantee', index: 3 },
    };
    function getPricesByPlaceName(data, placeName) {
        return data[placeRules[placeName].type][placeRules[placeName].index];
    }


};
