include('../forecast.js');
include('../../common/model.js');

(function() {

    var messages = {

        forecastPeriod: {
            'year' : iget('год'),
            'quarter' : {
                '1' : iget('1 квартал'),
                '2' : iget('2 квартал'),
                '3' : iget('3 квартал'),
                '4' : iget('4 квартал')
            },
            'month' : {
                '0' : iget('30 дней'),
                '1' : iget('январь'),
                '2' : iget('февраль'),
                '3' : iget('март'),
                '4' : iget('апрель'),
                '5' : iget('май'),
                '6' : iget('июнь'),
                '7' : iget('июль'),
                '8' : iget('август'),
                '9' : iget('сентябрь'),
                '10' : iget('октябрь'),
                '11' : iget('ноябрь'),
                '12' : iget('декабрь')
            },
            'week' : iget('неделя')
        },

        partnersExpense: (function(){
            var result = {};
            populateHashWithPercentSteps(result);

            return result;
        })(),

        forecastPlatform: {
            all: iget('все'),
            mobile: iget('только мобильные')
        },

        showDeclinedPhrases: {
            yes: iget('продолжаются'),
            no: iget('приостановлены')
        },

        addPhrasesExpense: (function(){
            var result = {};
            populateHashWithPercentSteps(result);

            return result;
        })()

    };

    function populateHashWithPercentSteps(hash) {
        $.each("100 90 80 70 60 50 40 30 20 10 0".split(" "), function(i, item){
           hash[parseInt(item,10)] = item + "%";
        })
    }

    var self = direct.forecast.Model = function(data) {
        if (data) {
            data._minus_words = data.minus_words;
            delete data.minus_words;
            this.update(data);
        }
    };

    self.prototype = $.extend({}, common.Model, new function() {
        var that = this;
        var parent = common.Model;

        this.minus_words = function() {
            return this._minus_words;
        };

        this.getDisabledPhrases = function() {
            var disabledPhrases = {};

            var phrases = [];
            if (this.window_phrases) phrases = phrases.concat(this.window_phrases);

            for( var i = 0, l = phrases.length; i < l; i++) {
                (function(item){
                    if (!item.isActive) disabledPhrases[item.key] = item.phrase;
                })(phrases[i])
            }

            for (var md5 in this.key2cat_id) {
                if (!this.key2cat_id.hasOwnProperty(md5)) continue;
                var catId = this.key2cat_id[md5];
                if (catId == '') {
                    disabledPhrases[md5] = catId;
                }
            }

            return disabledPhrases;
        };

        this.areAllPhrasesDisabled = function() {
            try {
                return common.utils.keys(this.getDisabledPhrases()) == this.window_phrases.length;
            } catch(e) {
                return false;
            }
        };

        this.getForecastOptions = function() {
            return {
                disabledPhrases: this.getDisabledPhrases(),
                periodType: model.forecastPeriodType,
                periodValue: model.forecastPeriod
            };
        };

        this.setEmptyValues = function() {
            this.update({
                data_distributed: [],
                data_by_positions: [],
                window_phrases: window.phrases,
                key2phrase: {},
                phrase2key: {},
                unglued_keys: [],
                key2cat_id: {}
            });
        };

        this.getGeoText = function() {
            return BEM.blocks['i-models-manager'].get('campaign&banner:0', 'b-regions-tree').get('geo_text') || '';
        };

        this.getGeoId = function() {
            return BEM.blocks['i-models-manager'].get('campaign&banner:0', 'b-regions-tree').get('geo') || 0;
        };


        this.hasUngluedPhrases = function() {
            return (this.unglued_keys && this.unglued_keys.length);
        };

        this.hasPopupMinusWords = function() {
            return !!this.popup_minus_words && !!this.popup_minus_words.length;
        };


        this.isDistributedBudgetMode = function() {
            return !!(FORECAST_MODE == DISTRIBUTED);
        };

        this.exportData = function(exportDisabledItems) {
            var result = {},
                isDistributed = this.isDistributedBudgetMode(),
                disabledPhrases = this.getDisabledPhrases(),
                _this = this,
                jsonPhrases = [],
                phrases = isDistributed ? this.window_phrases : this.phrases;

            // common fields
            result['pseudo_currency_id'] = this.currencyId;
            result['pseudo_currency_rate'] = this.currencyRate;
            result['pseudo_currency_name'] = this.currencyName;
            result['currency_rate'] = this.currencyRate;
            result['geo'] = this.getGeoId();
            result['geo_text'] = this.getGeoText();
            result['json_minus_words'] = JSON.stringify(BEM.blocks['i-utils'].minusWordsStringToArray(model.minus_words()));
            result['forecast_type'] = FORECAST_MODE;

            // Если это прогноз 2.0 - передаем флаг в данных
            result['separate_shows'] = '1';
            result['is_mobile'] = model.forecastPlatform == 'mobile' ? '1' : '0';
            result['currency'] = model.forecastCurrency;

            // forecast options
            $.each(this.getForecastOptions(), function(option, value){
                if (option == 'disabledPhrases') return;
                result[option] = value;
            });

            // forecast restrictions
            if (isDistributed) {
                window.showVCGAuction && (result['restriction_bid_price'] = common.number.format(this.bid_price / 1e6));
                result['restriction_amnesty_price'] = common.number.format(this.amnesty_price / 1e6);
                result['restriction_sum'] = common.number.format(this.sum / 1e6);
                result['restriction_clicks'] = this.clicks;
            }

            if (phrases) {
                Object.keys(phrases).reduce(function(result, key) {
                    var phrase = phrases[key],
                        md5 = isDistributed ? phrase.key : key;

                    if (!exportDisabledItems && (md5 in disabledPhrases)) return result;

                    result.push(isDistributed ?
                        _this._getPhraseDataByDistributedBudget(md5) :
                        _this._getPhraseDataByPositions(phrase, md5));

                    return result;
                }, jsonPhrases);

                result['json_phrases'] = JSON.stringify(this.phrasesSumsToExport(jsonPhrases));
            }

            return result;
        };

        /**
         * Преобразовываем суммы в правильный формат - DIRECT-47845
         * @param jsonPhrases
         * @returns {*|{fixVersions}|Array}
         */
        this.phrasesSumsToExport = function(jsonPhrases) {
            return jsonPhrases.map(function(phrase) {
                phrase.sum = common.number.format(phrase.sum / 1e6);
                phrase.amnesty_price = common.number.format(phrase.amnesty_price / 1e6);
                phrase.bid_price = common.number.format(phrase.bid_price / 1e6);

                return phrase;
            })
        };

        /**
         * Ковертирует переданный объект фразы для экспорта в xls из вкладки "Бюджет по позициям"
         *
         * @param {Object} phrase
         * @param {String} md5
         * @returns {Object} объект содержит данные фразы по позициям
         * @private
         */
        this._getPhraseDataByPositions = function(phrase, md5) {
            var pricePlaces = window.PRICE_PLACES,
                entryPlaces = window.ENTRY_PLACES,

                phrasePlaceToPricePlaceIdMap = {
                    'first-premium': pricePlaces.PREMIUM1,
                    'premium': entryPlaces.PREMIUM,
                    'first-place': pricePlaces.GUARANTEE1,
                    'guarantee': entryPlaces.GUARANTEE
                },

                phraseData;

            window.showVCGAuction && (phrasePlaceToPricePlaceIdMap['second-premium'] = pricePlaces.PREMIUM2);

            phraseData = Object.keys(phrasePlaceToPricePlaceIdMap).reduce(function(result, phrasePosition) {
                var positionInfo = {};

                positionInfo.clicks = phrase.clicks[phrasePosition];
                positionInfo.sum = phrase.sum[phrasePosition];
                positionInfo.ctr = phrase.ctr[phrasePosition];
                positionInfo.amnesty_price = phrase.amnesty_price[phrasePosition];
                positionInfo.shows = phrase.shows[phrasePosition];

                window.showVCGAuction ?
                    (positionInfo.bid_price = phrase.bid_price[phrasePosition]) :
                    (positionInfo.bid_price = phrase.amnesty_price[phrasePosition]);

                result[phrasePlaceToPricePlaceIdMap[phrasePosition]] = positionInfo;

                return result;
            }, {});

            phraseData.phrase = this.key2phrase[md5];
            phraseData.position = phrasePlaceToPricePlaceIdMap[phrase.position];
            phraseData.shows = phrase.requests;

            return phraseData;
        };


        /**
         * Ковертирует переданный объект фразы для экспорта в xls из вкладки "Распределенный бюджет"
         *
         * @param {String} md5
         * @returns {Object} объект содержит данные фразы по позициям
         * @private
         */
        this._getPhraseDataByDistributedBudget = function(md5) {
            var phraseData = {},
                dataObj = model.phrases && model.phrases[md5],
                fields = 'clicks sum str bid_price amnesty_price'.split(' '),
                pricePlaces = window.PRICE_PLACES,
                entryPlaces = window.ENTRY_PLACES,
                phrasePlaceToPricePlaceIdMap = {
                    'first_premium': pricePlaces.PREMIUM1,
                    'second_premium': pricePlaces.PREMIUM2,
                    'premium': entryPlaces.PREMIUM,
                    'first_place': pricePlaces.GUARANTEE1,
                    std: entryPlaces.GUARANTEE
                };

            phraseData.phrase = this.key2phrase[md5];

            if (!dataObj) return phraseData;

            phraseData.shows = dataObj.shows;
            phraseData.positions = [
                phrasePlaceToPricePlaceIdMap[dataObj.position_1],
                phrasePlaceToPricePlaceIdMap[dataObj.position_2]
            ];

            fields.forEach(function(field) {
                phraseData[field] = dataObj[field];
            });

            return phraseData;
        };

        this.getMessage = function(fieldName){

            var message = 'default';

            switch (fieldName) {
                case 'forecastPeriod':
                    message = messages.forecastPeriod[this.forecastPeriodType];
                    if (!({week: true, year: true})[this.forecastPeriodType]) message = message[this.forecastPeriod];
                    break;
                case 'forecastPlatform':
                case 'partnersExpense':
                case 'addPhrasesExpense':
                    message = messages[fieldName][this[fieldName]];
                    break;
                case 'showDeclinedPhrases':
                    message = messages[fieldName][this[fieldName] ? 'yes' : 'no'];
                    break;
            }

            return message;
        };

        this.getMinusWordsArray = function(popupMinusWords) {
            var minus_words = popupMinusWords ? this.popup_minus_words : this._minus_words;
            if (!minus_words) return [];
            return minus_words.split(/[-|,\+\!]*(?=\s)/);
        };

        this.getMinusWordsWithoutMinusesArray = function(popupMinusWords) {
            var minusWords = this.getMinusWordsArray(popupMinusWords);
            return $.map(minusWords, function(item) {
                var minusWord = "";
                if (item) minusWord = $.trim(item.toLowerCase())
                                        .replace(/[\"\«\»]/gi, '')
                                        .replace(/^-/gi, '')
                                        .replace(/[\"\«\»]/gi, '');
                return minusWord == "" ? null : minusWord;
            });
        };

        this.getPopupMinusWordsWithoutMinuses = function() {
            return this.getMinusWordsWithoutMinusesArray(true).join(",");
        };

        /**
         * All below-listed fields are merged into groups in order to trigger a single event
         * provided that at least one of the fields has been changed
         */

        // fields which set forecast calculation restriction used by slider (see b-advanced-forecast__calculated-expense.js)
        this.forecastRestricitonsFields = window.showVCGAuction ?
            "clicks bid_price amnesty_price sum shows".split(" ") :
            "clicks amnesty_price sum shows".split(" ");

        // fields which modification should cause forecast recalculation without ajax request
        this.forecastParamsFields = "addPhrasesExpense partnersExpense".split(" ");

        // fields which modification should cause ajax request calling to get new "raw" forecast data
        this.getTransitionsByPhrasesParams = "forecastPeriod forecastPeriodType forecastPlatform forecastCurrency".split(" ");

        // fields representing values total values for Budget by Positions mode
        this.phraseSumsByPositionsFields = window.showVCGAuction ?
            "allPhrasesSum firstPremiumPosSum secondPremiumPosSum guaranteePosSum firstPlacePosSum premiumPosSum"
                .split(" ") :
            "allPhrasesSum firstPremiumPosSum guaranteePosSum firstPlacePosSum premiumPosSum"
                .split(" ");


        this._triggerChange = function(e) {
            common.Model._triggerChange.call(this, e);

            // data_distributed is the hash containng data essential for performing distributed budget forecast calculation
            if (e.changes['data_distributed']) this.trigger('change.raw_forecast_data', e);

            // unglued_keys is an array containng md5 keys of phrases the minus words were automatically added to
            if (e.changes['unglued_keys']) this.trigger('change.unglued_keys', e);

            // these are the hashes containing normalized data for the given phrase/category
            if (e.changes['phrases']) this.trigger('change.phrases', e);

            // fields which trigger events the summary block of the result table is subscribed for
            if (e.changes['sum']) this.trigger('change.budget', e);
            if (e.changes['shows']) this.trigger('change.shows', e);
            if (e.changes['clicks']) this.trigger('change.clicks', e);
            if (e.changes['requests']) this.trigger('change.requests', e);

            // recommended budget
            if (e.changes['rec_budget']) this.trigger('change.rec_budget', e);

            // fields representing global arrays
            if (e.changes['window_phrases']) this.trigger('change.window_phrases', e);

            if (e.changes['_minus_words']) this.trigger('change.minus_words', e);
            if (e.changes['popup_minus_words']) this.trigger('change.popup_minus_words', e);

            // current restrictions for distributed budget
            if (e.changes['restrictions']) this.trigger('change.restrictions', e);

            if (e.changes['forecastPeriod']) this.trigger('change.forecast_period', e);
            if (e.changes['forecastPeriodType']) this.trigger('change.forecastPeriodType', e);
            if (e.changes['forecastPlatform']) this.trigger('change.forecastPlatform', e);
            if (e.changes['forecastCurrency']) this.trigger('change.forecastCurrency', e);

            // it's not me, but b-region-selector.js component stores geo data inside of model.common namespace
            if (e.changes['common']) {
                var cmn = e.changes['common'];
                if (!cmn[0] || cmn[0]["geo"] != cmn[1]["geo"]) {
                    this.trigger('change.transitions-by-phrases-params', e);
                }
            }

            this._initTriggeringBulkEvents(e, 'change.phrase-sum-by-positions', this.phraseSumsByPositionsFields);
            this._initTriggeringBulkEvents(e, 'change.transitions-by-phrases-params', this.getTransitionsByPhrasesParams);
            this._initTriggeringBulkEvents(e, 'change.forecast-params', this.forecastParamsFields);
            this._initTriggeringBulkEvents(e, 'change.forecast-restrictions', this.forecastRestricitonsFields);
        };

        this._initTriggeringBulkEvents = function(event, eventName, fields) {
            var that = this;
            $.each(fields, function() {
                if (this in event.changes) {
                    that.trigger(eventName, event);
                    return false;
                }
            });
        }
    });

})();
