
include('../direct.js');

direct.utils = {

    /**
     * Экранирует спец символы в строке для RegExp-а
     * @param {String} string строка для RegExp-а
     * @returns {String}
     */
    quoteRegExp: function(string) {
        return (string + '').replace(/([.?*+^$[\]\\(){}|-])/g, "\\$1");
    },

    escapeHTMLRe:  /[&<>\"\']/g,
    escapeHTMLReplacer: function(c) {
        return '&#' + c.charCodeAt(0) + ';';
    },

    unescapeHTMLRe: /(&(lt|gt|quot|apos|amp|#\d+);|.)/gi,
    unescapeHTMLHash: {lt: '<', gt: '>', quot: '"', apos: "'", amp: '&'},
    unescapeHTMLReplacer: function(s, c, d) {
        return direct.utils.unescapeHTMLHash[d] || (d ? String.fromCharCode(d.substring(1)) : c);
    },

    getKeywordsFromPhrase: function(phrase) {
        return (phrase + '').replace(/\s-.*$/g, '');
    },

    getMinuswordsFromPhrase: function(phrase) {
        return (phrase + '').replace(/^(\s(?!-)|\S)*(\s(?=-))?/g, '');
    },

    commonDomainRegExp:  '((pp|ac|boom|msk|spb|nnov)\.ru|(net|org|com|int|edu)\.[-a-z]+)',
    domainRegExp: '(?:(?:[a-z0-9][a-z0-9-]*|[а-яёЄІЇєіїҐґ0-9][а-яёЄІЇєіїҐґ0-9-]*)\\.)+([a-z0-9а-яёЄІЇєіїҐґ]{2,15}|xn--[a-z0-9-]+)',

    stripWww: function(domain) {
        domain = domain.toLowerCase();
        if (domain.match(/^www\..*\./) && !domain.match('^www\\.' + direct.utils.commonDomainRegExp + '$')) {
            domain = domain.replace(/^www\./, '');
        }
        return domain;
    },

    stripHttp: function(domain) {
        if (!this.extractDomain(domain)) return '';

        if (domain.match(/^http:\/\/.*\./)) {
            domain = domain.replace(/^^http:\/\//, '');
        } else if (domain.match(/^https:\/\/.*\./)) {
            domain = domain.replace(/^^https:\/\//, '');
        }

        return domain;
    },

    getUrlProtocol: function(domain) {
        if (domain.match(/^http:/)) {
            return 'http://';
        } else if (domain.match(/^https:/)) {
            return 'https://';
        }

        return '';
    },

    getMinusWordsLength: function(words) {
        words = $.trim(words)
                .replace(/\s-/g, '')
                .replace(/^-/g, '')
                .replace(/\[|\]/g, '')
                .replace(/-/g, 'a')
                .replace(/[\u0000-\u002F\u1D00-\uFFFF]/g, '');
        return words.length;
    },

    hrefRegExp: function() {
        return new RegExp('^\\s*(?:http:\/\/)?' + direct.utils.domainRegExp + '(?:\\s*$|\/|:\\d+|\\?|#)', 'i');
    },
    emailRegExp: function() {
        return new RegExp('^\\s*[a-z0-9_\\.\\-\\+]+\\@' + direct.utils.domainRegExp + '\\.?\\s*$', 'i');
    },

    validateHref: function(href) {
        return href.match(direct.utils.hrefRegExp());
    },

    validateEmail: function(email) {
        return email.match(direct.utils.emailRegExp());
    },

    validateYandexEmail: function(email) {
        return email.match(/^\s*[a-z0-9_\.\-\+]+\@yandex-team.ru\s*$/i);
    },

    validateIP: function(ip) {
        var bytes_str, b, ok;
        bytes_str=ip.match(/^([0-9]{1,3})\.([0-9]{1,3})\.([0-9]{1,3})\.([0-9]{1,3})$/i)
        if(!bytes_str ){
            return false;
        }
        for( var i = 1; i <= 4 ; i++ ) {
            b=parseInt(bytes_str[i],10);
            ok=(b<=255) && !((b>0)&&(bytes_str[i][0]=='0'));
            if(!ok) {
                return false;
            }
        }
        return true;
    },

    validateOGRN: function(ogrnNumber) {
        var VALID_OGRN_LENGTH = 13,
            VALID_OGRNIP_LENGTH = 15,
            number = ogrnNumber * 1,
            sNumber = number + '';

        if (sNumber.length != VALID_OGRN_LENGTH && sNumber.length != VALID_OGRNIP_LENGTH) {
            return false;
        }

        var numberType = (sNumber.length == VALID_OGRN_LENGTH) ? 'OGRN' : 'OGRNIP',
            numberLength = sNumber.length;

        var firstDigit = sNumber.substring(0, 1);
        var lastDigitOfNumber = sNumber.substring(numberLength - 1, numberLength);
        var leadingNumber = parseInt(sNumber.substring(0, numberLength - 1), 10);

        // The first digit of OGRN/OGRNIP number should be 1, 2, 3 or 5
        if (!firstDigit.match(/[1235]/)) {
            return false;
        }

        var remainder = leadingNumber % (numberType == 'OGRN' ? 11 : 13);
        if (remainder < 10) {
            var lessSignificantDigitOfRemainder = remainder;
        } else {
            var sRemainder = remainder + '';
            var lessSignificantDigitOfRemainder = sRemainder.substring(sRemainder.length - 1, sRemainder.length);
        }

        return  !!(lessSignificantDigitOfRemainder == lastDigitOfNumber);
    },

    getLevelDomain: function(url, levelDomain) {
        var domain = this.extractDomain(url);
        var res = domain.match(/([^\/^\?]+)[\/\?]/) ? $1 : domain;
        var parts = res.split(/\./).reverse();
        if (parts.length >= levelDomain ) {
            parts = parts.slice(0, levelDomain);
        }
        
        return parts.reverse().join('.');
    },

    extractDomain: function(url) {
		return $.trim(url).replace(/http(s)?:\/\//, '').replace(/[\/\?].*/, '').replace(/\#.*/, '').replace(/\/$/, '')
	},

    extractHttp: function(url) {
        return $.trim(url).replace(/^http(s)?:\/\//, '')
    },

    parseUri: function(str) {
      	var options = {
            strictMode: false,
            key: ["source","protocol","authority","userInfo","user","password","host","port","relative","path","directory","file","query","anchor"],
            q:   {
                name:   "queryKey",
                parser: /(?:^|&)([^&=]*)=?([^&]*)/g
            },
            parser: {
                strict: /^(?:([^:\/?#]+):)?(?:\/\/((?:(([^:@]*)(?::([^:@]*))?)?@)?([^:\/?#]*)(?::(\d*))?))?((((?:[^?#\/]*\/)*)([^?#]*))(?:\?([^#]*))?(?:#(.*))?)/,
                loose:  /^(?:(?![^:@]+:[^:@\/]*@)([^:\/?#.]+):)?(?:\/\/)?((?:(([^:@]*)(?::([^:@]*))?)?@)?([^:\/?#]*)(?::(\d*))?)(((\/(?:[^?#](?![^?#\/]*\.[^?#\/.]+(?:[?#]|$)))*\/?)?([^?#\/]*))(?:\?([^#]*))?(?:#(.*))?)/
            }
        };

        var	o   = options,
            m   = o.parser[o.strictMode ? "strict" : "loose"].exec(str),
            uri = {},
            i   = 14;

        while (i--) uri[o.key[i]] = m[i] || "";

        uri[o.q.name] = {};
        uri[o.key[12]].replace(o.q.parser, function ($0, $1, $2) {
            if ($1) uri[o.q.name][$1] = $2;
        });

        return uri;
    },

    generateId: function(prefix) {
        return (prefix || '') + ((new Date()).getTime() + Math.round(Math.random() * 10000));
    },

    isEmpty: function(str) {
        if (str === undefined || str === null) {
            return true;
        } else {
            return !!(str + '').match(/^\s*$/);
        }
    },

    //Удаляет двойные пробелы и начальные/концевые проблемы
    clearString: function(str) {
        return (str + '').replace(/^\s+|\s+$/g, '').replace(/\s\s+/g, ' ');
    },

    //Чистит строку вида "aaa, bbb, cccc"
    clearSequence: function(str) {
        return direct.utils.clearString(str).replace(/,\s+/g, ',').replace(/,$/g, '');
    },
    
    isEqual: function(val1, val2) {
        if (val1 && val2) {
            return val1 == val2;
        } else { return !val1 && !val2; }
    },    

    escapeHTML: function(str) {
        str = '' + str;
        return str.replace(this.escapeHTMLRe, this.escapeHTMLReplacer);
    },

    unescapeHTML: function(str) {
        str = '' + str;
        return str.replace(this.unescapeHTMLRe, this.unescapeHTMLReplacer);
    },
    
    binarySearch: function (value, array) {
        var low = 0, high = array.length, mid;
        while (low < high) {
            mid = (low + high) >> 1;
            array[mid] < value ? low = mid + 1 : high = mid;
        }
        return low;
    },
    
    clone: function(object) {
        if (!object || 'object' !== typeof object) { return object}
        var copy = ('function' === typeof object.pop ? [] : {});
        var value;

        for (var property in object)  {
            if (object.hasOwnProperty(property)) {
                value = object[property];
                if (value && 'object' === typeof value) {
                     copy[property] = $.clone(value)
                } else {
                     copy[property] = value
                }
            }
        }

        return copy;
    },
    
    addWbrs: function(text, chunkSize, splitter) {
        chunkSize = chunkSize || 25;
        splitter  = splitter  || '<wbr />';

        var parts = text.replace(/[\s\xA0]{2,}/g, " ").replace(/([\s\.\-\/])/g, "$1\u0001").split('\u0001'),
            splitParts = [], i, l, part, tmp;
        
        for (i=0, l = parts.length; i < l; i++) {
            part = parts[i];
            while( part.length > chunkSize ) {
                tmp = Math.min(chunkSize, part.length / 2);
                splitParts.push( part.substring(0, tmp) );
                part = part.substring(tmp);
            }
            splitParts.push(part);
        }
        return splitParts.join(splitter);
    },

    openWindow: function(url, name, width, height) {
        var leftPos = ($(window).width() - width) >> 1, 
            topPos = ($(window).height() - height) >> 1,
            win = window.open(url, name, 'width=' + width + ',height=' + height + ',top=' + topPos + ',left=' + leftPos + ',resizable=yes,scrollbars=yes,status=0');
        win.focus();
    },
    
    range: function (from, to) {
        var result = new Array(to - from);
        for (var idx = 0; from <= to; from++, idx++) {
            result[idx] = from;
        }
        return result;
    },

    supplant: function(string, data) {
        return string.replace(/{([^{}]*)}/g,
            function (a, b) {
                var r = data[b];
                return typeof r === 'string' || typeof r === 'number' ? r : a;
            }
        );
    },

    getKeywordsLength: function(words) {
        return (''+words).replace(/-!/g, '-').length;
    },

    inherit: function(Parent, Child) {
        var F = function(){};
        F.prototype = Parent.prototype;

        Child.prototype = new F();
        Child.prototype.constructor = Child;
    },

    /**
    * Функция обрезающая строку по определённую длину, заменяя отброшенную часть на троеточие
    * @param {String} string строка для обрезания
    * @param {Number} length номур символа, начиная с которого, все остальные символы будут заменены на троеточие (если строка поданая на вход длиннее)
    * @return {String}
    */
    hellipCut: function(string, length){
        return string.length > length ? string.substr(0, length) + '...' : string + ''
    },

    decorator: function(f, variables) {
        var args = Array.prototype.slice.call(arguments, 1);
        return function () {
            var p = arguments;
            Array.prototype.push.apply(p, args);
            f.apply(this, p);
        }
    },

    //Лексикографическая сортировка для турции
    sortTr: function(str1, str2) {
        var alphabetData = 'AaBbCcÇçDdEeFfGgĞğHhIıİiJjKkLlMmNnOoÖöPpRrSsŞşTtUuÜüVvYyZz'.split('');

        // обычное сравнение строк или чисел
        var compare = function(a, b) {
            var indexA = $.inArray(a, alphabetData),
                indexB = $.inArray(b, alphabetData);
            if (indexA != -1 && indexB != -1) {
                return (indexA < indexB) ? -1 : (indexA > indexB) ? 1 : 0;
            } else if(indexA == -1 && indexB != -1) {
                return 1;
            } else if (indexA != -1 && indexB == -1) {
                return -1;
            } else {
                return (a < b) ? -1 : (a > b) ? 1 : 0;
            }

        };



        var compareArr1 = str1.split(''),
            compareArr2 = str2.split('');

        // перебираем части
        while (true) {
            var first = compareArr1.shift();
            var second = compareArr2.shift();
            // если обе части существуют ...
            if (undefined !== first && null !== undefined) {
                var comp = compare(first, second);
                if (comp != 0) {
                    return comp;
                }
            } else {
                return compare(str1, str2);
            }
        }
    }
    
};
