/**
 * Виртуальная клавиатура
 *
 * Отправляет на window события:
 *   - keyboardLangChanged.lego: поменялся язык клавиатуры в следствие действий пользователя
 *
 * Слушает на window события:
 *   - keyboardSetLang.lego: установить новый язык клавиатуры
 */

(function($) {

Lego.block['b-keyboard'] = function(params) {
    var keyboard = new Lego.block['b-keyboard'].Keyboard(this, params);
};

Lego.block['b-keyboard'].lang = {};


var Keyboard = Lego.block['b-keyboard'].Keyboard = function(elem, params) {
    this.initialize(elem, params);
};

Keyboard.prototype = {
    rowsLayout: [14, 15, 14, 13],
    KEY_BLOCK: 'b-keyboard__key',

    targetField: null,

    hold: false,
    holdInterval: null,
    holdDelay: 0,
    clearHold: function() {
        this.hold = false;
        this.holdDelay = 0;
        clearInterval(this.holdInterval);
    },

    unescapeHTMLRe: /(&(lt|gt|quot|apos|amp|#\d+);|.)/gi,
    unescapeHTMLHash: {lt: '<', gt: '>', quot: '"', apos: "'", amp: '&'},

    initialize: function(elem, params) {
        this.elem = elem;
        this.params = params;

        if (params.fake !== false) {
            // Сейчас elem - временный элемент b-keyboard, удаляем его
            elem.remove();
            // Инициализируем настоящий элемент b-keyboard
            Lego.blockInit($('body'), '.b-keyboard');
        }

        // Блокируем выделение текста внутри блока
        elem[0].onmousedown = function() { return false; };
        elem[0].onselectstart = function() { return false; };

        if (params.fake === false) {
            var defaultLanguage;

            // Wrap in try-catch to avoid possible access violation in IE
            try { defaultLanguage = window.location.hostname.match(/.(ru|ua|by|kz)$/); }
            catch (e) { defaultLanguage = ['', 'en'] }

            this.attachEventsOnce();

            elem.data('accent', 'normal');
            this.language(params.lang || Lego.params.locale || defaultLanguage && defaultLanguage[1] || 'en');
            this.modality('normal', false);
        }

        var _this = this;

        $(window).bind('keyboardSetLang.lego', function(e, lang) {
            var item = $('.b-keyboard__lang-' + lang).get(0);
            if (item) {
                _this.switchLayout(item);
            }
        });
    },

    // Установка языка и лэйаута
    language: function(lang) {
        this.elem.data('lang', Lego.block['b-keyboard'].lang[lang]);
    },
    modality: function(modality, capslock) {
        var accent = modality.indexOf('accent_') > -1 && modality.split("_")[1];
        var elem = this.elem;

        // Если запрос на смену акцента - меняем акцент, оставляем текущую раскладку
        if (accent) {
            accent = elem.data('accent') == accent ? 'normal' : accent;
            modality = elem.data('modality') || 'normal';
            capslock = elem.data('capslock') || false;
        // Если запрос на смену раскладки - меняем раскладку, оставляем текущий акцент
        } else {
            accent = elem.data('accent') || 'normal';
            modality = elem.data('modality') == modality ? 'normal' : modality;
            capslock = capslock || false;
        }

        // Устанавливаем вычисленные значения раскладки и акцента и генерируем разметку
        elem
            .data('langModality', elem.data('lang')[modality])
            .data('modality', modality)
            .data('capslock', capslock)
            .data('accent', accent)
            .html(this.rows());

        this.initBlock();
        this.markLayout(modality, accent);
        this.attachEvents();
    },

    initBlock: function() {
        Lego.blockInit(this.elem, '.b-dropdown');
    },

    // Генераторы разметки
    rows: function() {
        var rows = '', start = 0;
        for (var i = 0; i < this.rowsLayout.length; i++) {
            var rowLength = parseInt(this.rowsLayout[i]);
            rows += '<table class="b-keyboard__row"><tr>' + this.keys(start, rowLength) + '</tr></table>';
            start += rowLength;
        }
        rows += this.bottomRow();
        return rows;
    },

    keys: function(start, rowLength) {
        var keys = '';
        for (var key = start; key < start + rowLength; key++) {
            keys += this.key(key);
        }
        return keys;
    },

    key: function(key) {
        var elem = this.elem;
        // Special keys can be defined either per-modality or per-language
        var specialKeys = elem.data('langModality').specialKeys || elem.data('lang').specialKeys;
        // Special keys are enumerated from 1
        var specialKey = specialKeys[key + 1];

        var keyClass = (specialKey ? (' ' + this.KEY_BLOCK + '_special-' + specialKey.type) : '');
        var keyLabel = (specialKey ? specialKey.label : elem.data('langModality').keys.charAt(key));

        // Certain layouts have accent modifiers
        var accent = elem.data('accent');
        var accentKeys = (accent && elem.data('langModality').accents) ? elem.data('langModality').accents[accent] : null;

        // Mark the accent modifier keys
        if (specialKey && specialKey.type.indexOf('accent') > -1) keyClass += ' ' + this.KEY_MARKED;

        if (accentKeys && !specialKey) {
            if (accentKeys[key + 1]) {
                // Set the alternative label (from accentKeys)
                keyLabel = accentKeys[key + 1].label;
            } else {
                // Disable the button, if it's not of 'accent' type
                keyClass += ' ' + this.KEY_DISABLED;
            }
        }

        if (keyLabel == "shift" || keyLabel == "capslock" || keyLabel == "enter" || keyLabel == "ua") {
            keyLabel = '<img class="b-keyboard__key__image" src="' + Lego.params['lego-static-host'] + '/blocks/b-keyboard/key/b-keyboard__key.' + keyLabel + '.png" alt="' + keyLabel +'"/>';
        }

        if (keyLabel == '\u0000') keyLabel = '';

        return '<td class="b-keyboard__row__cell">' +
                '<span class="' + this.KEY_BLOCK + keyClass + '">' +
                    '<span class="' + this.KEY_BLOCK + '-l"></span>' +
                    '<span class="' + this.KEY_BLOCK + '-m">' + keyLabel + '</span>' +
                    '<span class="' + this.KEY_BLOCK + '-r"></span>' +
                '</span>' +
               '</td>';
    },

    bottomRow: function() {
        return '<table class="b-keyboard__row">' +
                '<tr>' +
                    '<td class="b-keyboard__row__cell">' +
                        this.langSelector() +
                    '</td>' +
                    '<td class="b-keyboard__row__cell">' +
                        '<span class="b-keyboard__key b-keyboard__key_special-space">' +
                            '<span class="b-keyboard__key-l"></span>' +
                            '<span class="b-keyboard__key-m">\u0020</span>' +
                            '<span class="b-keyboard__key-r"></span>' +
                        '</span>' +
                    '</td>' +
                    '<td class="b-keyboard__row__cell">' +
                        '<span class="b-keyboard__key b-keyboard__key-spacer"></span>' +
                    '</td>' +
                    '<td class="b-keyboard__row__cell">' +
                        '<span class="b-keyboard__key b-keyboard__key_special-alt b-keyboard__key_state_marked">' +
                            '<span class="b-keyboard__key-l"></span>' +
                            '<span class="b-keyboard__key-m">« » { } ~</span>' +
                            '<span class="b-keyboard__key-r"></span>' +
                        '</span>' +
                    '</td>' +
                '</tr>' +
               '</table>';
    },

    langSelector: function() {
        var elem = this.elem;
        var curLang = elem.data('lang').id;
        return '<div class="b-keyboard__lang">' +
                '<div class="b-keyboard__lang-i">' +
                    '<div class="b-dropdown b-dropdown_direction_up b-dropdown_theme_black g-js" onclick="return {name:\'b-dropdown\', direction: \'up\'}">' +
                        '<ul class="b-dropdown__list">'+
                            this.langItems() +
                        '</ul>' +
                    '</div>' +
                '</div>' +
               '</div>';
    },

    langItems: function() {
        var langs = ['by', 'en', 'fr', 'de', 'it', 'kz', 'es', 'ru', 'tt', 'tr', 'ua'],
            langIndex = $.inArray(Lego.params.locale, langs);

        langIndex != -1 && langs.unshift(langs.splice(langIndex, 1)[0]);

        var elem = this.elem;
        var curLang = elem.data('lang').id;

        var items = '';
        for (var i = 0; i < langs.length; i++) {
            var lang = langs[i],
                langLabel = Lego.block['b-keyboard'].lang[lang].label;

            // Build the language list, skipping the current one
            if (lang != curLang) {
                items += '<li class="b-dropdown__item">' +
                            '<a class="b-dropdown__link b-keyboard__lang-' + lang + '" href="#">' +
                                '<i class="b-keyboard__lang-ic"></i>' + langLabel +
                            '</a>' +
                          '</li>';
            }
        }

        // Append current language to the bottom of the list
        items += '<li class="b-dropdown__item b-dropdown__visible">' +
                    '<a class="b-dropdown__or b-keyboard__lang-' + curLang + '" href="#">' +
                        '<i class="b-keyboard__lang-ic"></i>' + Lego.block['b-keyboard'].lang[curLang].label + '<i class="b-dropdown__or-ie">▼</i>' +
                    '</a>' +
                  '</li>';

        return items;
    },

    // Обработчики нажатий на клавиши виртуальной клавиатуры
    handleKey: function(key) {
        var specialKey = key.className.match(/b-keyboard__key_special-(\w+)/);

        if (specialKey) {
            this.handleSpecialKey(key, specialKey[1]);
        } else {
            this.handleNormalKey(key);
        }

        this.targetField && $(this.targetField).keydown();

        return (specialKey) ? specialKey[1] : null;
    },

    handleNormalKey: function(key, type) {
        var _this = this;
        var elem = this.elem;
        var targetField = this.targetField;
        if (key && targetField) {
            var $key = $(key);
            if ($key.hasClass(this.KEY_DISABLED)) return;

            // Получаем набранный символ
            var keyValue = $key.find('.' + this.KEY_BLOCK + '-m').html().replace(this.unescapeHTMLRe, function(s, c, d) {
                return _this.unescapeHTMLHash[d] || (d ? String.fromCharCode(d.substring(1)) : c);
            });

            // Устанавливаем значение с учетом положения каретки
            var $targetField = $(targetField);

            if (type != 'backspace') {
                if (type == 'enter') {
                    // если нажали экранный Enter в текстовом поле - сабмитим форму
                    if (targetField.tagName.toLowerCase() == 'input') {
                        try { $(targetField.form).submit(); } catch(e) {}
                        return;
                    }
                    else keyValue = '\n';
                }
                if (type == 'space') keyValue = ' ';
                if (type == 'hryvna') keyValue = '\u20b4';
                $targetField.insertAtCaretPos(keyValue);
            } else {
                $targetField.deleteAtCaretPos();
            }
        }

        // Reset modality if we're in Shift mode
        if (elem.data('modality') == 'shift' && elem.data('capslock') === false) {
            window.setTimeout(function() {
                _this.modality('normal', false);
            }, 1);
        }

        // Reset accent if we're in accent mode
        if (elem.data('accent').toString() != 'normal') {
            this.modality('accent_normal', false);
        }
    },

    handleSpecialKey: function(key, type) {
        switch (type) {
            case 'lshift':
            case 'rshift':
                this.modality('shift', false);
                break;
            case 'capslock':
                this.modality('shift', true);
                break;
            case 'alt':
                this.modality('alt', false);
                break;
            case 'accent_circumflex':
            case 'accent_umlaut':
            case 'accent_grave':
            case 'accent_acute':
            case 'accent_mod':
                this.modality(type, false);
                break;
            case 'hryvna':
            case 'backspace':
            case 'atmark':
            case 'enter':
            default:
                this.handleNormalKey(key, type);
                break;
        }
    },


    markLayout: function(type, accent) {
        // Mark modality button
        if (type == 'alt') {
            $('.' + this.KEY_BLOCK + '_special-alt').addClass(this.KEY_SUPPRESSED);
        }

        if (type == 'shift') {
            if (this.elem.data('capslock') === true)
                $('.' + this.KEY_BLOCK + '_special-capslock').addClass(this.KEY_SUPPRESSED);
            else {
                $('.' + this.KEY_BLOCK + '_special-lshift').addClass(this.KEY_SUPPRESSED);
                $('.' + this.KEY_BLOCK + '_special-rshift').addClass(this.KEY_SUPPRESSED);
            }
        }

        // Mark accent button
        if (accent) {
            $('.' + this.KEY_BLOCK + '_special-accent_' + accent).addClass(this.KEY_SUPPRESSED);
        }
    },

    switchLayout: function(layout) {
        var lang = layout.className.match(/b-keyboard__lang-(\w+)/);
        var elem = this.elem;

        if (lang) {
            this.language(lang[1]);
            elem.data('accent', 'normal');
            this.modality('normal', false);
            elem.click(); // Восстанавливаем фокус
            $(window).trigger('keyboardLangChanged.lego', [lang[1]]);
        }
    },

    attachEventsOnce: function() {

        var elem = this.elem;
        var _this = this;

        // Привязываем обработчики событий к клавишам виртуальной клавиатуры
        elem
            .find('.' + this.KEY_BLOCK)
            .live('click', function() {
                _this.targetField && $(_this.targetField).keypress();
                return false;
            })
            .live('mousedown', function(e) {
                // Only allow left-button click
                if (e.which && e.which != 1 || e.button && e.button != 1)
                    return false;

                if (!$(this).hasClass(_this.KEY_DISABLED)) {
                    $(this).addClass(_this.KEY_PRESSED).data("pressed", 1);
                }

                // Первое нажатие
                var isSpecialKey = _this.handleKey(this);
                if (isSpecialKey && isSpecialKey != 'atmark' && isSpecialKey != 'backspace') {
                    // Если спец. клавиша (кроме @ и Backspace) - автоповтор не нужен, заканчиваем
                    return;
                }

                // Включаем автоповтор
                this.hold = true;

                this.holdInterval = window.setInterval(function() {
                    if (this.hold === true) {
                        // Первые 3 цикла - пропускаем (задержка), потом начинаем повторять нажатую клавишу
                        if (_this.holdDelay < 3) {
                            _this.holdDelay++;
                        } else {
                            _this.handleKey(_this);
                        }
                    }
                }, 100);
            })
            .live('mouseup', function() {
                // Выключаем автоповтор
                _this.clearHold();

                $(this).removeClass(_this.KEY_PRESSED).data("pressed", 0);

                _this.targetField && $(_this.targetField).keyup();
            });

        // Защита от кражи фокуса из активного поля ввода при промахивании мимо клавиши
        elem.click(function(e) {
            if (_this.targetField && !$(e.target).is('.' + _this.KEY_BLOCK)) {
                $(_this.targetField).setCaretPos();
            }
        });

        // вводить можно только в текстовые поля
        // можно написать input[type=text], но иногда не пишут type
        var $inputs = $('input').filter(function() {
            return this.type === 'text' || this.type === 'search' || this.type === 'password';
        }).add('textarea').filter(':visible');

        // Привязываем клавиатуру к полю с активным фокусом
        $inputs
            // mouseup нужен для ловли выделения
            .bind('mouseup focus', function(e) {
                // блокировка обработки фокуса после возвращения коретки на нужное место после ввода
                if (blockFocus && e.type == 'focus') {
                    blockFocus = false;
                    return;
                }
                _this.targetField = this;
                // IE имеет странное поведение на domElem.focus() и выставляет его в начало строки
                // это можно отсечь с помощью проверки e.target == e.srcElement
                // в этом случае принудительно выставляем фокус в конец строки
                if (e.type == 'focus' && e.target == e.srcElement) {
                    $(this).setCaretPos(this.value.length);
                } else {
                    $(this).saveCaretPos();
                }
            });

        var blockFocus = false;
        // IE - сохраняем позицию каретки в активном поле при событиях keypress и change
        if ($.browser.msie) {

            var IESetCaretTimeout;

            // FIXME: тут какой-то ахтунг в логике, если честно :)
            $inputs.bind('keyup change', function(e) {
                // ненастоящие события
                if (e.type == 'change' || !e.keyCode) {
                    // Таймаут обязателен, так как фокус нужно перевыставить после blur
                    if (IESetCaretTimeout) {
                        window.clearTimeout(IESetCaretTimeout);
                    }
                    blockFocus = true;
                    IESetCaretTimeout = window.setTimeout(function() {
                        $(_this.targetField).setCaretPos();
                        IESetCaretTimeout = null;
                    }, 10);
                }
            });
        }

        $inputs
            // keyup нужен для ловли ввода символов с клавиатуры и правильного выставления фокуса после этого
            .bind('keyup', function(e) {
                if (_this.targetField && e.keyCode) {
                    $(_this.targetField).saveCaretPos();
                }
            });

        // Привязываем клавиатуру к первому полю ввода
        $inputs.each(function() {
            if ((this.tagName.toLowerCase() == 'input' && (this.type == 'text' || this.type == 'search')) ||
                this.tagName.toLowerCase() == 'textarea') {
                    var elem = $(this);
                    elem.saveCaretPos();
                    elem.data('lego:focused') && (_this.targetField = this);
                return false;
            }
        });
    },

    attachEvents: function() {
        var elem = this.elem;
        var _this = this;
        // Инициализируем переключатель языков
        elem
            .find('.b-dropdown__link')
            .bind('click', function() { _this.switchLayout(this); return false; });

        // Защита от залипания подсветки нажатой клавиши
        elem
            .find('.' + this.KEY_BLOCK)
            .bind('mouseleave', function() {
                var elem = $(this);
                if (elem.data("pressed") == 1) elem.removeClass(_this.KEY_PRESSED);

                // Выключаем автоповтор
                _this.clearHold();
            });
    },

    getDefaultLanguage: function(elem) {
        var defaultLanguage;

        // Wrap in try-catch to avoid possible access violation in IE
        try { defaultLanguage = window.location.hostname.match(/.(ru|ua|by|kz)$/); }
        catch (e) { defaultLanguage = ['', 'en'] }

        this.attachEventsOnce();

        this.elem.data('accent', 'normal');
        this.language(this.params.lang || Lego.params.locale || defaultLanguage && defaultLanguage[1] || 'en');
        this.modality('normal', false);
    }

};

Keyboard.prototype.KEY_STATE = Keyboard.prototype.KEY_BLOCK + '_state_';
Keyboard.prototype.KEY_PRESSED = Keyboard.prototype.KEY_STATE + 'pressed';
Keyboard.prototype.KEY_SUPPRESSED = Keyboard.prototype.KEY_STATE + 'suppressed';
Keyboard.prototype.KEY_MARKED = Keyboard.prototype.KEY_STATE + 'marked';
Keyboard.prototype.KEY_DISABLED = Keyboard.prototype.KEY_STATE + 'disabled';

})(jQuery);


include('jquery.fn.caret.js');
include('lang/b-keyboard__lang_en.js');
include('lang/b-keyboard__lang_de.js');
include('lang/b-keyboard__lang_fr.js');
include('lang/b-keyboard__lang_it.js');
include('lang/b-keyboard__lang_es.js');
include('lang/b-keyboard__lang_ua.js');
include('lang/b-keyboard__lang_ru.js');
include('lang/b-keyboard__lang_by.js');
include('lang/b-keyboard__lang_kz.js');
include('lang/b-keyboard__lang_tt.js');
include('lang/b-keyboard__lang_tr.js');
