(function($) {

BEM.DOM.decl('b-popupa', {

    onSetMod : {

        'js' : function() {

            /**
             * DOM-элемент, для которого открывается попап
             * @private
             * @type jQuery
             */
            this._owner = null;

            /**
             * Состояние видимости
             * @private
             * @type Boolean
             */
            this._isShowed = false;

            /**
             * Приоритетное направление для открытия попапа
             * @private
             * @type String
             */
            this._direction = this.getMod('direction') || 'down';

        }

    },

    /**
     * Показывает попап
     * @param {jQuery|Object} owner DOM-элемент или координаты { left : x, top : y }, относительно которых рассчитывается положение
     */
    show : function(owner) {

        if(!this._isShowed || this._owner !== owner) {
            this._owner = owner;
            this._getUnder().show({ left : -10000, top : -10000 });
            this.pos();
        }

        return this;

    },

    /**
     * Скрывает попап
     */
    hide : function() {

        this._isShowed && this._getUnder().hide();
        return this;

    },

    /**
     * Показывает/скрывает попап в зависимости от текущего состояния
     * @param {jQuery|Object} owner DOM-элемент или координаты { left : x, top : y }, относительно которых рассчитывается положение
     */
    toggle : function(owner) {

        return this.isShowed()?
            this.hide() :
            this.show(owner);

    },

    /**
     * Позиционирует попап
     */
    pos : function() {

        var params = this._calcParams(this._owner);

        this.elem('tail').css(params.tailOffsets);
        this
            .setMod('direction', params.direction)
            ._getUnder().show(params.offsets);

        return this;

    },

    /**
     * Возвращает состояние видимости попапа
     * @returns {Boolean}
     */
    isShowed : function() {

        return this._isShowed;

    },

    /**
     * Устанавливает приоритетное направление открытия попапа
     * @param {String} direction направление (up|right|down|left)
     */
    setDirection : function(direction) {

        if(this._direction != direction) {
            this._direction = direction;
            this.isShowed() && this.pos();
        }

    },

    /**
     * Устанавливает контент попапа
     * @param {String|jQuery} content контент
     * @param {Function} [callback] обработчик, вызываемый после инициализации
     * @param {Object} [callbackCtx] контекст обработчика
     */
    setContent : function(content, callback, callbackCtx) {

        BEM.DOM.update(this.elem('content'), content, callback, callbackCtx);
        return this.isShowed()? this.pos() : this;

    },

    /**
     * Проверяет, является ли владелец попапа DOM-элементом
     * @returns {Boolean}
     */
    _isOwnerNode : function() {

        return !!(this._owner && this._owner.jquery);

    },

    /**
     * Вычисляет необходимые метрики для расчета направления открытия попапа
     * @private
     * @returns {Object}
     */
    _calcDimensions : function() {

        var underDomElem = this._under.domElem,
            doc = this.__self.doc,
            owner = this._owner,
            isOwnerNode = this._isOwnerNode(),
            ownerOffset = isOwnerNode? owner.offset() : owner,
            ownerWidth = isOwnerNode? owner.outerWidth() : TAIL_OFFSET,
            ownerHeight = isOwnerNode? owner.outerHeight() : TAIL_OFFSET,
            scrollLeft = doc.scrollLeft(),
            scrollTop = doc.scrollTop(),
            windowSize = this.__self.getWindowSize(),
            borderWidth = parseInt(this.elem('content').css('border-top-width'), 10);

        return {
            ownerLeft : ownerOffset.left,
            ownerTop : ownerOffset.top,
            ownerRight : ownerOffset.left + ownerWidth,
            ownerBottom : ownerOffset.top + ownerHeight,
            ownerMiddle : ownerOffset.left + ownerWidth / 2,
            underWidth : underDomElem.outerWidth(),
            underHeight : underDomElem.outerHeight(),
            borderWidth : isNaN(borderWidth)? 0 : borderWidth,
            windowLeft : scrollLeft,
            windowRight : scrollLeft + windowSize.width,
            windowTop : scrollTop,
            windowBottom : scrollTop + windowSize.height
        };

    },

    /**
     * Вычисляет параметры открытия попапа
     * @private
     * @returns {Object} хэш вида { direction, offset.left, offset.top }
     */
    _calcParams : function() {

        var d = this._calcDimensions();

        if(this.hasMod('adjustable', 'no'))
            return calcDirectionParams(this._direction, d);

        var checkedDirections = {},
            allowedDirections = this.params.directions,
            currentDirectionIdx = $.inArray(this._direction, allowedDirections);

        // обработка случая когда текущее направление открытия не является допустимым
        currentDirectionIdx > -1 || (currentDirectionIdx = 0);

        var priorityDirectionIdx = currentDirectionIdx,
            currentDirection, params;

        do {
            currentDirection = allowedDirections[currentDirectionIdx];

            params = checkedDirections[currentDirection] = calcDirectionParams(currentDirection, d);
            if(!params.factor) // значит полностью попал в окно
                return params;

            // находим следующее возможное направление открытия, если оно есть
            ++currentDirectionIdx == allowedDirections.length && (currentDirectionIdx = 0);

        } while(currentDirectionIdx !== priorityDirectionIdx);

        return checkedDirections[allowedDirections[0]];

    },

    getDefaultParams : function() {

        return {
            directions : ALLOWED_DIRECTIONS
        };

    },

    destruct : function() {

        var under = this._under;
        if(!under) {
            this.__base.apply(this, arguments);
        }
        else if(!this._destructing) {
            this._destructing = true;
            BEM.DOM.destruct(false, under.domElem);
            this.__base(true);
        }

    },

    /**
     * Возвращает подложку
     * @private
     * @returns {BEM.DOM.blocks['i-popup']}
     */
    _getUnder : function() {

        var _this = this;

        if(!_this._under) {
            var under = $(BEM.HTML.build({
                block : 'i-popup',
                zIndex : this.params.zIndex,
                mods : {
                    autoclosable : _this.getMod('autoclosable') || 'yes',
                    fixed : _this.hasMod('direction', 'fixed') && 'yes'
                },
                underMods : _this.params.underMods,
                underMix : [{ block : 'b-popupa', elem : 'under' }]
            }));

            (_this._under = _this.findBlockOn(under, 'i-popup'))
                .on(
                    {
                        'show' : function() {

                            _this._isShowed = true;

                            _this.hasMod('adjustable', 'no') ||
                                (_this
                                    .bindToWin('resize', _this.pos)
                                    ._isOwnerNode() &&
                                        _this.bindToDomElem(_this._owner.parents(), 'scroll', _this.pos));

                            _this.trigger('show');

                        },
                        'hide' : function() {

                            _this._isShowed = false;

                            _this.hasMod('adjustable', 'no') ||
                                (_this
                                    .unbindFromWin('resize')
                                    ._isOwnerNode() &&
                                        _this.unbindFromDomElem(_this._owner.parents(), 'scroll'));

                            _this.trigger('hide');

                        },
                        'outside-click' : function() {

                            _this.trigger.apply(_this, arguments);

                        }
                    })
                .elem('content').append(_this.domElem);

        }

        return _this._under;

    }

}, {

    live : function() {

        this.liveBindTo('close', 'leftclick', function() {
            this.hide();
        });

    }

});

var TAIL_OFFSET = 19,
    TAIL_WIDTH_HORIZONTAL = 7,
    TAIL_WIDTH_VERTICAL = 15,
    TAIL_HEIGHT_HORIZONTAL = 15,
    TAIL_HEIGHT_VERTICAL = 7,
    SHADOW_SIZE = 8,
    ALLOWED_DIRECTIONS = [
        'down', 'down-right', 'down-left', 'up', 'up-right',
        'up-left', 'right', 'right-up', 'left', 'left-up'];

/**
 * Рассчитывает параметры открытия попапа в заданном направлении
 * @private
 * @param {String} direction направление
 * @param {Object} d метрики
 * @returns {Object}
 */
function calcDirectionParams(direction, d) {

    var factor, offsets, tailOffsets, calcDirection;

    switch(direction) {
        case 'down':
        case 'up':
            factor = calcInWindowFactor(offsets = {
                left : d.ownerMiddle - d.underWidth / 2,
                top : direction == 'down'?
                    d.ownerBottom + TAIL_HEIGHT_VERTICAL :
                    d.ownerTop - d.underHeight - TAIL_HEIGHT_VERTICAL
            }, d);
            tailOffsets = {
                marginLeft : (d.ownerRight - d.ownerLeft) / 2 + d.ownerLeft - offsets.left - TAIL_WIDTH_VERTICAL / 2,
                marginTop : (direction == 'down'? -TAIL_HEIGHT_VERTICAL + d.borderWidth : -d.borderWidth)
            };
        break;

        case 'down-right':
        case 'down-left':
        case 'up-right':
        case 'up-left':
            calcDirection = direction == 'down-right' || direction == 'down-left'? 'down' : 'up';
            factor = calcInWindowFactor(offsets = {
                left : (direction == 'down-right' || direction == 'up-right'?
                    d.ownerLeft :
                    d.ownerRight - d.underWidth),
                top : calcDirection == 'down'?
                    d.ownerBottom + TAIL_HEIGHT_VERTICAL :
                    d.ownerTop - d.underHeight - TAIL_HEIGHT_VERTICAL
            }, d);
            tailOffsets = {
                marginLeft : (d.ownerRight - d.ownerLeft) / 2 + d.ownerLeft - offsets.left - TAIL_WIDTH_VERTICAL / 2,
                marginTop : (calcDirection == 'down'? -TAIL_HEIGHT_VERTICAL + d.borderWidth : -d.borderWidth)
            };
        break;

        case 'left':
        case 'right':
            factor = calcInWindowFactor(offsets = {
                left : (direction == 'left'?
                    d.ownerLeft - d.underWidth - TAIL_WIDTH_HORIZONTAL :
                    d.ownerRight + TAIL_WIDTH_HORIZONTAL),
                top : d.ownerTop - TAIL_OFFSET + TAIL_HEIGHT_HORIZONTAL / 2
            }, d);
            tailOffsets = {
                marginLeft : direction == 'left'? -d.borderWidth : -TAIL_WIDTH_HORIZONTAL + d.borderWidth,
                marginTop : TAIL_OFFSET - TAIL_HEIGHT_HORIZONTAL / 2
            };
        break;

        case 'left-up':
        case 'right-up':
            factor = calcInWindowFactor(offsets = {
                left : (direction == 'left-up'?
                    d.ownerLeft - d.underWidth - TAIL_WIDTH_HORIZONTAL :
                    d.ownerRight + TAIL_WIDTH_HORIZONTAL),
                top : d.ownerTop + TAIL_HEIGHT_HORIZONTAL / 2 + TAIL_OFFSET - d.underHeight
            }, d);
            calcDirection = direction == 'left-up'? 'left' : 'right';
            tailOffsets = {
                marginLeft : calcDirection == 'left'? -d.borderWidth : -TAIL_WIDTH_HORIZONTAL + d.borderWidth,
                marginTop : d.ownerTop - offsets.top + SHADOW_SIZE - TAIL_HEIGHT_HORIZONTAL / 2
            };
    }

    return {
        direction : calcDirection || direction,
        factor : factor,
        offsets : offsets,
        tailOffsets : tailOffsets
    };

}

/**
 * Вычисляет фактор попадания объекта в окно
 * @param {Object} pos параметры объекта
 * @param {Object} d метрики
 * @returns {Number} фактор попадания (если 0 - то полностью попадает, если нет -- то чем он больше, тем хуже попадает)
 */
function calcInWindowFactor(pos, d) {

    var res = 0;

    d.windowTop > pos.top && (res += d.windowTop - pos.top);
    pos.top + d.underHeight > d.windowBottom && (res += pos.top + d.underHeight - d.windowBottom);
    d.windowLeft > pos.left && (res += d.windowLeft - pos.left);
    pos.left + d.underWidth > d.windowRight && (res += pos.left + d.underWidth - d.windowRight);

    return res;


}

BEM.HTML.decl('b-popupa', {

    onBlock : function(ctx) {

        var hasClose = false;
        $.each(ctx.param('content'), function(i, item) {
            return !(hasClose = item.elem == 'close');
        });
        ctx
            .mods({ theme : 'ffffff', direction : 'down', 'has-close' : hasClose && 'yes' })
            .js(true)
            .afterContent({ elem : 'shadow' });

    },

    onElem : {

        'content' : function(ctx) {

            ctx
                .wrap({ elem : 'wrap-cell', tag : 'td' })
                .wrap({ tag : 'tr' })
                .wrap({ elem : 'wrap', tag : 'table' });

        },

        'close' : function(ctx) {

            ctx.tag('i');

        },

        'shadow' : function(ctx) {

            ctx.tag('i');

        },

        'tail' : function(ctx) {

            ctx
                .tag('i')
                .wrapContent({ elem : 'tail-i', tag : 'i' });

        }

    }

});

})(jQuery);
