/**
 * Модуль для работы с предметной областью Лего: блоки, элементы блоков, модификаторы.
 */

var fs = require('fs'),
    path = require('path'),
    util = require('./util'),
    readFile = util.readFile,
    arrayUnique = util.arrayUnique,
    Lego = exports.Lego = function(levelDir) {
        this.getLevelDir = function() { return (levelDir || 'blocks').replace(/\/+$/, '') };
    };

util.object.update(Lego.prototype, {

    getBlockDir : function(blockName) {
        return this.getLevelDir() + '/' + blockName + '/';
    },

    getElemDir : function(elemName, blockName) {
        return this.getBlockDir(blockName) + elemName + '/';
    },

    getModDir : function(modName, blockName, elemName) {
        return (elemName ? this.getElemDir(elemName, blockName) : this.getBlockDir(blockName)) +
            '_' + modName + '/';
    },

    getExamplesDir : function(blockName, elemName) {
        return (elemName ? this.getElemDir(elemName, blockName) : this.getBlockDir(blockName)) +
            'examples/';
    },

    getBlockFilePrefix : function(blockName) {
        return this.getBlockDir(blockName) + blockName;
    },

    getElemFilePrefix : function(elemName, blockName) {
        return this.getElemDir(elemName, blockName) + blockName + '__' + elemName;
    },

    getModFilePrefix : function(modName, blockName, elemName) {
        return this.getModDir(modName, blockName, elemName) +
            blockName +
            (elemName ? '__' + elemName : '') +
            '_' + modName;
    },

    getExampleFilePrefix : function(exampleName, blockName, elemName) {
        return this.getExamplesDir(blockName, elemName) + exampleName;
    },

    isName : function(s) {
        return /^_{0,2}[a-zA-Z0-9-]+$/.test(s);
    },

    isBlockName : function(s) {
        return /^[bigl]-[a-zA-Z0-9-]+$/.test(s);
    },

    isElemName : function(s) {
        return /^(__)?[a-zA-Z0-9-]+$/.test(s) && s != 'examples'; // костыль про examples (см. LEGO-853)
    },

    isModName : function(s) {
        return /^_[a-zA-Z0-9-]+$/.test(s);
    },

    isExampleName : function(s) {
        return /^[^._][a-zA-Z0-9-_]*$/.test(s);
    },


    getDirs : function(path_) {
        try {
            return fs.statSync(path_).isDirectory()?
                fs.readdirSync(path_)
                    .filter(function(d) {
                        return !(/^\.svn$/.test(d)) && fs.statSync(path.join(path_, d)).isDirectory();
                    })
                    .sort() :
                [];
        } catch (e) {
            return [];
        }
    },

    getFiles : function(path_) {
        try {
            return fs.statSync(path_).isDirectory()?
                fs.readdirSync(path_)
                    .filter(function(f) {
                        return fs.statSync(path.join(path_, f)).isFile();
                    })
                    .sort() :
                [];
        } catch (e) {
            return [];
        }
    },

    getBlockNamesFromDirs : function() {
        return this.getDirs(this.getLevelDir()).filter(this.isBlockName);
    },

    getElemNamesFromDirs : function(blockName) {
        return this.getDirs(this.getBlockDir(blockName))
            .filter(this.isElemName)
            .map(function(n){ return n.replace(/^__/, '') });
    },

    getModNamesFromDirs : function(blockName, elemName) {
        return this.getDirs(elemName ? this.getElemDir(elemName, blockName) : this.getBlockDir(blockName))
            .filter(this.isModName)
            .map(function(n){ return n.replace(/^_/, '') });
    },

    getExampleNamesFromFiles : function(blockName, elemName) {
        return arrayUnique(
                this.getFiles(this.getExamplesDir(blockName, elemName))
                    .filter(function(s) {
                        return !(/.make./.test(s)) && !(/^Makefile$/.test(s)) && !(/.[png|jpg|gif|tt2]$/.test(s));
                        })
                    .map(function(n){ return n.replace(/\..*$/, '') })
                    .filter(this.isExampleName)
            );
    },

    getBlockFromFileSys : function(blockName) {
        var blockFilePrefix = this.getBlockFilePrefix(blockName);
        return {
            name: blockName,
            title: readFile(blockFilePrefix + '.title.txt'),
            desc: readFile(blockFilePrefix + '.desc.wiki'),
            mods: this.getModsFromFileSys(blockName),
            elems: this.getElemsFromFileSys(blockName),
            psd: this.getBlockFilesFromFileSys(blockName, 'psd'),
            examples: this.getExamplesFromFileSys(blockName),
            labels: this.getLabelsFromFileSys(blockName)
        };
    },

    getModsFromFileSys : function(blockName, elemName) {
        var _this = this;
        return this.getModNamesFromDirs(blockName, elemName).map(function(modName){
            var vals = arrayUnique(
                    _this.getFiles(_this.getModDir(modName, blockName, elemName))
                        .map(function(f){
                            //console.log(sys.inspect(f));
                            var valRegex = new RegExp(
                                    blockName +
                                    (elemName ? '__' + elemName : '') +
                                    '_' + modName + '_([^._]+)'
                                );
                            return (f.match(valRegex) || [])[1];
                        })
                        .filter(function(v){ return !!v })
                ),
                modFilePrefix = _this.getModFilePrefix(modName, blockName, elemName);

            return {
                name: modName,
                title: readFile(modFilePrefix + '.title.txt'),
                desc: readFile(modFilePrefix + '.desc.wiki'),
                vals: vals.map(function(v){
                    return {
                        name: v,
                        title: readFile(modFilePrefix + '_' + v + '.title.txt')
                    }
                })
            }
        })
    },

    getElemsFromFileSys : function(blockName) {
        var _this = this;
        return this.getElemNamesFromDirs(blockName).map(function(elemName){
            var elemFilePrefix = _this.getElemFilePrefix(elemName, blockName);
            return {
                name: elemName,
                title: readFile(elemFilePrefix + '.title.txt'),
                desc: readFile(elemFilePrefix + '.desc.wiki'),
                mods: _this.getModsFromFileSys(blockName, elemName),
                examples: _this.getExamplesFromFileSys(blockName, elemName)
            }
        })
    },

    getBlockFilesFromFileSys : function(blockName, type) {
        // это надо переделать так, чтобы получать массив всех доступных файлов
        // возможно, связать с bem-tools?
        var _this = this;
        return this.getFiles(this.getBlockDir(blockName))
                   .filter(function(f){
                            var valRegex = new RegExp(
                                    '.(psd$)'
                                );
                            return f.match(valRegex);
                           })
                   .map(function(f){ return {path: _this.getBlockDir(blockName) + f, file: f} });
    },

    getExamplesFromFileSys : function(blockName, elemName) {
        var _this = this;
        return this.getExampleNamesFromFiles(blockName, elemName)
            .map(function(exampleName){
                var exampleFilePrefix = _this.getExampleFilePrefix(exampleName, blockName, elemName),
                    source = {
                        xml: 'xml',
                        params: 'params.xml',
                        css: 'css',
                        cssie: 'ie.css',
                        js: 'js'
                    };

                for (var k in source) {
                    var file = exampleFilePrefix + '.' + source[k];
                    source[k] = path.existsSync(file) ? file : false;
                };
                source['html'] = '../../tools/eval/eval.xml?' +
                    'lego-xml-file=' + source.xml +
                    (source.params ? '&amp;lego-params-file=' + source.params : '') +
                    (source.css ? '&amp;lego-css-file=/' + source.css : '') +
                    (source.cssie ? '&amp;lego-css-ie-file=/' + source.cssie : '') +
                    (source.js ? '&amp;lego-js-file=/' + source.js : '');

                return {
                    name: exampleName,
                    title: readFile(exampleFilePrefix + '.title.txt'),
                    desc: readFile(exampleFilePrefix + '.desc.wiki'),
                    source: source
                }
            })
    },

    getLabelsFromFileSys : function(blockName) {
        if (path.existsSync(this.getBlockFilePrefix(blockName) + '.labels.json')) {
            try {
                return JSON.parse(readFile(this.getBlockFilePrefix(blockName) + '.labels.json'));
            } catch(e) {
                return {};
            }
        }
        else return {};
    }

});
