#!/usr/bin/env node

var fs = require('fs'),
    vm = require('vm'),
    bemjsonFile = process.argv[2];

fs.readFile(bemjsonFile, 'utf-8', function(err, content) {
    if (err) throw err;

    var bemjson = content;

    try {
        bemjson = vm.runInThisContext(bemjson);
    } catch(e) {
        console.log(bemjson);
        throw e;
    }

    fs.writeFile(
        bemjsonFile,
        '(' + indentOne(prettyOne(bemjson)) + ')',
        function (err) {
            if (err) throw err;
            console.log(bemjsonFile);
        });
});

function prettyBEMHTML(o) {
    var pretty = prettyOne(o)
}

function prettyOne(o, buf) {
    buf || (buf = []);

    if(isSimple(o)) {
        stringify(o, buf);
    } else if(!o) {
    } else if(isArray(o)) {
        var l = o.length,
            subbuf = [],
            i = 0;
        subbuf.push('[');
        while(i < l) prettyOne(o[i++], subbuf);
        subbuf.push(']');
        buf.push(subbuf);
    } else {
        var subbuf = [];

        subbuf.push('{');

        var keys = [ 'block', 'mods', 'elem', 'elemMods', 'mix', 'tag', 'attrs', 'cls', 'js', 'bem' ],
            doneKeys = {},
            k;
        while(k = keys.shift()) {
            o.hasOwnProperty(k) &&
                subbuf.push(prettyOne(o[k], [k + ': ']));
            doneKeys[k] = true;
        }

        for(k in o) {
            if(!o.hasOwnProperty(k) || doneKeys.hasOwnProperty(k) || k == 'content') continue;
            subbuf.push(prettyOne(o[k], [stringify(k) + ': ']));
        }

        var c = simplifyContent(o.content);
        c && subbuf.push(prettyOne(c, ['content: ']));

        subbuf.push('}');
        buf.push(subbuf);
    }

    return buf;
}

function indentOne(o, indent, buf) {
    indent || (indent = 0);
    buf || (buf = []);
    if(isSimple(o)) {
        buf.push(o);
    } else if(!o) {
    } else if(isArray(o)) {
        var l = o.length;
        if(l == 1) {
            buf.push(indentOne(o[0], indent));
        } else if(l == 2) {
            buf.push(indentOne(o[0], indent));
            buf.push(indentOne(o[1], indent));
        } else {
            var start = o.shift(),
                end = o.pop(),
                body = o.map(function(oo) { return indentOne(oo, indent + 1) }),
                short = [ start, body.join(', '), end ].join(' ');

            short.length < 42 ?
                buf.push(short) :
                buf.push(
                    start + indentStr(indent + 1),
                    body.join(',' + indentStr(indent + 1)),
                    indentStr(indent) + end);
        }
    }
    return buf.join('');
}

function indentStr(i, s) {
    return '\n' + Array(Math.max(i, 0) + 1).join(s || '    ')
}

function simplifyContent(c) {
    if(c) {
        if(isSimple(c)) {
            return c
        } else if(isArray(c)) {
            c = c
                .map(function(i) { return simplifyContent(i) })
                .filter(function(i) { return typeof i !== 'undefined' });
            var l = c.length;
            if(l) return l - 1 ? c : c[0];
        } else {
            return c
        }
    }
}


function isArray(o) {
    return toString.call(o) === "[object Array]";
}

function isSimple(o) {
    var t = typeof o;
    return t === 'string' || t === 'number' || t === 'boolean';
}

function stringify(o, buf) {
    var res = JSON.stringify(o)
        .replace(/'/g, "\\'")
        .replace(/\\"/g, '"')
        .replace(/^"(.*)"$/, "'$1'");

    buf && buf.push(res);

    return res;
}
