/*global require */

// увеличен лимит, чтобы при выполнении тестов не ругалось на большое количество запущенных chrome
require('events').EventEmitter.defaultMaxListeners = 20;

var dev = require('direct-dev/lib/index'),
    transporter = require('./enb-techs/tests/transporter'),
    browserStat = require('../browser-stat.js'),
    transporterPlugins = dev.transporterPlugins,
    techs = {
        bem: require('enb-bem-techs'),
        dev: dev.techs,
        enb: {
            provideFile: require('enb/techs/file-provider'),
            fileMerge: require('enb/techs/file-merge'),
            fileCopy: require('enb/techs/file-copy')
        },
        stylus: require('enb-stylus/techs/stylus'),
        borschik: require('enb-borschik/techs/borschik'),
        xjst: {
            bemtree: require('enb-bemtree-utils'),
            bemhtml: require('enb-bemxjst/techs/bemhtml'),
            bemhtmlSchemaValidation: require('./enb-techs/bemhtml'),
            bemjsonToHtml: require('enb-bemxjst/techs/bemjson-to-html')
        },
        chromeTesting: require('enb-headless-chrome-testing'),
        direct: {
            regions: require('./enb-techs/regions-json')
        },
        i18n: {
            mergeKeysets: require('enb-bem-i18n/techs/i18n-merge-keysets'),
            langJs: require('enb-bem-i18n/techs/i18n-lang-js'),
            langJsWithFallback: require('./enb-techs/i18n-lang-js-with-fallback')
        }
    };

module.exports = function(config) {

    switch (process.env.ENB_MODE)
    {
        case 'tests':
            registerSeparatedTests(config);
            break;
        default:
            config.setLanguages(['ru', 'uk', 'by', 'kz', 'en', 'tr']);

            //@heliarian Разнесла всю декларацию зависимостей по модам - с закладкой под дальнейшее создание специального .bemdecl файла
            // в котором содержатся зависимости строго для development-сборки
            config.node('desktop.bundles/direct', function(nodeConfig) {
                nodeConfig.addTargets([
                    '_?.bemtree.xjst.js',
                    '_?.{lang}.js',
                    '_?.css',
                    '_?.ie.css',
                    '?.schema.js',
                    '?.client.schema.js',
                    '?.{lang}.regions.js'
                ]);
            });

            config.node('touch.bundles/direct', function(nodeConfig) {
                nodeConfig.addTargets([
                    '_?.bemtree.xjst.js',
                    '_?.{lang}.js',
                    '_?.css',
                    '?.schema.js',
                    '?.client.schema.js',
                    '?.{lang}.regions.js'
                ]);
            });

            config.node('desktop.bundles/morda', function(nodeConfig) {
                nodeConfig.addTargets([
                    '_?.bemtree.xjst.js',
                    '_?.{lang}.js',
                    '_?.css',
                    '_?.ie.css',
                    '?.schema.js',
                    '?.client.schema.js',
                    '?.{lang}.regions.js'
                ]);
            });

            config.node('desktop.bundles/head', function(nodeConfig) {
                nodeConfig.addTargets([
                    '_?.bemtree.xjst.js',
                    '_?.{lang}.js',
                    '_?.css'
                ]);
            });

            config.mode('development', function() {

                config.node('desktop.bundles/direct', function(nodeConfig) {
                    var levels = getLevels(config, 'desktop', ['dev.blocks']);

                    addCommonTechs(nodeConfig, levels);
                    addMainTechs(nodeConfig, true);
                });

                config.node('touch.bundles/direct', function(nodeConfig) {
                    var levels = getLevels(config, 'touch', ['dev.blocks']);

                    addCommonTechs(nodeConfig, levels);
                    addMainTechs(nodeConfig, true);
                });

                config.node('desktop.bundles/morda', function(nodeConfig) {
                    var levels = getLevels(config, 'desktop', ['dev.blocks']);

                    addCommonTechs(nodeConfig, levels);
                    addMainTechs(nodeConfig, true);
                });

                config.node('desktop.bundles/head', function(nodeConfig) {
                    var levels = getLevels(config, 'desktop');

                    addCommonTechs(nodeConfig, levels, { head: true });
                    addHeadTechs(nodeConfig, true);
                });
            });

            config.mode('production', function() {
                //YENV=production make direct
                config.node('desktop.bundles/direct', function(nodeConfig) {
                    var levels = getLevels(config, 'desktop');

                    addCommonTechs(nodeConfig, levels);
                    addMainTechs(nodeConfig, false);
                });
                //YENV=production make direct-touch
                config.node('touch.bundles/direct', function(nodeConfig) {
                    var levels = getLevels(config, 'touch');

                    addCommonTechs(nodeConfig, levels);
                    addMainTechs(nodeConfig, false);
                });

                //YENV=production make morda
                config.node('desktop.bundles/morda', function(nodeConfig) {
                    var levels = getLevels(config, 'desktop');

                    addCommonTechs(nodeConfig, levels);
                    addMainTechs(nodeConfig, false);
                });

                //YENV=production make head
                config.node('desktop.bundles/head', function(nodeConfig) {
                    var levels = getLevels(config, 'desktop');

                    addCommonTechs(nodeConfig, levels, { head: true });
                    addHeadTechs(nodeConfig, false);
                });

            });
    }
};

/**
 * Технологии для сборки и шапки и основного бандла
 */
function addCommonTechs(nodeConfig, levels, options) {
    var langs = nodeConfig._projectConfig.getLanguages();

    options = options || {};

    // локализационные js
    nodeConfig.addTechs([
        [techs.enb.provideFile, { target: '?.bemdecl.js' }],
        require('enb-bem-techs/techs/files'),
        [require('enb-bem-techs/techs/levels'), { levels: levels }],
        [techs.i18n.mergeKeysets, { lang: 'all' }],
        [techs.i18n.mergeKeysets, { lang: '{lang}' }],
        [techs.i18n.langJs, { target: '?.lang.all.js', lang: 'all' }],
        [techs.i18n.langJsWithFallback, { target: '?.lang.{lang}.js', lang: '{lang}' }],
        [transporter.createTech(), {
            target: '?.const.{lang}.js',
            lang: '{lang}',
            sourceSuffixes: 'js.{lang}',
            applyResult: {
                handler: 'join',
                separator: '\n;\n'
            }
        }],
        [transporter.createTech(), {
            target: '?.merged.js',
            sourceSuffixes: ['js', 'utils.js', 'vm.js'],
            apply: [
                { handler: 'wrap', begin: '/* begin: {relativePath} */\n', end: '\n/* end: {relativePath} */' }
            ],
            applyResult: {
                handler: 'join',
                separator: '\n;\n'
            }
        }],

        // регионы
        [techs.direct.regions, { lang: '{lang}' }]
    ]);

    // клиентский bemhtml
    nodeConfig.addTechs([
        [require('enb-bem-techs/techs/deps-by-tech-to-bemdecl'), { bemdeclFormat: 'deps', target: '?.bemhtml.bemdecl.js', sourceTech: 'js', destTech: 'bemhtml' }],
        [require('enb-bem-techs/techs/deps-old'), { bemdeclFile: '?.bemhtml.bemdecl.js', target: '?.bemhtml.deps.js' }],
        [require('enb-bem-techs/techs/files'), { filesTarget: '?.bemhtml.files', dirsTarget: '?.bemhtml.dirs', depsFile: '?.bemhtml.deps.js' }],
        [techs.enb.fileMerge, {
            target: options.head ? '?.{lang}.pre.js' : '?.{lang}.js',
            lang: '{lang}',
            sources: [
                '?.lang.all.js',
                '?.lang.{lang}.js',
                '?.const.{lang}.js',
                '?.client.bemhtml.js',
                '?.merged.js',
                '?.{lang}.regions.js'
            ]
        }]
    ]);

    // Заменяем в шапочном файле BEM на BEMN
    if (options.head) {
        nodeConfig.addTechs([
            [require('./enb-techs/bemn-replacer'), { target: '?.{lang}.js', lang: '{lang}' }]
        ]);
    }

    // css
    nodeConfig.addTechs([
        [
            techs.stylus, {
                autoprefixer: {
                    browsers: [
                        '> 0.5% in my stats'
                    ],
                    stats: browserStat
                }
            }
        ],
        [require('./enb-techs/css-split')]
    ]);

    // серверный bemhtml + локализационный bemtree.xjst
    nodeConfig.addTechs([
        [
            require('./enb-techs/bemtree-xjst-i18n'),
            {
                langTargets: ['all']
                    .concat(langs)
                    .map(function(lang) {
                        return '?.lang.' + lang + '.js'
                    })
            }
        ]
    ]);
}

/**
 * Технологии для сборки шапки
 */
function addHeadTechs(nodeConfig, dev) {

    nodeConfig.addTechs([
        require('enb-bem-techs/techs/deps-old'),
        [techs.xjst.bemhtml, { sourceSuffixes: 'bemhtml.js', engineOptions: { elemJsInstances: true } }],
        [techs.xjst.bemhtml, { filesTarget: '?.bemhtml.files', target: '?.client.bemhtml.js', sourceSuffixes: 'bemhtml.js', engineOptions: { elemJsInstances: true } }],
        [techs.xjst.bemtree, { engineOptions: { escapeContent: false } }],
        [techs.borschik, { sourceTarget: '?.all.bemtree.xjst.js', destTarget: '_?.bemtree.xjst.js', minify: false }],
        [techs.borschik, { sourceTarget: '?.{lang}.js', destTarget: '_?.{lang}.js', minify: !dev }],
        [techs.enb.fileCopy, { sourceTarget: '?.css', destTarget: '_?.css' }]
    ]);
}

/**
 * Технологии для сборки основного бандла
 */
function addMainTechs(nodeConfig, dev) {

    nodeConfig.addTechs([
        [require('./enb-techs/bemdecl-by-page'), { target: '?.page-bemdecl.js' }],
        [require('enb-bem-techs/techs/deps-old'), { sourceTarget: '?.page-bemdecl.js', bemdeclFile: '?.page-bemdecl.js' }],
        [require('./enb-techs/schema-json')],
        [require('./enb-techs/client-schema-json')], // т. к. bem-builder собирает в продакшен режиме
        [techs.borschik, { sourceTarget: '?.all.bemtree.xjst.js', destTarget: '_?.bemtree.xjst.js', minify: false }],
        [techs.borschik, { sourceTarget: '?.{lang}.js', destTarget: '_?.{lang}.js', minify: !dev }]
    ]);

    if (dev) {
        nodeConfig.addTechs([
            [require('./enb-techs/ctx-schema-json')],
            [require('./enb-techs/bemhtml-validated')],
            [techs.xjst.bemhtmlSchemaValidation, { sourceSuffixes: 'bemhtml.js', engineOptions: { elemJsInstances: true } }],
            [techs.xjst.bemhtmlSchemaValidation, { filesTarget: '?.bemhtml.files', sourceSuffixes: 'bemhtml.js', target: '?.client.bemhtml.js', engineOptions: { elemJsInstances: true } }],
            [techs.xjst.bemtree, { engineOptions: { escapeContent: false } }],
            [techs.enb.fileCopy, { sourceTarget: '?.css', destTarget: '_?.css' }]
        ]);

        nodeConfig.addTargets([
            //это таргет из генерится в результате работы bemhtml-validated
            'b-ctx-validator.bemhtml.js',
            '?.ctx.schema.js'
        ]);
    } else {
        nodeConfig.addTechs([
            // `production: false` для того чтобы в случае ошибки интерфейс падал, а не пытался отрендерить остальные куски, т.к. это может повлечь непредвиденные ошибки (DIRECT-71136)
            [techs.xjst.bemhtml, { engineOptions: { production: false, elemJsInstances: true } }],
            [techs.xjst.bemtree, { engineOptions: { production: false, escapeContent: false } }],
            [techs.xjst.bemhtml, { engineOptions: { production: true, elemJsInstances: true }, filesTarget: '?.bemhtml.files', target: '?.client.bemhtml.js', sourceSuffixes: 'bemhtml.js' }],
            [techs.borschik, { sourceTarget: '?.css', destTarget: '_?.css', minify: true }]
        ]);
    }
}

// region READY

/**
 * Настраивает сборку тестов
 * @param {Object} config - Конфигурация ENB
 */
function registerSeparatedTests(config) {
    var bundlesConfig = require('../bundles.json'),
        targetBlock = process.env.TARGET_BLOCK;

    config.setLanguages(['ru']);

    bundlesConfig.forEach(function(bundle) {
        (!targetBlock || (targetBlock === bundle.block)) && configureBundles(config, bundle);
    });
}


/**
 * Формирует список уровней переопределения
 * @param {Object} config - Конфигурация ENB
 * @param {String} platform - название платформы
 * @param {String[]|Object[]} [appendLevels] - Уровни, которые нужно добавить в конец
 * @param {String[]|Object[]} [prependLevels] - Уровни, которые нужно добавить в начало
 * @returns {String[]} Массив абсолютных путей к заданным папкам уровней переопределения
 */
function getLevels(config, platform, appendLevels, prependLevels ) {
    var levels = {
        touch: [
            { path: 'node_modules/@yandex-lego/ui.i-bem/common.blocks', check: false },
            { path: 'node_modules/islands/common.blocks', check: true },
            { path: 'node_modules/@yandex-lego/ui.islands-deprecated/common.blocks', check: false },
            { path: 'node_modules/@yandex-lego/ui.islands-deprecated/touch.blocks', check: false },
            { path: 'node_modules/@yandex-lego/ui.islands-deprecated/touch-phone.blocks', check: false },
            { path: 'node_modules/bem-history/common.blocks', check: true },
            { path: 'node_modules/islands/touch.blocks', check: true },
            { path: 'node_modules/islands/touch-phone.blocks', check: true },
            { path: 'node_modules/bem-mvc-direct/blocks', check: true },
            { path: 'node_modules/bem-mvc-direct/override.blocks', check: true },
            { path: 'node_modules/bricks/source.blocks', check: true },
            { path: 'common.blocks', check: true },
            { path: 'desktop.blocks', check: true },
            { path: 'touch.blocks', check: true },
            { path: 'bricks.translations', check: true }
        ],
        desktop: [
            { path: 'node_modules/@yandex-lego/ui.i-bem/common.blocks', check: false },
            { path: 'node_modules/islands/common.blocks', check: true },
            { path: 'node_modules/@yandex-lego/ui.islands-deprecated/common.blocks', check: false },
            { path: 'node_modules/@yandex-lego/ui.islands-deprecated/desktop.blocks', check: false },
            { path: 'node_modules/bem-history/common.blocks', check: true },
            { path: 'node_modules/islands/desktop.blocks', check: true },
            { path: 'node_modules/bem-mvc-direct/blocks', check: true },
            { path: 'node_modules/bem-mvc-direct/override.blocks', check: true },
            { path: 'node_modules/calendar/blocks', check: true },
            { path: 'node_modules/bricks/source.blocks', check: true },
            { path: 'node_modules/highcharts', check: true },
            { path: 'node_modules/tableau-iframe/common.blocks', check: true },
            { path: 'node_modules/tableau-iframe/deskpad.blocks', check: true },
            { path: 'node_modules/tableau-iframe/desktop.blocks', check: true },
            { path: 'common.blocks', check: true },
            { path: 'desktop.blocks', check: true },
            { path: 'bricks.translations', check: true }
        ]
    }

    return []
        .concat(prependLevels || [])
        .concat(levels[platform] || [])
        .concat(appendLevels || [])
        .filter(Boolean)
        .map(function(levelPath) {
            return config.resolvePath(levelPath);
        });
}

/**
 * Возвращает список уровней, в которых надо запускать тесты
 * @param {'desktop'|'touch'} platform
 * @return {String[]}
 */
function getTargetLevels(platform) {
    switch (platform) {
        case 'desktop':
            return ['common.blocks', 'desktop.blocks'];
        case 'touch':
            return ['touch.blocks'];
        default:
            throw new Error('Unknown platform');
    }
}

/**
 *
 * @param {Object}          config                  - Конфигурация ENB
 * @param {Object}          bundle                  - Конфигурация бандла
 * @param {String|String[]} bundle.targets          - Конфигурация бандла
 * @param {'desktop'|'touch'} bundle.platform       - Назание платформы
 * @param {String[]}        bundle.entities         - Список БЭМ-сущностей, которые должны попасть в бандл
 * @param {Boolean}         bundle.single           - Собрать общий js бандл с клиентским и серверным кодом
 * @param {Boolean}         bundle.coverage         - Собирать статистику о покрытии тестами
 * @param {String[]}        bundle.appendLevels     - Уровни переопределения, которые нужно добавить в конец списка
 * @param {String[]}        bundle.prependLevels    - Уровни переопределения, которые нужно добавить в начало списка
 */
function configureBundles(config, bundle) {
    var targets = [].concat(bundle.targets || []),
        entities = [].concat(bundle.entities || []),
        inlineImages = true,
        single = bundle.single,
        hasTests = bundle.hasTests,
        needCoverage = bundle.coverage,
        targetLevels = getTargetLevels(bundle.platform),
        sourceLevels = getLevels(config, bundle.platform, bundle.appendLevels, bundle.prependLevels),

        filter = new dev.BlockFilter(
            { targetBlock: bundle.block, targetLevels: targetLevels },
            { rootPath: config.getRootPath() }
        );

    config.node(bundle.path, function(nodeConfig) {

        targets.length && nodeConfig.addTargets(targets);

        nodeConfig.addTechs([
            // bundle endpoint
            entities.length ?
                [techs.dev.devDeclaration, { target: '?.bemdecl.js', entities: entities }] :
                [techs.enb.provideFile, { target: '?.bemdecl.js' }],

            // essential
            [techs.bem.levels, { levels: sourceLevels }],
            [techs.bem.depsOld],
            [techs.bem.files],

            // localization
            [techs.i18n.mergeKeysets, { target: '?.keysets.all.js', lang: 'all' }],
            [techs.i18n.mergeKeysets, { target: '?.keysets.{lang}.js', lang: '{lang}' }],
            [techs.i18n.langJs, { target: '?.lang.all.js', lang: 'all' }],
            [techs.i18n.langJs, { target: '?.lang.{lang}.js', lang: '{lang}' }],
            //регионы
            [techs.direct.regions, { lang: '{lang}' }],

            // const localization
            [transporter.createTech(), {
                target: '?.const.{lang}.js',
                lang: '{lang}',
                sourceSuffixes: 'js.{lang}',
                applyResult: {
                    handler: 'join',
                    separator: '\n;\n'
                }
            }],

            // client js
            [techs.dev.transporter('js', { noCache: needCoverage }), {
                target: '?.merged.js',
                sourceSuffixes: ['js', 'utils.js', 'vm.js'],
                apply: [
                    needCoverage && transporterPlugins.coverage({ filter: filter.vinyl }),
                    transporterPlugins.wrap({ before: '\n/* begin: ${relative} */\n', after: ';\n/* end: ${relative} */\n' })
                ]
            }],

            // client bemhtml
            [techs.bem.depsByTechToBemdecl, {
                target: '?.bemhtml.bemdecl.js',
                sourceTech: 'js',
                destTech: 'bemhtml',
                bemdeclFormat: 'deps'
            }],

            single ?
                // внимание! для объединения деклараций используем технологию mergeDeps, т.к. декларации в формате описания зависимостей
                [techs.bem.mergeDeps, { target: '?.client.bemdecl.js', sources: ['?.bemdecl.js', '?.bemhtml.bemdecl.js'] }] :
                [techs.enb.fileCopy, { target: '?.client.bemdecl.js', source: '?.bemhtml.bemdecl.js' }],

            [techs.bem.depsOld, { bemdeclFile: '?.client.bemdecl.js', target: '?.client.deps.js' }],
            [techs.bem.files, { filesTarget: '?.client.files', dirsTarget: '?.client.dirs', depsFile: '?.client.deps.js' }],

            [techs.xjst.bemhtml, { engineOptions: { elemJsInstances: true } }],
            [techs.xjst.bemhtml, { target: '?.client.bemhtml.js', filesTarget: '?.client.files', engineOptions: { elemJsInstances: true } }],

            [techs.borschik, { sourceTarget: '?.merged.js', destTarget: '?.resolved.js', minify: false }],

            [techs.enb.fileMerge, {
                target: '?.{lang}.js',
                lang: '{lang}',
                sources: [
                    '?.lang.all.js',
                    '?.lang.{lang}.js',
                    '?.const.{lang}.js',
                    '?.client.bemhtml.js',
                    '?.resolved.js',
                    '?.{lang}.regions.js']
            }],

            [techs.stylus, {
                target: '?.css',
                autoprefixer: {
                    browsers: ['> 0.5% in my stats'],
                    stats: browserStat
                },
                url: inlineImages ? 'inline' : undefined
            }],

            [techs.dev.sandbox, { filter: filter }],
            [techs.dev.jsTest, { filter: filter }],
            [techs.dev.devPageBemjson, { target: '?.test.bemjson', type: 'test', js: '?.ru.js', devJs: '?.test.js', css: '?.css' }],
            [techs.dev.devPageBemjson, { target: '?.sandbox.bemjson', type: 'sandbox', js: '?.ru.js', devJs: '?.sandbox.js', css: '?.css' }],
            [techs.xjst.bemjsonToHtml, { target: '?.test.html', bemjsonFile: '?.test.bemjson' }],
            [techs.xjst.bemjsonToHtml, { target: '?.sandbox.html', bemjsonFile: '?.sandbox.bemjson' }],
            hasTests ? [techs.chromeTesting, { html: '?.test.html' }]
                : [techs.dev.emptyTestResult(needCoverage), {
                    sourceSuffixes: ['utils.js', 'vm.js', 'js'],
                    filter: filter
                }]
        ]);
    });
}

// endregion
