/**
 * i18n-lang-js
 * ============
 *
 * Собирает `?.lang.<язык>.js`-файлы на основе `?.keysets.<язык>.js`-файлов.
 *
 * Используется для локализации с фолбеком в JS с помощью BEM.I18N.
 * Если нет перевода для фразы на целевом языке, будет использован один из переводов - "фолбеков"
 *
 * **Опции**
 *
 * * *String* **target** — Результирующий таргет. По умолчанию — `?.lang.{lang}.js`.
 * * *String* **lang** — Язык, для которого небходимо собрать файл.
 *
 * **Пример**
 *
 * ```javascript
 * nodeConfig.addTechs([
 *   [require('./enb-techs/i18n-lang-js-with-fallback'), { lang: '{lang}' }],
 * ]);
 * ```
 */
var path = require('path'),
    tanker = require('enb-bem-i18n/exlib/tanker'),
    enb = require('enb'),
    buildFlow = enb.buildFlow || require('enb/lib/build-flow'),
    asyncRequire = require('enb-async-require'),
    clearRequire = require('clear-require'),
    R = require('ramda'),
    Vow = require('vow');

/**
 * Возвращает функцию, которая возвращает перевод по кейсету и ключу
 * Перевод берется или из data, или, если его в data нет, то по очереди из fallbackDataList
 * Если перевод нигде не найден, возвращает пустую строку
 * @param {Object} data исходные кейсеты с переводами
 * @param {Object} fallbackDataList массив объект с кейсетами для фолбека
 * @returns {Function}
 */
var makeFallbackGetter = function(data, fallbackDataList) {
    return function(keyset, key) {
        var getter = R.path([keyset, key]),
            value = getter(data);

        if (!value) {
            var haveFallback = fallbackDataList.some(function(fallbackData) {
                var fallbackValue = getter(fallbackData);

                if (fallbackValue) {
                    value = fallbackValue;
                    return true;
                } else {
                    return false;
                }
            });

            if (!haveFallback) {
                console.error('Для ключа ' + keyset + ' ' + key + ' не найдено ни одного перевода; ', fallbackDataList.length);
                value = '';
            }
        }

        return value;
    };
};


module.exports = buildFlow.create()
    .name('i18n-lang-js-with-fallback')
    .target('target', '?.lang.{lang}.js')
    .defineRequiredOption('lang')
    .useSourceFilename('keysetsFile', '?.keysets.{lang}.js')
    .useSourceFilename('ruFilename', '?.keysets.ru.js')
    .useSourceFilename('enFilename', '?.keysets.en.js')
    .optionAlias('keysetsFile', 'keysetsTarget')
    .builder(function (keysetsFilename, ruFilename, enFilename) {
        var node = this.node,
            cache = node.getNodeCache(this._target),
            basename = path.basename(keysetsFilename),
            cacheKey = 'keysets-file-' + basename,
            promise;

        if (cache.needRebuildFile(cacheKey, keysetsFilename)) {
            clearRequire(keysetsFilename);
            promise = asyncRequire(keysetsFilename)
                .then(function (keysets) {
                    cache.cacheFileInfo(cacheKey, keysetsFilename);
                    return keysets;
                });
        } else {
            promise = asyncRequire(keysetsFilename);
        }

        // языки для фолбека
        // * -> ru
        // tr -> en -> ru
        var fallbackFiles = R.cond([
            [R.equals('tr'), R.always([enFilename, ruFilename])],
            [R.T, R.always([ruFilename])]
        ])(this._lang);

        return Vow.all([
            asyncRequire(ruFilename), // предполагаем что все ключи всегда есть в русских переводах
            Vow.all(fallbackFiles.map(asyncRequire)),
            promise
        ]).spread(function(allKeysData, fallbackDataList, keysets) {
            var lang = this._lang,
                valueGetter = makeFallbackGetter(keysets, fallbackDataList),
                res = Object.keys(allKeysData).sort().reduce(function (prev, keysetName) {
                    prev.push(this.__self.getKeysetBuildResult(
                        keysetName,
                        allKeysData[keysetName],
                        lang,
                        valueGetter));

                    return prev;
                }.bind(this), []);

            return this.getPrependJs(lang) + res.join('\n\n') + this.getAppendJs(lang);
        }.bind(this));
    })
    .methods({
        getPrependJs: function (lang) {
            return 'if (typeof BEM !== \'undefined\' && BEM.I18N) {';
        },
        getAppendJs: function (lang) {
            return '\n\nBEM.I18N.lang(\'' + lang + '\');\n\n}\n';
        }
    })
    .staticMethods({
        getKeysetBuildResult: function (keysetName, keyset, lang, valueGetter) {
            // для сборки кейсета all нужно использовать enb-bem-i18n/techs/i18n-lang-js
            var res = [];
            res.push('BEM.I18N.decl(\'' + keysetName + '\', {');
            Object.keys(keyset).map(function (key, i, arr) {
                tanker.xmlToJs(valueGetter(keysetName, key), function (js) {
                    res.push('    ' + JSON.stringify(key) + ': ' + js + (i === arr.length - 1 ? '' : ','));
                });
            });
            res.push('}, {\n"lang": "' + lang + '"\n});');
            return res.join('\n');
        }
    })
    .createTech();
