/**
 * @see https://github.yandex-team.ru/lego/tanker-kit/tree/dev/ext/parsers
 */
const path = require('path'),
      bemNaming = require('@bem/naming'),
      originalParser = require(path.join(__dirname, 'javascript-parser.js')),
      otherLangs = ['en', 'tr', 'uk'],
      _ = require('lodash'),
      isString  = _.isString,
      PARAM_RE  = /\{([^{|\s]+)\}/g,
      i18nFilesCache = {},
      escapeValue = str => {
          return str.replace(/\n/, '\\n');
      };

function makeDynamicKey(forms) {
    return [
        '<i18n:dynamic project="tanker" keyset="dynamic" key="plural">',
        '<i18n:count><i18n:param>count</i18n:param></i18n:count>',
        '<i18n:one>'  + (forms[0] || '') +  '</i18n:one>',
        '<i18n:some>' + (forms[1] || '') + '</i18n:some>',
        '<i18n:many>' + (forms[2] || '') + '</i18n:many>',
        '<i18n:none>' + (forms[3] || '') + '</i18n:none>',
        '</i18n:dynamic>'
    ].join('');
}

function makeCanonicalProps(forms) {
    var result = [].concat(forms).map(function(form) {
        return !_.isString(form) ? form : form.replace(PARAM_RE, function(param, value) {
            return '<i18n:param>' + value + '</i18n:param>';
        });
    });

    return result.length > 1 ? result : result[0];
}

/**
 * Флаг "происходит ли сейчас первоначальная загрузка переводов"
 * Нужен для того, чтобы атоматически забирать переводы из i18n-файлов
 * и выгружать не только русские переводы, но и все остальные
 * @param {Boolean} isFirstUpload
 */
const isFirstUpload = false;

module.exports = arg => {
    let result = originalParser(arg),
        newResult = [];

    if (result.length) {
        result.forEach(key => {
            let entityName = path.basename(key.path).replace(/\..+/, ''),
                entity = bemNaming.parse(entityName),
                keyset = entity.block,
                blockDirPath = path.dirname(key.path).split(path.sep).filter(chunk => !chunk.startsWith('_')).join(path.sep),
                isBricksKeyset = /bricks/.test(blockDirPath),
                i18nDir = path.resolve(isBricksKeyset ? ('bricks.translations/' + keyset + '/') : blockDirPath, keyset + '.i18n'),
                params = key.params || {};

            if (key.keyset === 'common-i18n') { //программно генерируемые ключи игнорируем
                return;
            }

            key.comment = params.comment || '';
            key.context = params.context || '';
            key.upload = true;

            if (isFirstUpload) {
                // TODO: потом, если надо, подумать что с этим сделать и сделать
                // DIRECT-72915: Сохранять отдельно переводы для клиента и сервера
                // Если когда-нибудь будете выгружать все переводы в танкер этот код не будет работать
                // потому что мы раскладываем ключи по технологиям, а тут они не раскладывются
                if (!i18nFilesCache[i18nDir]) {
                    i18nFilesCache[i18nDir] = {};
                    ['ru'].concat(otherLangs).forEach(lang => {
                        try {
                            var data = require(path.join(i18nDir, lang + '.js'));
                            i18nFilesCache[i18nDir][lang] = data[key.keyset];
                        } catch (e) {
                            i18nFilesCache[i18nDir][lang] = {};
                        }
                    });
                }
                key.value = escapeValue(i18nFilesCache[i18nDir]['ru'][key.key] || '');
                key.plural = false;
                key.language = 'ru';
                newResult.push(key);

                otherLangs.forEach(lang => {
                    let clone = _.cloneDeep(key);

                    clone.hash = clone.hash + lang;
                    clone.language = lang;
                    clone.value = escapeValue(i18nFilesCache[i18nDir][lang][key.key] || '');
                    isBricksKeyset && (clone.keyset = 'bricks:' + clone.keyset);

                    newResult.push(clone);
                });

            } else {
                key.value = makeCanonicalProps(escapeValue(key.phrase));

                // Если в параметрах ключа есть поля 'some', 'many' или 'none':
                // - Считать ключ склоняемым
                // - Считать первой формой склонения значение ключа, определенное выше
                // - Вторую и третью берем из параметров some и many соответственно
                // - В русском языке "четвертая форма" всегда равна третьей
                if(_.intersection(['some', 'many', 'none'], Object.keys(params)).length) {
                    key.plural = true;
                    key.value = [
                        key.value,
                        params.some && makeCanonicalProps(escapeValue(params.some)) || null,
                        params.many && makeCanonicalProps(escapeValue(params.many)) || null,
                        params.none && makeCanonicalProps(escapeValue(params.none)) ||
                            params.many && makeCanonicalProps(escapeValue(params.many)) || null
                    ];
                }
                else {
                    key.plural = false;
                }

                // Если ключ склоняемый и содержит параметры:
                // - Переводим его в базовую форму
                // - Склонения выражаем через i18n:dynamic
                if (key.plural) {
                    key.plural = false;
                    key.value = makeDynamicKey(key.value);
                }
            }

            // помечаем ключи из bricks особым образом
            isBricksKeyset && (key.keyset = 'bricks:' + key.keyset);

            newResult.push(key);
        });
    }

    return newResult;
};
