(function() {
    BEM.DOM.decl({ block: 'b-banner-preview', modName: 'updatable', modVal: 'yes' }, {
        onElemSetMod: {
            sitelink: {
                valid: {
                    yes: function(link) {
                        this.unbindFrom(link, 'click', this.showAlert);
                    },
                    no: function(link) {
                        this.bindTo(link, 'click', this.showAlert);
                    }
                }
            }
        },

        /**
         * Правила для обработки события click по заголовку баннера
         * @returns {Boolean}
         * @private
         */
        _onTitleClickValidate: function() {
            return !this.model.get('href_model').get('href');
        },

        /**
         * Вывод предупреждения при клике по невалидной ссылке
         * @param {Event} e Jquery.Event
         */
        showAlert: function(e) {
            e.preventDefault();
            BEM.blocks['b-confirm'].alert(iget2('b-banner-preview', 'adres-sayta-ne-ukazan', 'Адрес сайта не указан или указан неверно'));
        },

        /**
         * Задать баннер для отображения
         * @param {Object} modelParams параметры модели баннера
         * @param {String} modelParams.id id инстанса модели
         * @param {String} modelParams.name имя модели
         * @param {String} modelParams.parentId id инстанса родительской модели
         * @param {String} modelParams.parentName имя родительской модели
         */
        setBanner: function(modelParams) {
            if (this.model) {
                this._unbindFromBanner();
                this._unbindFromPhrase();
            }

            if (typeof modelParams.id !== 'undefined') {
                this.model = BEM.MODEL.getOrCreate(modelParams);
                this.groupModel = BEM.MODEL.getOrCreate({ name: modelParams.parentName, id: modelParams.parentId });
            }

            this._bindToBanner();

            !this.params.isEasy && this.model.fields['phrases'] && this._bindToPhrase();

            this.updateView();
        },

        /**
         * Обновить вид баннера
         */
        updateView: function() {

            this._updateTitle(this._getTitleValue());
            this._updateBody(this._getBodyValue());

            this.getMod('type') == 'image' &&
                this.updateImage(
                    this.model.get('mdsGroupId'),
                    this.model.get('image') || 'default',
                    this.model.get('image_type') === 'wide'
                );

            this._changeEvents.href_model.call(this);

            if (this._getSitelinksModel() && !!this.elem('sitelinks').length) {
                this._updateSitelinks();
            }
            if (!this.model.get('has_vcard')) {
                var vcard = this.elem('vcard');
                vcard && this.setMod(vcard, 'visibility', 'hidden');
            }
        },

        /**
         * Возвращает отформатированный заголовок объявления
         * @returns {String}
         */
        _getTitleValue: function() {
            return u.replaceTemplate(
                this.model.get('title'),
                u['b-banner-preview2'].calcTitleTemplateLimit(this.model.get('title')),
                this.randPhrase && u.preview.formatPhrase(this.randPhrase.get('key_words')));
        },

        /**
         * Возвращает отформатированный текст объявления
         * @returns {String}
         */
        _getBodyValue: function() {
            return u.replaceTemplate(
                this.model.get('body'),
                u['b-banner-preview2'].calcBodyTemplateLimit(this.model.get('body')),
                this.randPhrase && u.preview.formatPhrase(this.randPhrase.get('key_words')));
        },

        /**
         * Хранилище обработчиков событий
         */
        _changeEvents: {
            title: function() {
                this._updateTitle(this._getTitleValue());
            },
            body: function() {
                this._updateBody(this._getBodyValue());
            },

            domain: function(e, data) {
                this._updateTitleHref(data.value, 'http://');
                this._updateDomain(data.value);
            },

            href_model: function() {
                var hrefModel = this._getHrefModel();

                if (hrefModel) {
                    this._updateTitleHref(
                        hrefModel.get('href'),
                        hrefModel.get('url_protocol'));

                    this._updateDomain(
                        hrefModel.get('href') ?
                            hrefModel.get('domain') || hrefModel.get('domain_redir') :
                            ''
                    );
                }
            },
            sitelinks: function(e, data) {
                if (data && data.fields && this._getSitelinksModel())
                    data.fields.forEach(function(name) {
                        var index = /^(title|href|url_protocol)(\d)$/g.exec(name),
                            sitelinksModel = this._getSitelinksModel();

                        index = index && index[2];

                        if (index !== null)
                            this._updateSitelink(index, {
                                title: sitelinksModel.get('title' + index),
                                href: sitelinksModel.get('href' + index),
                                url_protocol: sitelinksModel.get('url_protocol' + index)
                            });
                    }, this);
            },
            has_vcard: function(e, data) {
                this.setMod(this.elem('vcard'),
                            'visibility',
                            !this.model.get('has_vcard') ? 'hidden' : 'visible');
            }
        },

        _getSitelinksModel: function() {
            return this.model.get('sitelinks');
        },

        /**
         * Навесить обработчики событий на баннер
         * @returns {*}
         * @private
         */
        _bindToBanner: function() {
            var events = this._changeEvents;

            [
                'domain',
                'body',
                'has_vcard',
                'title'
            ].forEach(function(fieldName) {
                if (this.model.fields[fieldName]) {
                    this.model.on(fieldName, 'change', events[fieldName], this);
                }
            }, this);
            this.model.fields.has_href && this.model.on('has_href href_model', 'change', events['href_model'], this);

            if (this.elem('sitelinks').length) {
                this.model.on('sitelinks', 'change', events.sitelinks, this);
            }

            return this;
        },

        /**
         * Снять обработчики событий с баннера
         * @returns {*}
         * @private
         */
        _unbindFromBanner: function() {
            var events = this._changeEvents;

            this.model
                .un('title', 'change', events['title'], this)
                .un('body', 'change', events['body'], this)
                // todo: слушать только изменения полей
                .un('has_href href_model', 'change', events['href_model'], this)
                .un('sitelinks', 'change', events['sitelinks'], this)
                .un('has_vcard', 'change', events['has_vcard'], this);

            this.model.fields['domain'] && this.model.un('domain', 'change', events['domain'], this);

            if (this.elem('sitelinks').length) {
                this.model.un('sitelinks', 'change', events.sitelinks, this);
            }

            return this;
        },

        /**
         * Колбек изменения фразы: обновление частей превью, где могут участвовать шаблоны
         */
        _onPhraseChange: function() {
            ['title', 'body', 'href_model'].forEach(function(key) {
                this._changeEvents[key].call(this);
            }, this);
        },

        /**
         * Подписка на изменение случайно выбранной ключевой фразы
         */
        _bindToPhrase: function() {
            var group = this.groupModel;

            this.randPhrase = group.getRandomActivePhraseModel();

            if (this.randPhrase) {
                //подписываемся на изменение фразы
                this.randPhrase.on('change', this._onPhraseChange, this);

                this._onPhraseChange(); //и обновляем превью
            } else {
                //если фразы нет, пробуем подписаться еще раз, когда она появится
                group.onFirst('phrasesIds', 'change', this._bindToPhrase, this);
            }
        },

        /**
         * Отписка от изменения случайно выбранной ключевой фразы
         * @returns {BEM.DOM} this
         */
        _unbindFromPhrase: function() {
            if (this.randPhrase) {
                this.randPhrase.un('change', this._changeEvents['href_model'], this);
            } else {
                this.model.fields.phrasesIds && this.groupModel.un('phrasesIds', 'change', this._bindToPhrase, this);
            }

            return this;
        },

        /**
         * Обновить заголовок объявления
         * @param {String} title
         * @private
         */
        _updateTitle: function(title) {
            this.elem('title').html(title || iget2('b-banner-preview', 'zagolovok-obyavleniya', 'Заголовок объявления'));
        },

        /**
         * Обновить ссылку на заголовке
         * @param {String} href
         * @param {String} protocol
         * @private
         */
        _updateTitleHref: function(href, protocol) {
            if (!!u.isUrl(href) && this.model.get('has_href')) {
                protocol = protocol || 'http://';
                href = protocol + href;
                this.elem('title').prop('href', this._formatUrl(href));
            } else {
                this.elem('title').prop('href', '#');
            }
        },

        /**
         * Возвращает модель с данными об урле в превью
         * @returns {BEM.MODEL}
         * @private
         */
        _getHrefModel: function() {
            return this.model.get('href_model');
        },

        /**
         * Клик по заголовку превью
         * @param {Object} e
         * @returns {*}
         * @private
         */
        _onTitleClick: function(e) {
            var href = this._getHrefModel().get('href');

            if (href) {
                if (!u.isUrl(href)) {
                    e.preventDefault();
                    this.showAlert(e);
                }
            } else {
                e.preventDefault();
                this._showVCard();
            }

            return this;
        },

        /**
         * Обновить текст объявления
         * @param {String} body
         * @private
         */
        _updateBody: function(body) {
            this.elem('body').html(body || iget2('b-banner-preview', 'tekst-vashego-obyavleniya-o', 'Текст вашего объявления о рекламе услуги или товара.'));
        },

        /**
         * Обновляет картинку
         * @param {String} mdsGroupId номер группы в Аватарнице
         * @param {String} image hash картинки
         * @param {Boolean} isWide
         */
        updateImage: function(mdsGroupId, image, isWide) {
            this.elem('image')
                .attr('src', image && image != 'default' ?
                    u.getImageUrl({ mdsGroupId: mdsGroupId, hash: image, size: (isWide ? 'wx300' : 'x90') }) :
                    '/data3/desktop.blocks/b-banner-preview/b-banner-preview__image_type_default.png');
        },

        /**
         * Обновить домен
         * @param {String} domain
         * @private
         */
        _updateDomain: function(domain) {
            this.__self.replace(this.findElem('domain'), BEMHTML.apply({
                block: 'b-banner-preview',
                elem: 'domain',
                domain: u.hellipCut(domain, 35) || iget2('b-banner-preview', 'domen', 'домен'),
                hrefHasParams: this._getHrefModel().get('isHrefHasParams'),
                showTooltip: !this.params.isEasy
            }));

            this.setMod(this.findElem('domain'), 'visibility', this.model.get('has_href') ? '' : 'hidden');
        },

        /**
         * Обновить сайтлинк
         * @param {Number} index
         * @param {Object} data
         * @private
         */
        _updateSitelink: function(index, data) {
            if (this.params.checkLinks) {
                var isValid = !!u.isUrl(data.href);

                this.setMod(this.elem('sitelink', 'num', '' + index), 'valid', isValid ? 'yes' : 'no');
            }

            var elem = this.elem('sitelink', 'num', '' + index);

            this.setMod(elem, 'empty', data.title ? '' : 'yes');
            elem
                .attr('href', (data.href ? u.formatHref(data.url_protocol + data.href) : '#'))
                .text(data.title ||
                    iget2('b-banner-preview', 'tekst-ssylki-no-s', 'Текст ссылки № {foo}', {
                        foo: 1 + parseInt(index)
                    }));
        },

        /**
         * Обновить сайтлинки
         * @private
         */
        _updateSitelinks: function() {
            var sitelinksModel = this._getSitelinksModel();

            for (var i = 0; i < CONSTS.SITELINKS_NUMBER; i++) {
                this._updateSitelink(i, {
                    title: sitelinksModel.get('title' + i),
                    href: sitelinksModel.get('href' + i),
                    url_protocol: sitelinksModel.get('url_protocol' + i)
                });
            }
        },

        /**
         * Отформатировать текст для вывода
         * @param {String} text
         * @returns {String}
         * @private
         */
        _getFormatedText: function(text) {
            return BEM.blocks['i-utils']
                .escapeHTML(text)
                .replace(/#(.*)#/, BEMHTML.apply({
                    block: 'b-banner-preview',
                    elem: 'template',
                    tag: 'span',
                    content: '$1'
                }));
        }

    }, {
    });

}());
