u.register({
    'b-banner-preview2': {

        /**
         * Формирует данные визитной карточки из серверных данных баннера
         *
         * @param {Object} bannerData Объект баннера
         * @returns {Object}
         */
        getVCard: function(bannerData) {
            var vcard = bannerData.vcard || {};

            return {
                vcard: !!(bannerData.has_vcard || bannerData.phone ? true : +bannerData.with_phone),
                phone: u['b-banner-preview2'].calcPhone(vcard),
                worktime: vcard.worktime || '',
                city: vcard.city || '',
                metro: vcard.metro || ''
            };
        },

        /**
         * Формирует единый хэш про регионы
         * @param {String} geoIds id-регионов через запятую
         * @param {Array} geoText Названия регионов
         * @param {Object} [geoException] Объект со странами в которых показы невозможны
         * @returns {{names: {Array}, count: Number, exceptions: {Object}}}}
         */
        processGeo: function(geoIds, geoText, geoException) {
            var exceptions = u._.extend({}, geoException);

            // убираем all для того, чтобы
            // не показывалось предупреждение "Объявление не будет показываться!" в блоке `b-group-regions`,
            // т.к. это предупреждение переехало в блок `b-banner-preview2-warnings`
            delete exceptions.all;

            return {
                names: geoText,
                count: geoIds == 0 ?
                    8 : // когда не выбран ни один регион - в поле geo, объекта `groups[n]`, приходит '0',
                        // поэтому отображаем "8 регионов", т.к. объявление будет показываться для всех регионов, а их 8
                    geoIds.split(',').filter(function(id) { return id > 0 }).length || 1,
                exceptions: exceptions
            };
        },

        /**
         * Возвращает имя vm-а по типу объявления (ТГО/РМП...)
         * @param {'text'|'dynamic'|'mobile_content'|'performance'|'mcbanner'|'content_promotion'|'cpm_banner'|'cpm_deals'|'cpm_yndx_frontpage'|'cpm_price'} mediaType
         * @param {String} [viewType]
         */
        getNameByType: function(mediaType, viewType) {
            if (u._.contains(['text', 'mobile_content', 'mcbanner', 'cpm_banner', 'cpm_deals', 'cpm_yndx_frontpage', 'cpm_price'], mediaType) &&
                u._.contains(['base-image', 'campaign-image', 'search-image', 'creative', 'campaign-creative'], viewType)) {

                return 'b-banner-preview2_type_image';
            }

            return {
                text: 'b-banner-preview2_type_text',
                dynamic: 'b-banner-preview2_type_dynamic',
                mobile_content: 'b-banner-preview2_type_mobile-content',
                performance: 'b-banner-preview2_type_performance',
                content_promotion: 'b-banner-preview2_type_content-promotion'
            }[mediaType];
        },

        /**
         * Формирует url сайтлинка
         * @param {String} url ссылка без протокола
         * @param {String} protocol протокол ссылки
         * @param {Boolean} isOpenStat
         */
        formatSitelinkUrl: function(url, protocol, isOpenStat) {
            var normalizedlUrl = u.stripHttp(url || ''),
                normalizedProtocol = protocol ? protocol.replace('://', '') : 'http';

            return u.formatHref(normalizedProtocol + '://' + normalizedlUrl, isOpenStat);
        },

        /**
         * Формирует отображаемую ссылку для превью баннера
         * @param {String} domain
         * @param {String} displayHref
         * @param {String} [phrase] если ссылка содержит шаблон,
         * то он будет заменен на эту фразу в соответствии с правилами замены
         * @param {Number} [maxLength] максимальное количество символов в результирующей ссылке
         * @returns {String}
         */
        getDomainWithDisplayHref: function(domain, displayHref, phrase, maxLength) {
            domain = domain || iget2('b-banner-preview2', 'domen', 'домен');

            displayHref = u.replaceTemplate(
                displayHref || '',
                u.consts('MAX_DISPLAY_HREF_LENGTH'),
                u.preview.prettifyDisplayHref(phrase),
                null);

            return u.hellipCut(displayHref ? domain + '/' + displayHref : domain, maxLength || 35, '...');
        },

        /**
         * Возвращает поля, необходимые для построения визитки на клиенте
         * @returns {Array}
         */
        getVCardFields: function() {
            return [
                'cid', 'bid', 'title', 'body', 'url', 'rating', 'domain', 'displayHref', 'vcard', 'image', 'image_type',
                'callouts', 'sitelinks', 'phone', 'ext', 'worktime', 'flags', 'flagsSettings', 'geo', 'country',
                'country_code', 'city', 'city_code', 'street', 'house', 'metro', 'auto_bounds', 'phrase', 'name',
                'contactperson', 'build', 'apart', 'auto_point', 'auto_precision', 'manual_point', 'manual_bounds',
                'contact_email', 'im_client', 'im_login', 'extra_message', 'org_details_id', 'ogrn'
            ];
        },

        /**
         * Считает суммарную длину символов связки телефон + время работы + метро + город
         * @param {Object} data
         * @param {Number} worktimeMaxLength максимальная длина, начиная с которой обрезается worktime
         * @returns {Number}
         */
        calcContactsTotalLength: function(data, worktimeMaxLength) {
            if (!data.vcard) return 0;

            var phone = u['b-banner-preview2'].calcPhone(data),
                worktime = u.hellipCut(u.worktime(data.worktime), worktimeMaxLength),
                metro = data.metro ? u['b-banner-preview2'].calcMetroString(data.metro) : '';

            return (phone + data.city + worktime + metro).length;
        },

        /**
         * Формирует строку с телефоном на основании пришедших данных визитки
         * @param {Object} vcard данные визитки
         * @returns {String}
         */
        calcPhone: function(vcard) {
            if (!vcard || !(vcard.country_code && vcard.city_code && vcard.phone)) return '';

            return vcard.country_code + ' (' + vcard.city_code + ') ' + vcard.phone +
                (vcard.ext ? ' ' + iget2('b-banner-preview2', 'dob', 'доб. ') + vcard.ext : '');
        },

        /**
         * Формируем строку с метро для показа в превью
         * @param {String} metro
         * @returns {String}
         */
        calcMetroString: function(metro) {
            return iget2('b-banner-preview2', 'm-namesubwaystation-stanciya-metro', 'м. {nameSubwayStation}', {
                nameSubwayStation: function() {
                    return u.getNameSubwayStation(metro);
                }(iget2('b-banner-preview2', 'm-namesubwaystation-stanciya-metro-param-nameSubwayStation', 'Станция метро')),

                context: 'nameSubwayStation - Станция метро'
            });
        },

        /**
         * Специальный метод, возвращающий данные pcode для совсем пустого объявления
         * Используется на массовых действиях для видео
         * @returns {{type: string, ads: *[]}}
         * @private
         */
        getEmptyDataToPcode: function() {
            return {
                type: u.pcode.getDefaultViewType(),
                ads: [
                    {
                        // партнерский код требует обязательный параметр adId, который допустимо заменить фэйковым
                        adId: String(+new Date()),
                        title: iget2('b-banner-preview2', 'zagolovok-obyavleniya', 'Заголовок объявления'),
                        body: iget2('b-banner-preview2', 'tekst-vashego-obyavleniya-o', 'Текст вашего объявления о рекламе услуги или товара.'),
                        url: 'javascript:///void(0)', // jshint ignore:line
                        domain: '',
                        punyDomain: '',
                        favicon: 'https://favicon.yandex.net/favicon//',
                        vcardUrl: '',
                        callUrl: '',
                        age: '',
                        warning: '',
                        sitelinks: '',
                        dynamic_disclaimer: '0',
                        banner_flags: '',
                        lang: u.consts('lang'),
                        linkTail: '',
                        debug: '',
                        addInfo: undefined
                    }
                ]
            };
        },

        buildImagesDataForDNA: function(ctxData) {
            var image;

            if (ctxData.image) {
                image = {
                    height: ctxData.imageHeight,
                    imageHash: ctxData.image,
                    url: u.getImageUrl({
                        mdsGroupId: ctxData.mdsGroupId,
                        hash: ctxData.image,
                        // картинку преобразуем так, чтобы высота не превышала 300 пикселей для того,
                        // чтобы она быстрее загружалась
                        size: 'y300'
                    }),
                    width: ctxData.imageWidth
                }
            }

            return image;
        },

        /**
         * Преобразует данные для PCODE баннеров для передачи DNA-блоку BannerPreview
         * @param {Object} previewBlock
         */
        getDataToDNAPreview: function(previewBlock) {
            var ctxData = previewBlock.getCtxData(),
                bannerType = previewBlock.getMod('type'),
                isMobileContent = bannerType === 'mobile-content',
                bannerHasTemplate =
                u.hasTemplateSymbols(ctxData.title) || (ctxData.titleExt && u.hasTemplateSymbols(ctxData.titleExt)) || u.hasTemplateSymbols(ctxData.body);

            return {
                advType: isMobileContent ? 'MOBILE_CONTENT' : 'TEXT',
                id: ctxData.bid,
                group: {
                    campaign: {
                        strategy: {

                        }
                    }
                },
                title: u['b-banner-preview2'].replaceTemplateSimple(ctxData.title),
                titleExtension: ctxData.titleExt ? u['b-banner-preview2'].replaceTemplateSimple(ctxData.titleExt) : undefined,
                body: u['b-banner-preview2'].replaceTemplateSimple(ctxData.body),
                image: u['b-banner-preview2'].buildImagesDataForDNA(ctxData),
                domain: ctxData.domain,
                linkTail: '',
                flags: ctxData.flags,
                additionsFlags: {},
                href: ctxData.url,
                bannerHasTemplate: bannerHasTemplate,
            }
        },

        /**
         * Преобразует входные данные b-banner-preview2 под формат партнерского кода
         * @param {Object} previewBlock
         * @returns {DataToPcode}
         * @private
         */
        getDataToPcode: function(previewBlock) {
            var ctxData = previewBlock.getCtxData(),
                ctxDataWithOutShowPrice = u._.extend({}, ctxData, { showPrice: false }),
                currencySymbol = (u.getCurrency(ctxData.currency) || {}).currency_sign || ctxData.currency;

            // ветвление по типу баннера организовано тут, потому что есть договоренность DIRECT-51172
            // про отказ от модификаторов на элементах, которые могут быть перерисованы при изменениях в моделях
            switch (previewBlock.getMod('type')) {
                case 'mobile-content':
                    return {
                        type: '240x400',
                        ads: [
                            // в РМП возрастную метку ставит пользователь, она может быть и у нового баннера
                            // в остальных типах кампаний метка приходит с модерации, в новом баннере ее нет
                            u._.extend(u['b-banner-preview2']
                                .buildCommons(u._.extend(ctxData, { canHaveAgeLabel: true })), {
                                    // для превью в Директе ширина картинки не должна превышать 220px,
                                    // чтобы она полностью умещалась в контейнер превью и не обрезалась (DIRECT-52086)
                                    images: u['b-banner-preview2'].buildImagesData(
                                        ctxData,
                                        ctxData.imageWidth > 220 ? (220 / ctxData.imageWidth) : 1
                                    ),
                                    addInfo: {
                                        type: 'mobile-app',
                                        rating: ctxData.showRating ? ctxData.rating : '',
                                        // DIRECT-61985: для корректного отображения цены нужно передать след 3 параметра
                                        // price_currency_code - трехбуквенный ISO-код валюты
                                        // price_currency_symbol - символ валюты, который и будет отображаться в объявлении
                                        // при этом на уровне партнерского кода символ рубля переопределяется на '.р'
                                        price: ctxData.showPrice ? ctxData.price : '',
                                        price_currency_code: ctxData.showPrice ? ctxData.currency : '',
                                        price_currency_symbol: ctxData.showPrice ? currencySymbol : '',
                                        review_count: ctxData.showRatingVotes ? ctxData.ratingVotes : '',
                                        icon: ctxData.showIcon ? u['b-banner-preview2'].buildIconsData(ctxData) : [],
                                        // для DIRECT-59297 здесь нужно получить только название кнопки без цены
                                        call_to_action: u['b-banner-preview2']
                                            .getActionButtonText(ctxDataWithOutShowPrice),
                                        call_to_action_url: ctxData.url || '#',
                                        warning: u['b-banner-preview2'].prettifyDisclaimer(ctxData.disclaimer)
                                    },
                                    title: ctxData.title ?
                                        // заменяет шаблон на элемент b-banner-preview__template
                                        // без экранирования кавычек
                                        u.replaceTemplate(
                                            ctxData.title,
                                            u.consts('NEW_MAX_TITLE_LENGTH_MOBILE'),
                                            ctxData.phrase,
                                            null,
                                            true
                                        ) :
                                        iget2('b-banner-preview2', 'zagolovok-obyavleniya', 'Заголовок объявления'),
                                    body: ctxData.body ?
                                        u.replaceTemplate(
                                            ctxData.body,
                                            u['b-banner-preview2'].calcLimitWithNarrows(
                                                ctxData.body,
                                                +u.consts('MAX_BODY_LENGTH_MOBILE'),
                                                u.consts('NARROW_SYMBOLS')
                                            ),
                                            ctxData.phrase,
                                            null,
                                            true
                                        ) :
                                        iget2('b-banner-preview2', 'tekst-vashego-obyavleniya-o', 'Текст вашего объявления о рекламе услуги или товара.'),
                                    domain: ctxData.publisherDomain,
                                    punyDomain: ctxData.publisherDomain
                                })
                        ]
                    };
                default:
                    return {
                        type: u.pcode.getDefaultViewType(),
                        ads: [
                            u._.extend(u['b-banner-preview2'].buildCommons(ctxData), {
                                images: u['b-banner-preview2'].buildImagesData(ctxData)
                            })
                        ]
                    };
            }
        },

        /**
         * Формируем название для кнопки действия в превью
         * @param {Object} data
         * @param {String} data.actionType - Тип действия
         * @param {String} data.currency код валюты
         * @param {Number} data.price - Цена приложения
         * @param {Boolean} data.showPrice - Флаг отображения цены
         * @returns {String}
         */
        getActionButtonText: function(data) {
            var price = data.price && data.showPrice ?
                ' – ' + u.currencies.formatSum(data.currency, data.price) :
                '';

            return {
                download: iget2('b-banner-preview2', 'download-zagruzit', 'Загрузить') + price,
                get: iget2('b-banner-preview2', 'get-skachat', 'Скачать') + price,
                install: iget2('b-banner-preview2', 'ustanovit', 'Установить') + price,
                more: iget2('b-banner-preview2', 'podrobnee', 'Подробнее') + price,
                open: iget2('b-banner-preview2', 'otkryt', 'Открыть') + price,
                update: iget2('b-banner-preview2', 'obnovit', 'Обновить') + price,
                play: iget2('b-banner-preview2', 'igrat', 'Играть') + price,
                buy: data.price ? iget2('b-banner-preview2', 'kupit', 'Купить') + price : iget2('b-banner-preview2', 'besplatno', 'Бесплатно')
            }[data.actionType];
        },

        /**
         * Возвращает данные для передачи в партнерский код по умолчанию
         * @param {Object} ctxData
         * @returns {Object}
         * @private
         */
        buildCommons: function(ctxData) {
            var domain = ctxData.domain,
                flags = ctxData.flags || {},
                adWarnings = u.consts('AD_WARNINGS'),
                ageWarning = flags.age ? flags.age + '+' : undefined,
                warnings = Object.keys(flags).reduce(function(result, key) {
                    key !== 'age' && adWarnings[key] &&
                    (result += key === 'baby_food' ?
                        iget2(
                            'b-banner-preview2',
                            'baby-food',
                            'Проконсультируйтесь со специалистом. Для питания детей с {month} месяца.',
                            {
                                count: flags[key],
                                some: 'Проконсультируйтесь со специалистом. Для питания детей с {month} месяцев.',
                                many: 'Проконсультируйтесь со специалистом. Для питания детей с {month} месяцев.',
                                month: flags[key]
                            }
                        ) :
                        iget2('common-i18n', adWarnings[key].long_text || '', flags[key])
                    );

                    return result;
                }, ''),
                disclaimer = ctxData.disclaimer || '',
                isNewBanner = ctxData.isNewBanner,
                valuedFlags = u._.reduce(adWarnings, function(res, value, key) {
                    !!value.variants && res.push(key);

                    return res;
                }, []),
                turbolandingUrl = (ctxData.turbolanding || {}).href,
                organizationUrl = (ctxData.organization || {}).profileUrl,

                pcodeFlags = u._.reduce(ctxData.flags, function(res, value, key) {
                    res !== '' && (res += ',');
                    res += key;
                    // если есть варианты, конкатенируем с ними
                    (valuedFlags.indexOf(key) !== -1) && (res += '_' + value);

                    return res;
                }, '');

            return {
                // партнерский код требует обязательный параметр adId, который допустимо заменить фэйковым
                adId: String(+new Date()),
                title: ctxData.title ?
                    // заменяет шаблон на элемент b-banner-preview__template
                    // без экранирования кавычек
                    u.replaceTemplate(
                        ctxData.title,
                        this.calcTitleTemplateLimit(ctxData.title),
                        ctxData.phrase,
                        null,
                        true
                    ) :
                    iget2('b-banner-preview2', 'zagolovok-obyavleniya', 'Заголовок объявления'),
                body: ctxData.body ?
                    u.replaceTemplate(
                        ctxData.body,
                        this.calcBodyTemplateLimit(ctxData.body),
                        ctxData.phrase,
                        null,
                        true
                    ) :
                    iget2('b-banner-preview2', 'tekst-vashego-obyavleniya-o', 'Текст вашего объявления о рекламе услуги или товара.'),
                url: ctxData.url || turbolandingUrl || organizationUrl || 'javascript:///void(0)', // jshint ignore:line
                domain: domain,
                punyDomain: domain,
                favicon: 'https://favicon.yandex.net/favicon/' + ctxData.domain + '/',
                // мы не можем сделать e.preventDefault(), pcode просит посылать в таких случаях javascript:///void(0)
                vcardUrl: (ctxData.vcard && !organizationUrl) ? 'javascript:///void(0)' : '', // jshint ignore:line
                callUrl: '',
                age: !isNewBanner || ctxData.canHaveAgeLabel ?
                    ageWarning :
                    '',
                warning: isNewBanner ?
                    '' :
                    disclaimer ?
                        u['b-banner-preview2'].prettifyDisclaimer(disclaimer) :
                        warnings,
                sitelinks: ctxData.sitelinks,
                // для отображения динамического дисклеймера
                // https://st.yandex-team.ru/DIRECT-70375
                // непустой дисклеймер превращаем в строку "1"/"0"
                dynamic_disclaimer: (+(!!disclaimer.length)).toString(),
                banner_flags: pcodeFlags,
                lang: ctxData.lang || u.consts('lang'),
                linkTail: '',
                debug: '',
                addInfo: this.buildAddInfo(ctxData)
            };
        },

        /**
         * Возвращает дополнительные данные
         * @param {Object} ctxData
         * @return {*}
         */
        buildAddInfo: function(ctxData) {
            var callouts = ctxData.callouts;

            if (callouts && callouts.length) {
                return {
                    type: 'callouts',
                    callouts_list: callouts.map(function(callout) {
                        return callout.callout_text;
                    })
                }
            } else {
                return undefined;
            }

        },

        /**
         * Возвращает данные об изображении, либо undefined, если нет ни изображения не превью
         * @param {Object} ctxData
         * @param {Number} scale коэффициент масштабирования изображения
         * @returns {Array[]|undefined}
         * @private
         */
        buildImagesData: function(ctxData, scale) {
            var images;

            scale = scale || (300 / ctxData.imageHeight);

            if (ctxData.image) {
                images = [
                    [
                        u.getImageUrl({
                            mdsGroupId: ctxData.mdsGroupId,
                            hash: ctxData.image,
                            // картинку преобразуем так, чтобы высота не превышала 300 пикселей для того,
                            // чтобы она быстрее загружалась
                            size: 'y300'
                        }, { fullPath: true }),
                        ctxData.imageWidth * scale,
                        ctxData.imageHeight * scale
                    ]
                ];
            }

            return images;
        },

        /**
         * Возвращает данные об иконке
         * @param {Object} ctxData
         * @returns {Array[]}
         * @private
         */
        buildIconsData: function(ctxData) {
            var icons = [];

            u._.forOwn(u.preview.getPCodeIconSizes(), function(sizes, suffix) {
                icons.push([
                    ctxData.icon ? ctxData.icon + '-' + suffix : u.preview.getEmptyIcon(),
                    sizes.width,
                    sizes.height
                ]);
            });

            return icons;
        },

        /**
         * DIRECT-68417: убрать из превью видео дополнения символ &#8203
         * В прошлом добавлял мягкие переносы после символа '/'
         * По хорошему необходимо выпилить функцию
         * @param {String} text
         */
        prettifyDisclaimer: function(text) {
            return text || '';
        },

        /**
         * Фильтрует флаги с одинаковыми родительскими категориями
         * (например med_equipment, med_services, medicine, pharmacy)
         * @param {Object} flags
         * @returns {Object}
         */
        filterFlagsWithSameParent: function(flags) {
            flags = flags || {};

            var adWarnings = u.consts('AD_WARNINGS'),
                parents = {};

            return Object.keys(flags || {}).reduce(function(result, key) {
                var parentKey = u._.get(adWarnings, key + '.parent');

                if (parentKey) {
                    if (!parents[parentKey]) {
                        result[key] = flags[key];

                        parents[parentKey] = true;
                    }
                } else {
                    result[key] = flags[key];
                }

                return result;
            }, {});
        },

        /**
         * Подсчет "узких" (бесплатных) символов с учетом лимита
         * Если символов больше, то возвращается переданный лимит
         * @param {String} str Исходная строка
         * @param {Number} limit Лимит
         * @param {String} narrowSymbols
         * @returns {Number}
         */
        calcLimitWithNarrows: function(str, limit, narrowSymbols) {
            // strWithoutTemplate - строка в которой шаблон (#текст#) заменен нулями
            // для корректного подсчета символов, поскольку внутри шаблона бесплатных символов нет
            var strWithoutTemplate = (str || '').replace(/\#([^\#]*?)\#/i, function(match, content) {
                    return Array(content.length + 1).join('0');
                }),
                narrowLimit = u.consts('MAX_NUMBER_OF_NARROW_CHARACTERS'),
                narrowSymbolsRegExp = new RegExp('([' + narrowSymbols + '])', 'g'),
                strNarrowLength = (strWithoutTemplate.match(narrowSymbolsRegExp) || '').length;

            return u._.sum([
                limit,
                (strNarrowLength > narrowLimit ? narrowLimit : strNarrowLength)
            ]);
        },

        /**
         * Подсчет "узких" (бесплатных) символов с учетом лимита для заголовка
         * Если символов больше, то возвращается переданный лимит
         * @param {String} title Заголовок 1
         * @returns {Number}
         */
        calcTitleTemplateLimit: function(title) {
            return this.calcLimitWithNarrows(
                title,
                u.consts('NEW_MAX_TITLE_LENGTH'),
                ''
            );
        },

        /**
         * Подсчет "узких" (бесплатных) символов с учетом лимита для заголовка
         * Если символов больше, то возвращается переданный лимит
         * @param {String} [titleExt] Заголовок 2
         * @returns {Number}
         */
        calcTitleExtTemplateLimit: function(titleExt) {
            return this.calcLimitWithNarrows(titleExt, u.consts('MAX_TITLE_EXTENSION_LENGTH'), u.consts('NARROW_SYMBOLS'));
        },

        /**
         * Подсчет "узких" (бесплатных) символов с учетом лимита для текста объявления
         * Если символов больше, то возвращается переданный лимит
         * @param {String} body
         * @returns {Number}
         */
        calcBodyTemplateLimit: function(body) {
            return this.calcLimitWithNarrows(body, +u.consts('MAX_BODY_LENGTH'), u.consts('NARROW_SYMBOLS'));
        },

        /**
         *
         * @param {string} str
         * @returns {string}
         */
        replaceTemplateSimple: function(str) {
            return str
                .split(/(#[^#]*#)/)
                .map(function(chunk) {
                    var matches = chunk.match(/#([^#]*)#/);

                    return matches ? matches[1] : chunk;
                })
                .join('')
        },

        /**
         * Формирует данные баннера для отправки в конструктор креативов через postMessage
         * @param {Object} ctxData
         * @return {Object|undefined}
         */
        getVideoPreviewPostMessage: function(ctxData) {
            var adType = ctxData.adType;

            switch (adType) {
                // для `cpc_video` заголовок, описание, домен и возрастная метка задаётся в конструкторе
                // но title и body нужно отправлять, чтобы бэк не подставлял дефолты про путешествие в космос
                // для тех креативов, которые созданы без текстов
                case 'cpc_video':
                    return u._.extend(
                        {},
                        u._.omit(
                            u['b-banner-preview2'].buildCommons(ctxData),
                            ['punyDomain', 'domain']
                        ),
                        {
                            title: '',
                            body: ''
                        }
                    );

                case 'cpm_banner':
                    return u._.omit(
                        u['b-banner-preview2'].buildCommons(ctxData),
                        ['title', 'body', 'punyDomain', 'domain']
                    );

                default:
                    return undefined;
            }
        }
    }

});
