BEM.DOM.decl({ block: 'b-banners-search', elem: 'form' }, {

    /**
     * @typedef {Object} BannerMode - описывает тип поиска
     * Используется для определения режима работы инпута поиска по объявлениям
     *
     * @property {String} value - тип поиска (используется в radio-button)
     * @property {String} text - название типа поиска (используется в radio-button)
     * @property {String} label - название типа поиска для инпута
     * @property {String} type - тип значения, которое должно быть введено пользователем при поиске в этом типе
     * @property {String} [errorText] - сообщение об ошибке, которое будет выведено при несоответствии типа введенного значения
     * @property {Boolean} [checked] - изначально выбранный вариант (используется в radio-button)
     */

    onSetMod: {
        /**
         * @property {Object} params
         * @property {Array<BannerMode>} bannerModes - массив возможных типов поиска
         */
        js: function() {
            this._input = this.findBlockOn('input-value', 'input');
            this._searchBy = this.findBlockOn('search-by', 'radio-button');
            this._popup = this.findBlockOutside('popup');

            this._inputValues = {};
            this._currentSearchKey = this._searchBy.val();
            this._checkAndShowLabel(this._currentSearchKey);

            this._subMan = BEM.create('i-subscription-manager')
                .on(this._input.elem('control'), 'keypress', this._onKeyPress.bind(this))
                .on(this._input, 'focus', this._clearInputErrorMessage, this)
                .on(this._searchBy, 'change', this._onSearchByChange, this);

            this._popup && this._subMan.on(this._popup, 'show', function() {
                this._input.setMod('focused', 'yes');

            }, this)
        }
    },

    /**
     * Убирает сообщение об ошибке в инпуте
     *
     * @private
     */
    _clearInputErrorMessage: function() {
        this._getTooltip().delMod('shown');
    },

    /**
     * Вызывается при смене типа поиска по объявлению, меняет состояния инпута подгоняя под конкретный тип
     *
     * @param {Event} e
     * @private
     */
    _onSearchByChange: function(e) {
        var nextType = e.block.val(),
            nextMode = u._.find(this.params.bannerModes, { value: nextType });

        this._clearErrorMessages();

        this._prepareInput(nextMode);

        this._input.setMod('focused', 'yes');
    },

    /**
     * Преобразует поле ввода для конкретного типа поиска
     *
     * @param {BannerMode} nextMode - применяемый тип
     * @private
     */
    _prepareInput: function(nextMode) {
        this._checkAndShowLabel(nextMode.value);
        this._changeInputValue(nextMode.value);
        this._input.elem('control').attr('placeholder', nextMode.label);
    },

    /**
     * Очищает форму и блок от ошибочных сообщений
     *
     * @private
     */
    _clearErrorMessages: function() {
        this._clearInputErrorMessage();
        this.getParent()._hideNotFoundMessage();
    },

    /**
     * Проверяет по типу, требуется ли показть надпись слева от инпута и показывает
     *
     * @param {String} type - тип поиска
     * @private
     */
    _checkAndShowLabel: function(type) {
        this.getParent().setMod('show-label', type == 'num' ? 'yes' : '');
    },

    /**
     * Сохраняет текущее значение инпута для текущего типа и пытается восстановить для следующего
     *
     * @param {String} nextType
     * @private
     */
    _changeInputValue: function(nextType) {
        this._inputValues[this._currentSearchKey] = this._input.val();

        this._input.val(this._inputValues[nextType] || '');

        this._currentSearchKey = nextType;
    },

    /**
     * Обработка нажатия на enter в поле поиска
     *
     * @param {Event} e
     * @private
     */
    _onKeyPress: function(e) {
        if (e.keyCode == 13) {
            e.preventDefault();
            e.stopPropagation();
            this.getParent()._getBannersCount();
        }
        this.getParent()._hideNotFoundMessage();
    },

    /**
     * Показывает сообщение ошибки в инпуте
     *
     * @param {String} message - текст сообщения
     * @private
     */
    _showInputError: function(message) {
        this._getTooltip()
            .setOwner(this._input)
            .setContent(message);

        this._getTooltip().setMod('shown', 'yes');
    },

    /**
     * Возвращает инстанс блока подсказки
     *
     * @returns {BEM.DOM}
     * @private
     */
    _getTooltip: function() {
        return this._tooltip || (this._tooltip = $(BEMHTML.apply({
            block: 'tooltip',
            mods: { size: 's', theme: 'error', autoclosable: 'yes' },
            mix: {
                block: 'b-banners-search',
                elem: 'input-tooltip'
            }
        })).bem('tooltip'));
    },

    /**
     * Валидирует введеный в инпут критерий
     *
     * @return {boolean}
     */
    validate: function() {
        var bannerModes = this.params.bannerModes,
            error,
            val = this._input.val(),
            mode = u._.find(bannerModes, { value: this._searchBy.val() });

        if (mode.type == 'number' && !val.match(/\d+/)) {
            error = mode.errorText || iget2('b-banners-search', 'ispolzuyte-tolko-cifry', 'Используйте только цифры');
        } else if (mode.type == 'string' && !val.match(/.+/)) {
            error = iget2('b-banners-search', 'vy-dolzhny-vvesti-tekst', 'Вы должны ввести текст');
        }

        error && this._showInputError(error);

        return !error;
    },

    /**
     * Отправляет форму
     */
    submit: function() {
        this.findBlockInside('b-form').submit();
    },

    /**
     * Уничтожает блок
     * @returns {BEM.DOM}
     */
    destruct: function() {
        this._subMan.dispose();
        this._getTooltip() && this._getTooltip().destruct();

        return this.__base.apply(this, arguments);
    }

})
