/**
 * @fires b-callouts-selector#change Изменение выбора уточнений
 */

/**
 * Триггерит событие удаления уточнения в channel callouts
 * @event destroy
 * @type {Object}
 * @property {String} id - id уточнения
 */

BEM.DOM.decl('b-callouts-selector', {

    onSetMod: {

        js: function() {
            this._model = BEM.MODEL.create(this._getModelName());
            this._subMan = BEM.create('i-subscription-manager');

            this._bChooser = this.findBlockOn('available-items', 'b-chooser');
            this._inputSearchAdd = this.findBlockOn('search-add-input', 'input');
            this._buttonAddNew = this.findBlockOn('new-item-button', 'button');

            this._initEvents();
        },

        loading: {

            yes: function() {
                BEM.DOM.append(this.elem('items'), BEMHTML.apply({
                    block: this.__self.getName(),
                    elem: 'loader'
                }));
            },

            '': function() {
                BEM.DOM.destruct(this.findElem('loader'));
            }

        },

        selection: {

            'can-not-be': function() {
                this._setSelectionMessage(iget2('b-callouts-selector', 'net-ni-odnogo-utochneniya', 'Нет ни одного уточнения'));
            },

            empty: function() {
                this._setSelectionMessage(iget2('b-callouts-selector', 'nichego-ne-vybrano', 'Ничего не выбрано'))
            },

            same: function() {
                this._setSelectionMessage('');
            }

        }
    },

    onElemSetMod: {

        search: {

            loading: function(elem, modName, modVal) {
                this._inputSearchAdd.setMod('disabled', modVal);
                this._buttonAddNew.setMod('disabled', modVal);
                this._getNewItemSpin().setMod('progress', modVal);
            }

        }

    },

    /**
     * Возвращает имя view-модели
     * @returns {String}
     * @private
     */
    _getModelName: function() {
        return 'b-callouts-selector';
    },

    /**
     * Подписывает блок на события
     * @private
     */
    _initEvents: function() {
        var linkClearAll = this.findBlockOn('clear-all', 'link');

        this._subMan.wrap(this._model)
            .on('list', 'add', this._onCreateItem, this)
            .on('list', 'change', this._onChangeItem, this)
            .on('list', 'remove', this._onRemoveItem, this)
            .on('selectedTextLength', 'change', this._updateTotalLength, this)
            .on('selectedIds', 'change', this._onDependsValueFieldsChange, this)
            .on('haveSelected haveAvailable', 'change', this._updateSelectionStatus, this);

        this._subMan
            .on(this._buttonAddNew, 'click', this._onButtonAddNewClick, this)
            .on(this._inputSearchAdd, 'change', this._onInputSearchAddChange, this)
            .on(this._bChooser, 'change', this._chooserChange, this)
            .on(linkClearAll, 'click', this._onButtonClearAllClick, this);

        this._subMan.wrap(BEM.blocks['b-callouts-selector-item'])
            .on(this.elem('available-items-list'), 'delete', this._onDeleteAvailable, this)
            .on(this.elem('selected-items-list'), 'delete', this._onDeleteSelected, this);

        this._subMan.wrap(BEM.blocks['b-callouts-selector-item'])
            .on(this.elem('available-items-list'), 'basket-mouse', this._onAvailableBasketMouse, this)
            .on(this.elem('selected-items-list'), 'basket-mouse', this._onSelectedBasketMouse, this);

        this
            .bindToDoc('keydown', this._onKeyDown)
            .bindTo('selected-items-list-wrap', 'scroll', $.debounce(this._onSelectedItemsScroll, 30, this));
    },

    /**
     * Обработчик события клика по кнопке добавления нового уточнения
     * @private
     */
    _onButtonAddNewClick: function() {
        var value = this._inputSearchAdd.val();

        if (value) {
            this.setMod(this.elem('search'), 'loading', 'yes');

            this._addNewCallout(value)
                .done(function(result) {
                    this.delMod(this.elem('search'), 'loading');

                    // Если добавляли уже существующее значение - то поиск не очищаем
                    result && this._inputSearchAdd.val('');

                    // Скролим к низу обе панели, т.к. добавление происходит в конец
                    this._scrollToBottom();

                    this._inputSearchAdd.setMod('focused', 'yes');
                }.bind(this))
                .fail(function() {
                    this.delMod(this.elem('search'), 'loading');
                }.bind(this));

        }
    },

    /**
     * Обработчик изменения инпута добавления нового
     * @param {jQuery.Event} e
     * @private
     */
    _onInputSearchAddChange: function(e) {
        var calloutText = e.block.val();

        this._buttonAddNew.setMod('disabled',
            !calloutText.length || calloutText.length > u.consts('MAX_CALLOUT_LENGTH') ? 'yes' : '');
    },

    /**
     * Обработчик изменения выбора
     * @param {jQuery.Event} e
     * @param {Object} item
     * @param {Boolean} item.selected Флаг выбранности
     * @param {String} item.name Идентификатор уточнения
     * @private
     */
    _chooserChange: function(e, item) {
        this._model.setCalloutSelection(item.selected, item.name);
    },

    /**
     * Обработчик клика по кнопке "Очистить все"
     * @private
     */
    _onButtonClearAllClick: function() {
        this._model.resetAll();
    },

    /**
     * Обработчик удаления уточнения из выбранных
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {Object} data.id Идентификатор уточнения
     * @private
     */
    _onDeleteSelected: function(e, data) {
        // Убираем тултип, который появляется по скролу
        this._getToolTip().delMod('shown');

        this._model.setCalloutSelection(false, data.id);
    },

    /**
     * Обработчик удаления уточнения из существующих
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {Object} data.id Идентификатор уточнения
     * @private
     */
    _onDeleteAvailable: function(e, data) {
        BEM.blocks['b-confirm'].open({
            message: [
                iget2(
                    'b-callouts-selector',
                    'utochnenie-budet-udaleno-navsegda',
                    'Уточнение будет удалено навсегда и перестанет показываться во всех объявлениях, для которых оно было добавлено.'
                ),
                iget2(
                    'b-callouts-selector',
                    'esli-vy-hotite-ubrat',
                    'Если вы хотите убрать уточнение только из текущего объявления, удалите его из списка справа.'
                ),
                '<br/>',
                iget2('b-callouts-selector', 'udalit', 'Удалить?')
            ],
            onYes: function() {
                this._deleteCallout(data.id);
            }.bind(this)
        });
    },

    /**
     * Удаляет уточнение из VM и из всех моделей баннеров, использующих это уточнение
     * @param {Object} id Идентификатор уточнения
     * @fires channel:callouts#destroy
     * @private
     */
    _destructCallout: function(id) {
        this._model.deleteCalloutById(id);

        this.channel('callouts').trigger('destroy', { id: id });
    },

    /**
     * Возвращает блок b-callouts-selector-item по ID
     * @param {String} listElemName Название элемента списка, в котором произвести поиск
     * @param {Object} id Идентификатор уточнения
     * @private
     */
    _getCalloutItemBlock: function(listElemName, id) {
        return this.findBlockInside(listElemName, {
            block: 'b-callouts-selector-item',
            modName: 'id',
            modVal: id
        });
    },

    /**
     * Обработчик скролла в окошке выбранных
     * @param {jQuery.Event} e
     * @private
     */
    _onSelectedItemsScroll: function(e) {
        this.setMod(this.elem('selected-items'), 'scroll', e.data.domElem.scrollTop() ? 'yes' : '');
    },

    /**
     * Удаляет уточнение по ID
     * @param {String|Number} id идентификатор уточнения
     * @returns {$.Deferred}
     * @private
     */
    _deleteCallout: function(id) {
        this._bChooser.disable(id);
        this._getCalloutItemBlock('available-items-list', id).setMod('deleting', 'yes');

        return this._doRequest('POST', {
            cmd: 'deleteBannersAdditions',
            ulogin: u.consts('ulogin'),
            json_banner_additions: JSON.stringify({
                callouts: [{
                    additions_item_id: id,
                    callout_text: ''
                }]
            })
        })
            .done(function(data) {
                data.success && this._destructCallout(id);
            }.bind(this))
            .fail(function(data) {
                var serverErrors = u._.get(data, 'callouts.array_errors[0]'),
                    message = serverErrors ?
                    serverErrors.map(function(error) { return error.description }).join(' ') :
                    iget2('b-callouts-selector', 'proizoshla-oshibka', 'Произошла ошибка.');

                BEM.blocks['b-confirm'].alert(message);

                this._bChooser.enable(id);
                this._getCalloutItemBlock('available-items-list', id).setMod('deleting', '');
            }.bind(this));
    },

    /**
     * Добавляет новое уточнение к пользователю
     * @param {String} text Текст уточнения
     * @returns {$.Deferred<Boolean>}
     * @private
     */
    _addNewCallout: function(text) {
        var duplicateCallout = this._model.getCalloutByText(text);

        if (duplicateCallout) {
            this._model.setCalloutSelection(true, duplicateCallout.get('additions_item_id'));

            return $.Deferred().resolve(false).promise();
        }

        return this._doRequest('POST', {
            cmd: 'saveBannersAdditions',
            ulogin: u.consts('ulogin'),
            json_banner_additions: JSON.stringify({
                callouts: [{ callout_text: text }]
            })
        })
            .done(function(data) {
                return data.callouts && this._model.addNewCallout(data.callouts[0]) || false;
            }.bind(this))
            .fail(function(data) {
                var serverErrors = u._.get(data, 'callouts.array_errors[0]'),
                    message = serverErrors ?
                    serverErrors.map(function(error) { return error.description }).join(' ') :
                    iget2('b-callouts-selector', 'proizoshla-oshibka', 'Произошла ошибка.');

                this._alertError(message);
            }.bind(this));
    },

    /**
     * Обработчик на создание модели уточнения
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {BEM.MODEL} data.model
     * @private
     */
    _onCreateItem: function(e, data) {
        var calloutModel = data.model;

        this._bChooser.add(this._generateChooserItem(calloutModel));

        // т.к. в методе `add` блока `b-chooser` не зашита логика обновления поска, после добавления элемента,
        // делаем это следующим способом
        this._inputSearchAdd.trigger('change');

        calloutModel.get('selected') && this._appendSelected(calloutModel);
    },

    /**
     * Прокручивает скролл в списке/списках к низу
     * @param {undefined|'available'|'selected'} [listKey] какой список нужно проскроллить, если не указан то применяется к обоим
     * @private
     */
    _scrollToBottom: function(listKey) {
        // Только после полной инициализации
        if (!this._model.get('inited')) return;

        if (!listKey || listKey === 'available') {
            this.elem('available-items-list-wrap').scrollTop(this.elem('available-items-list').height());
        }

        if (!listKey || listKey === 'selected') {
            this.elem('selected-items-list-wrap').scrollTop(this.elem('selected-items-list').height());
        }
    },

    /**
     * Обработчик на изменение модели уточнения
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {BEM.MODEL} data.model
     * @param {Boolean} data.value
     * @param {String} data.innerField
     * @private
     */
    _onChangeItem: function(e, data) {
        var calloutModel, calloutId;

        if (data.innerField === 'selected') {
            calloutModel = data.model;
            calloutId = calloutModel.get('additions_item_id');

            if (data.value) {
                this._bChooser.check(calloutId);
                this._appendSelected(calloutModel);
                this._scrollToBottom('selected');
            } else {
                this._bChooser.uncheck(calloutId);
                this._removeSelected(calloutId);
            }
        }
    },

    /**
     * Обработчик на удаление модели уточнения
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {BEM.MODEL} data.model
     * @private
     */
    _onRemoveItem: function(e, data) {
        var calloutId = data.model.get('additions_item_id');

        this._bChooser.remove(calloutId);

        // т.к. в методе `remove` блока `b-chooser` не зашита логика обновления поска, после удаления элемента,
        // делаем это следующим способом
        this._inputSearchAdd.trigger('change');

        this._removeSelected(calloutId);
    },

    /**
     * Обработчик на изменение длины текстов выбранных уточнений
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {String} data.value
     * @private
     */
    _updateTotalLength: function(e, data) {
        var value = data.value,
            totalLengthElem = this.elem('selected-items-total-length'),
            overflow = u.consts('MAX_CALLOUT_LENGTH_ON_BANNER') - value;

        totalLengthElem.text(overflow);
        this.setMod(totalLengthElem, 'oversized', overflow < 0 ? 'yes' : '');
    },

    /**
     * Обновляет модификатор выбранности уточнений `_selection`
     * @private
     */
    _updateSelectionStatus: function() {
        var haveSelected = this._model.get('haveSelected'),
            haveAvailable = this._model.get('haveAvailable');

        if (haveAvailable) {
            this.setMod('selection', haveSelected ? 'same' : 'empty');
        } else {
            this.setMod('selection', 'can-not-be');
        }
    },

    /**
     * Сообщает о факте изменения входных и выходных данных
     * @returns {Boolean}
     * @private
     */
    _isChanged: function() {
        return this._model.isChanged('selectedIds');
    },

    /**
     * Обработчик на изменение полей, которые влияют на итоговое значение
     * @private
     */
    _onDependsValueFieldsChange: function() {
        var model = this._model,
            validationResult;

        if (model.get('inited')) {
            validationResult = this._validate();

            if (validationResult.valid) {
                this.delMod('error');
            } else {
                this._renderErrors(validationResult.errors);
                this.setMod('error', 'yes');
            }

            this.trigger('change', {
                isChanged: this._isChanged(),
                isValid: !!validationResult.valid
            });
        }
    },

    /**
     * Сбрасывает выставленные ранее уточнения и отображает новые
     * @param {Object} blockData
     * @param {Object[]} blockData.callouts Выбранные уточнения к баннеру
     * для массовых действий
     */
    reset: function(blockData) {
        this.setMod('loading', 'yes');

        this._model.set('inited', false);

        // Очищаем инпут
        this._inputSearchAdd.val('');

        this._doRequest('GET', {
            cmd: 'getBannersAdditions',
            ulogin: u.consts('ulogin'),
            additions_type: 'callout',
            limit: 10000,
            offset: 0
        })
            .done(function(serverData) {
                this._initPopup(blockData, serverData);
                this.delMod('loading');
            }.bind(this))
            .fail(function() {
                this._alertError(iget2('b-callouts-selector', 'proizoshla-oshibka', 'Произошла ошибка.'), true);
                this.delMod('loading');
            }.bind(this));
    },

    /**
     * Выполняет запрос
     * @param {String} [method='POST'] метод запроса
     * @param {Object} data Параметры запроса
     * @returns {$.Deferred}
     * @private
     */
    _doRequest: function(method, data) {
        var dfd = $.Deferred();

        this._request || (this._request = BEM.create('i-request_type_ajax', {
            cache: false,
            url: '/registered/main.pl',
            dataType: 'json',
            callbackCtx: this
        }));

        this._request.get(
            data,
            function(result) { result.success ? dfd.resolve(result) : dfd.reject(result) },
            function(err) { dfd.reject(err) },
            { type: (method || 'POST').toLowerCase() });

        return dfd.promise();
    },

    /**
     * Заполняет модель данными и отключает прелоадер
     * @param {Object} blockData - входные данные блока
     * @param {Object[]} blockData.callouts Выбранные уточнения к баннеру
     * @param {Object[][]} [blockData.calloutsKits] Выбранные уточнения баннеров
     * @param {Object} serverData - серверные данные
     * @param {Object[]} serverData.callouts - серверные данные
     * @private
     */
    _initPopup: function(blockData, serverData) {
        var selectedIds = (blockData.callouts || blockData.calloutsKits && blockData.calloutsKits[0] || [])
            .map(function(item) { return item.additions_item_id; });

        this._model
            .set('list', this._getAvaliableItems(serverData.callouts, selectedIds))
            .set('inited', true)
            .fix();

        this._updateSelectionStatus();
        this._inputSearchAdd.setMod('focused', 'yes');
    },

    /**
     * Возвращает список уточнений для заполнения поля list view-модели
     * @param {Object[]} availableCallouts Уточнения пользователя
     * @param {Array} selectedIds Идентификаторы выбранных уточнений
     * @returns {Object[]}
     * @private
     */
    _getAvaliableItems: function(availableCallouts, selectedIds) {
        return availableCallouts.map(function(calloutItem) {
            return u._.extend(calloutItem, {
                selected: selectedIds.indexOf(calloutItem.additions_item_id) >= 0
            });
        });
    },

    /**
     * Добавляет в DOM выбранное уточнение
     * @param {BEM.MODEL} calloutModel
     * @private
     */
    _appendSelected: function(calloutModel) {
        BEM.DOM.append(this.elem('selected-items-list'), BEMHTML.apply({
            block: 'b-callouts-selector-item',
            mods: { type: 'selected' },
            id: calloutModel.get('additions_item_id'),
            text: calloutModel.get('callout_text'),
            state: calloutModel.get('status_moderate')
        }));
    },

    /**
     * Удаляет из DOM выбранное уточнение
     * @param {String} id Идентификатор уточнения
     * @private
     */
    _removeSelected: function(id) {
        var calloutItem = this._getCalloutItemBlock('selected-items-list', id);

        calloutItem && calloutItem.destruct();
    },

    /**
     * Формирует bemjson для элемента блока `b-chooser`
     * @param {BEM.MODEL} calloutModel
     * @returns {Object}
     * @private
     */
    _generateChooserItem: function(calloutModel) {
        var calloutId = calloutModel.get('additions_item_id'),
            calloutText = calloutModel.get('callout_text');

        return {
            elemMods: calloutModel.get('selected') ? { selected: 'yes' } : {},
            js: {
                search: { indexed: calloutText }
            },
            name: calloutId,
            content: {
                block: 'b-callouts-selector-item',
                mods: { type: 'available' },
                id: calloutId,
                text: calloutText,
                mixToText: [{ block: 'b-chooser', elem: 'indexed' }],
                state: calloutModel.get('status_moderate')
            }
        };
    },

    /**
     * Обновляет сообщение о выбранных уточнениях
     * @param {String} text
     * @private
     */
    _setSelectionMessage: function(text) {
        this.elem('selection-status').text(text);
    },

    /**
     * Обработчик на нажатие клавиш
     * @param {jQuery.Event} e
     * @private
     */
    _onKeyDown: function(e) {
        // при нажатии Enter
        if (e.keyCode === 13 && this._inputSearchAdd.hasMod('focused', 'yes')) {
            this._buttonAddNew.domElem.trigger('click');
        }
    },

    /**
     * Возвращает выбранные пользователем значения
     * @returns {{calloutsKits: object[], callouts: object}}
     */
    val: function() {
        var result = this._model.getSelected().map(function(callout) {
            return callout.toJSON();
        });

        return {
            callouts: result,
            calloutsKits: [result]
        };
    },

    _outerPopup: undefined,

    /**
     * Ищет попап снаружи
     * @returns {null|BEM.DOM}
     * @private
     */
    _getOuterPopup: function() {
        return this._outerPopup === undefined ?
            (this._outerPopup = this.findBlockOutside('popup')) :
            this._outerPopup;
    },

    /**
     * Отображает сообщение в диалоговом окне
     * @param {String} message Сообщение
     * @param {Boolean} [needClose] Флаг, при котором по клику на "Ок" и "Отмена" в диалоговом окне закроется попап (при наличи) снаружи блока
     * @private
     */
    _alertError: function(message, needClose) {
        var popup = this._getOuterPopup(),
            callBack = popup && needClose ?
                function() { popup.hide() } :
                undefined;

        BEM.blocks['b-confirm'].open({
            type: 'alert',
            message: message,
            fromPopup: popup,
            onYes: callBack,
            onNo: callBack
        }, this);
    },

    _tooltip: null,

    /**
     * Возвращает инстанс блока подсказки
     * @returns {BEM.DOM}
     * @private
     */
    _getToolTip: function() {
        return this._tooltip || (this._tooltip = $(BEMHTML.apply({
            block: 'tooltip',
            js: { to: ['right', 'bottom'], zIndexGroupLevel: 6 },
            mods: { size: 's', theme: 'normal' }
        })).bem('tooltip'));
    },

    /**
     * Отображает подсказку
     * @param {jQuery} owner DOM-нода куда должна указывать подсказка
     * @param {String} text Текст подсказки
     * @private
     */
    _showToolTip: function(owner, text) {
        this._getToolTip()
            .setOwner(owner)
            .setContent(BEMHTML.apply({
                block: this.__self.getName(),
                elem: 'tooltip-message',
                content: text
            }))
            .setMod('shown', 'yes');
    },

    /**
     * Прячет подсказку
     * @private
     */
    _hideTooltip: function() {
        this._getToolTip().delMod('shown');
    },

    /**
     * Обработчик клика по корзине в поле "Выбрано" в списке выбранных уточнений
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {Object} data.domElem DOM-нода корзины
     * @param {'over'|'out'} data.mode Режим
     * @private
     */
    _onSelectedBasketMouse: function(e, data) {
        if (data.mode === 'over') {
            this._showToolTip(data.domElem, iget2('b-callouts-selector', 'ubrat-utochnenie-iz-spiska', 'Убрать уточнение из списка объявления'));
        } else {
            this._hideTooltip();
        }
    },

    /**
     * Обработчик клика по корзине в поле "Выбрано" в списке доступных
     * @param {jQuery.Event} e
     * @param {Object} data
     * @param {Object} data.domElem DOM-нода корзины
     * @param {'over'|'out'} data.mode Режим
     * @private
     */
    _onAvailableBasketMouse: function(e, data) {
        if (data.mode === 'over') {
            this._showToolTip(data.domElem, iget2('b-callouts-selector', 'udalit-utochnenie-iz-spiska', 'Удалить уточнение из списка клиента'));
        } else {
            this._hideTooltip();
        }
    },

    _newItemSpin: null,

    /**
     * Возвращает инстанс блока `spin`, которые внутри поискп
     * @returns {BEM.DOM}
     * @private
     */
    _getNewItemSpin: function() {
        return this._newItemSpin || (this._newItemSpin = this.findBlockOn('new-item-loader', 'spin'));
    },

    destruct: function() {
        this._subMan.dispose();
        this._subMan.destruct();
        this._model.destruct();

        BEM.DOM.destruct(this.domElem, true); // чистим внутренности

        this.__base.apply(this, arguments);
    },

    /**
     * Валидирует модель уточнений
     * @returns {{errors: Array, valid: Boolean}}
     */
    _validate: function() {
        return this._model.validate()
    },

    /**
     * Отрисовывает ошибки валидации
     * @param {Array} errors массив ошибок
     * @private
     */
    _renderErrors: function(errors) {
        var content = errors
            .map(function(error) {
                return {
                    block: 'icon-text',
                    mods: { theme: 'alert', size: 'ms' },
                    text: error.text
                }
            });

        BEM.DOM.update(this.elem('errors'), BEMHTML.apply(content));
    }

});
