/**
 * Модуль отрисовки графиков статистики
 * Данные statData - это массив dates в объекте статистики
 */
(function() {
    var DEFAULT_CHART_WIDTH = 900,
        MAX_COLUMN_CHART_POINTS = 65;

    BEM.DOM.decl(
        'b-chart-pane', {
            onSetMod: {
                js: function() {
                    this._chart = this.elem('b-inner-chart');

                    var statData = this.params.statData,
                        points = this.params.points,
                        group = this.params.group;

                    if (statData === undefined) {
                        throw new Error(iget2('b-chart-pane', 'ne-peredany-dannye-dlya', 'Не переданы данные для графиков'));
                    }
                    if (statData.length == 0) {
                        BEM.blocks['b-confirm'].alert(iget2('b-chart-pane', 'net-statistiki-dlya-otobrazheniya', 'Нет статистики для отображения'));
                    }
                    this._drawChart(this._chart, statData, points, group, iget2('b-chart-pane', 'grafiki-statistiki', 'Графики статистики'));
                }
            },

            /**
             * Уничтожает блок
             * @returns {BEM.DOM}
             */
            destruct: function() {
                return this.__base.apply(this, arguments);
            },

            /**
             * Отрисовывает график с заданными данными
             * @param {jQuery} chart jQuery-объект, элемент, в котором рисуется график
             * @param {Object} series - данные в Highcharts-совместимом виде
             * @param {Number} points - количество точек на графике. Для маленьких графиков рисуем столбчатую диагармму
             * с широкими столбцами, для средних - с узкими, для больших рисуем areaspline
             * @param {String} group - тип группировки
             * @param {String} title - заголовок таблицы
             * @private
             */
            _drawChart: function(chart, series, points, group, title) {
                var DEFAULT_COLUMN_WIDTH = (points < 15 ? 50 : 12),
                    graphWidth = points * 2 * DEFAULT_COLUMN_WIDTH,
                    today = u.moment().format('DD.MM.YY'),
                    // общие настройки графикв
                    commonSettings = {
                        exporting: {
                            chartOptions: {
                                plotOptions: {
                                    series: {
                                        dataLabels: {
                                            enabled: false
                                        }
                                    }
                                }
                            },
                            filename: iget2('b-chart-pane', 'statistika-ot', 'статистика_от') + '_' + today,
                            scale: 3,
                            fallbackToExportServer: false
                        },
                        title: {
                            text: title
                        },
                        credits: {
                            enabled: false
                        },
                        // цвет легенды для не показываемых столбцов
                        itemHiddenStyle: {
                            color: '#ccc'
                        },
                        // тултип - убираем время
                        tooltip: {
                            formatter: function() {
                                var value = '<span>' + this.series.name + '</span>' + ': ' + this.y,
                                    date;

                                group = (group == undefined ? 'day' : group);
                                switch (group) {
                                    case 'day': date = u.moment(this.x).format('dddd, DD MMM YYYY'); break;
                                    case 'week':
                                        date = iget2('b-chart-pane', 'nedelya-s', 'Неделя с') + ' ' + u.moment(this.x).format('ddd, DD MMM YYYY');
                                        break;
                                    case 'month': date = u.moment(this.x).format('MMMM YYYY'); break;
                                    case 'quarter':
                                        var d = u.moment(this.x),
                                            month = d.month();

                                        date = Math.ceil((month + 1) / 3) + ' ' + iget2('b-chart-pane', 'kvartal', 'квартал') + ' ' +
                                            d.format('YYYY');
                                        break;
                                    case 'year': date = u.moment(this.x).format('YYYY') + ' ' + iget2('b-chart-pane', 'god', 'год'); break;
                                }
                                return '<span><b>' + date + '</b></span><br/>' + value;
                            }
                        },
                        // настройки шкалы с датами
                        xAxis: {
                            type: 'datetime',
                            labels: {
                                formatter: function() {
                                    group = (group == undefined ? 'day' : group);

                                    switch (group) {
                                        case 'day': return u.moment(this.value).format('DD.MM');
                                        case 'week': return u.moment(this.value).format('DD.MM');
                                        case 'month': return u.moment(this.value).format('MM.YY');
                                        case 'quarter':
                                            var d = u.moment(this.value),
                                                month = d.month();
                                            return Math.ceil((month + 1) / 3) + ' ' + iget2('b-chart-pane', 'kv', 'кв.') + ' ' +
                                                d.format('YY');
                                        case 'year': return u.moment(this.value).format('YYYY');
                                    }
                                }
                            }
                        },
                        // настройки вертикальных шкал
                        yAxis:
                        {
                            min: 0,
                            title: {
                                text: iget2('b-chart-pane', 'kolichestvo', 'Количество')
                            }
                        }
                    };
                // языковые настройки
                u.Highcharts.setOptions(
                    {
                        global: {
                            timezoneOffset: -3 * 60
                        },
                        lang: {
                            downloadJPEG: iget2('b-chart-pane', 'vygruzit-jpeg', 'Выгрузить JPEG'),
                            downloadPDF: iget2('b-chart-pane', 'vygruzit-pdf', 'Выгрузить PDF'),
                            downloadSVG: iget2('b-chart-pane', 'vygruzit-svg', 'Выгрузить SVG'),
                            downloadPNG: iget2('b-chart-pane', 'vygruzit-png', 'Выгрузить PNG'),
                            printChart: iget2('b-chart-pane', 'pechat', 'Печать'),
                            months: u.moment.months(),
                            shortMonths: u.moment.monthsShort(),
                            weekdays: u.moment.weekdays()
                        }
                    });
                // если точек на графике сравнительно мало
                // рисуем стобчатую диаграмму
                if (points < MAX_COLUMN_CHART_POINTS) {
                    // у графиков на 1-2точки (маленьких) если ставить кастомную ширину - разъезжается highchart-овская верстка
                    var chartSetting = graphWidth >= 600 ?
                        { chart: { type: 'column', width: graphWidth } } :
                        { chart: { type: 'column' } };
                    chart.highcharts(
                        u._.assign(
                            commonSettings, chartSetting, {
                                plotOptions: {
                                    series: {
                                        pointWidth: DEFAULT_COLUMN_WIDTH,
                                        allowPointSelect: true
                                    }
                                },
                                series: series
                            }));
                    this.domElem.width(chart.highcharts().chartWidth);
                    // а иначе - arespline
                } else {
                    this.domElem.width(DEFAULT_CHART_WIDTH);
                    chart.highcharts(
                        u._.assign(
                            commonSettings, {
                                chart: {
                                    type: 'areaspline',
                                    trackByArea: true,
                                    width: DEFAULT_CHART_WIDTH
                                },
                                series: series
                            }));
                }
            }
        });
})();
