/**
 * @typedef {Object} Point - объект, содержащий данные одной точки
 *
 * @property {Number} x - значение оси X ()
 * @property {Number} y - значение оси Y
 */

/**
 * @typedef {Object} Series - объект, содержащий данные одного ряда для графика
 *
 * @property {String} name - название ряда
 * @property {String} column - идентификатор столбца
 * @property {Array<Point>} data - содержит точки
 * @property {String} [groupBy] - идентификатор группировки
 * @property {Number} [yAxis] - ось Y к которой относятся данные
 */

/**
 * @typedef {Object} ChartOptions - входные настройки блока
 *
 * @property {String} [groupByDate] - группировка значений
 */

/**
 * Определяет параметры и строит график
 *
 * @param {Object} params - входные параметры
 * @param {ChartOptions} params.options - входные настройки блока
 * @param {Array<Series>} params.series - группировка значений
 */

BEM.DOM.decl('b-chart', {

    onSetMod: {

        js: function() {
            this.buildChart(this.params);
        },

        view: function() {
            this.afterCurrentEvent(function() {
                this._chart.update(this._getChartViewParams())
            });
        }

    },

    _tooltopPinner: function(tooltop) {
        tooltop.state = !tooltop.state;
        BEM.blocks['b-metrika2'].reachGoal('FREEZE_RELEASE_TOOLTIP');
    },

    /**
     * Возвращает базовые значения
     * @return {Object}
     * @private
     */
    _getDefaultOptions: function() {
        var context = this,
            isTooltipPinned = {
                state: false
            }; // @tgnc: состояние тултипа DIRECT-91894

        this.cache = []; // @tgnc: для кеширования данных в тултипе
        return {
            chart: {
                height: 330,
                width: 1040,
                style: { fontFamily: '\'YS Text\', Arial' },
                events: {
                    click: function(e) {
                        context._tooltopPinner(isTooltipPinned)
                    }
                }
            },
            credits: {
                enabled: false  // убирает надпись highcharts.com
            },
            title: {
                text: null  // убирает заголовок
            },
            exporting: {
                chartOptions: {
                    chart: {
                        spacing: [40, 40, 50, 40]
                    },
                    title: {
                        text: this.params.options.title || null,
                        y: 0,
                        margin: 10,
                        align: 'left'
                    },
                    subtitle: {
                        text: this.params.options.subtitle || null
                    },
                    legend: {
                        x: 0,
                        y: 15
                    }
                }
            },
            legend: {
                align: 'right',
                verticalAlign: 'top',
                layout: 'vertical',
                x: 0,
                y: -16,
                margin: 40,
                itemMarginBottom: 5,
                itemMarginTop: 5,
                symbolHeight: 10,
                symbolPadding: 10,
                symbolWidth: 10,
                itemStyle: {
                    color: '#000000',
                    cursor: 'default',
                    fontSize: '13px',
                    fontWeight: 'normal'
                },
                labelFormatter: function() {
                    return u.hellipCut(this.name, 35, '...');
                }
            },
            loading: {
                labelStyle: {
                    fontSize: '15px',
                    color: 'rgba(0, 0, 0, 0.5)',
                    fontWeight: 'normal',
                    position: 'relative',
                    top: '45%'
                },
                style: {
                    position: 'absolute',
                    backgroundColor: '#ffffff',
                    opacity: 0.9,
                    textAlign: 'center'
                }
            },
            tooltip: {
                useHTML: true,
                style: {
                    pointerEvents: 'auto'
                },
                followPointer: false,
                backgroundColor: 'rgba(255,255,255,0)',
                borderWidth: 0,
                borderRadius: 0,
                hideDelay: 1500, // чтобы успевать кликнуть по ссылке
                padding: 0,
                positioner: function() {
                    return context._tooltipPositioner.call(this, isTooltipPinned.state, arguments)
                },
                formatter: function() {
                    return context._tooltipFormatter.call(
                        this, context.params.options.filters,
                        isTooltipPinned.state,
                        context.cache
                    )
                },
                shadow: false,
                shared: true, // общий попап для всех series
            },
            plotOptions: {
                series: {
                    events: {
                        legendItemClick: function() {
                            return false; // выключает возможность вкл/выкл series через легенду
                        },
                        click: function(event) {
                            context._tooltopPinner(isTooltipPinned)
                        }
                    },
                    turboThreshold: 0
                }
            },
            xAxis: {
                type: 'datetime',
                labels: {
                    style: {
                        color: '#000000',
                        opacity: 0.5,
                        cursor: 'default',
                        fontSize: '13px'
                    }
                },
                title: {
                    text: null
                },
                tickLength: 0
            },
            yAxis: [
                { // левая ось
                    labels: {
                        style: {
                            color: '#000000',
                            opacity: 0.5,
                            cursor: 'default',
                            fontSize: '13px'
                        }
                    },
                    title: {
                        text: null
                    }
                },
                { // правая ось
                    labels: {
                        style: {
                            color: '#000000',
                            opacity: 0.5,
                            cursor: 'default',
                            fontSize: '13px'
                        }
                    },
                    title: {
                        text: null
                    },
                    opposite: true
                }
            ],

            navigation: {
                buttonOptions: {
                    enabled: false
                }
            }
        }
    },

    /**
     * @typedef {Object} TooltipParams
     * @property {Number} labelWidth - ширина тултипа
     * @property {Number} labelHeight - высота тултипа
     * @property {Object} point - информация о точке
     */

    /**
     * Позиционирует тултип
     * @param {Boolean} isTooltipPinned - состояние тултипа: зафиксирован или движется за курсором
     * @param {TooltipParams} tooltipParams
     * @returns {{x: Number, y: Number}}
     * @private
     */
    _tooltipPositioner: function(isTooltipPinned, tooltipParams) {
        var labelWidth = tooltipParams[0],
            labelHeight = tooltipParams[1],
            point = tooltipParams[2],
            tooltipX,
            tooltipY,
            chart = this.chart,
            margin = 10; // общий отступ, что бы не притираться тултипу к точкам и границам графика

        if (isTooltipPinned) {
            return this._cachedPos;
        }

        // выбираем сторону открытия по оси X
        tooltipX = point.plotX + labelWidth > chart.plotWidth ?
            point.plotX + chart.plotLeft - labelWidth - margin :    // слева
            point.plotX + chart.plotLeft + margin;                  // справа

        // отступ от верхней границы,
        // -20 для того что бы тултип не находится на одном уровне с точкой
        tooltipY = point.plotY + chart.plotTop - 20;

        // если тултип выходит за нижние границы
        if (tooltipY + labelHeight > chart.chartHeight) {
            tooltipY = chart.chartHeight - labelHeight - margin;

            // если тултип выходит за верхние границы
        } else if (tooltipY < margin) {
            tooltipY = margin;
        }

        this._cachedPos = {
            x: tooltipX,
            y: tooltipY
        };

        return {
            x: tooltipX,
            y: tooltipY
        }
    },

    /**
     * Форматирует внешний вид тултипа
     * @returns {String}
     * @private
     */
    _tooltipFormatter: function(filters, isPinned, cache) {
        var columns = [],
            groupBy = [],
            points = (this.points || [this.point]).map(function(point) {
                var options = point.series.options || {};

                columns.push(options.columnText);
                groupBy.push(options.groupByText);

                return {
                    name: u.hellipCut(point.series.name, 35, '...'),
                    x: point.x,
                    y: point.y,
                    color: point.color
                }
            });

        if (!isPinned) {
            cache.length = 0;
            points.forEach(function(point) {
                cache.push(point);
            });
        }

        return BEMHTML.apply({
            block: 'b-chart',
            elem: 'tooltip',
            points: isPinned ? cache : points,
            filters: filters,
            title: {
                columns: u._.compact(columns),
                groupBy: u._.compact(groupBy)
            }
        });
    },

    /**
     * Экспорт графика
     * @param {Object} params
     * @param {Boolean} params.print - напечатать (type игнорируется)
     * @param {String} params.type - MIME тип
     * @param {String} params.filename - имя файла
     */
    exportChart: function(params) {
        if (params.print) {
            this._chart.print();
        } else {
            this._chart.exportChartLocal({
                type: params.type,
                filename: params.filename || u.moment().format('DD-MM-YYYY')
            });
        }
    },

    /**
     * Построение графика
     * @param {Object} params - входные параметры
     * @param {ChartOptions} params.options - входные настройки блока
     * @param {Array<Series>} params.series - группировка значений
     * @return {buildChart}
     */
    buildChart: function(params) {
        this._chart = u.Highcharts.chart(this.domElem[0], this._getChartParams(params));
        params.series.length === 0 && this.showMessage(iget2('b-chart', 'net-dannyh', 'Нет данных'));
        return this;
    },

    showErrorMessage: function(text) {
        var errorMessage = BEMHTML.apply([
            {
                block: 'b-chart',
                elem: 'message-title',
                content: iget2('b-chart', 'nevozmozhno-postroit-grafik', 'Невозможно построить график')
            },
            text
        ]);
        this.showMessage(errorMessage);
    },

    showMessage: function(content) {
        var message = this.findElem('message'),
            newMessage = BEMHTML.apply({
                block: 'b-chart',
                elem: 'message',
                content: content
            });

        if (message.length) {
            BEM.DOM.replace(message, newMessage);
        } else {
            BEM.DOM.append(this.domElem.find('.highcharts-container'), newMessage);
        }
    },

    hideMessage: function() {
        this.findElem('message').remove();
    },

    /**
     * Показывает индикатор загрузки
     */
    showLoading: function() {
        this.hideMessage();
        this._spin || (this._spin = BEMHTML.apply({
            block: 'spin2',
            mix: {
                block: 'b-chart',
                elem: 'spin'
            },
            mods: { progress: 'yes', size: 's' }
        }));
        this._chart.showLoading(this._spin + iget2('b-chart', 'dannye-zagruzhayutsya', 'Данные загружаются...'));

        return this;
    },

    /**
     * Скрывает индикатор загрузки
     */
    hideLoading: function() {
        this._chart.hideLoading();

        return this;
    },

    /**
     * Возвращает параметры для конкретного типа графика (переопределяется в модификаторе)
     * @return {Object}
     * @private
     */
    _getChartViewParams: function() {

        return {
            chart: {
                type: 'column'
            }
        };
    },

    /**
     * Сбор параметров
     * @param {Object} params - входные параметры
     * @param {ChartOptions} params.options - входные настройки блока
     * @param {Array<Series>} params.series - группировка значений
     * @return {Object}
     */
    _getChartParams: function(params) {
        var result = u._.merge({},
                this._getDefaultOptions(),
                this._getChartViewParams(),
                this._calcDependentDataParams(params)
            );

        // u._.merge на больших данных зло, значения для графика вставляем отдельно
        result.series = params.series;

        return result;
    },

    /**
     * Формирует параметры, которые зависят от входных данных
     * @param {Object} params - входные параметры
     * @param {ChartOptions} params.options - входные настройки блока
     * @param {Array<Series>} params.series - группировка значений
     * @return {Object}
     */
    _calcDependentDataParams: function(params) {

        return {
            plotOptions: {
                series: {
                    groupPadding: this._calcGroupPadding(params),
                    pointPadding: 0
                }
            },
            xAxis: {
                labels: {
                    formatter: function() {
                        var group = u._.get(params, 'options.groupByDate') || 'day',
                            date = u.moment(this.value);

                        switch (group) {
                            case 'day': return date.format('DD.MM');
                            case 'week': return date.format('DD.MM');
                            case 'month': return date.format('MM.YY');
                            case 'quarter':
                                var d = date,
                                    month = d.month();
                                return Math.ceil((month + 1) / 3) + ' ' + iget2('b-chart', 'kv', 'кв.') + ' ' + d.format('YY');
                            case 'year': return date.format('YYYY');
                        }
                    }
                },
                tickPositioner: this._calcTickPositioner(params)
            },
            yAxis: [
                {
                    labels: {
                        format: this._calcYAxisLabelFormat(0, params)
                    }
                },
                {
                    labels: {
                        format: this._calcYAxisLabelFormat(1, params)
                    }
                }
            ]
        };
    },

    /**
     * Определяет значения параметра groupPadding
     * @param {Object} params - входные параметры
     * @param {ChartOptions} params.options - входные настройки блока
     * @param {Array<Series>} params.series - группировка значений
     * @return {number}
     * @private
     */
    _calcGroupPadding: function(params) {
        var pointsCount = u._.get(params, 'series[0].data') && params.series[0].data.length || 0;

        return pointsCount > 50 ? 0 : 0.15;
    },

    /**
     * Определяет значения параметра tickPositioner
     * @param {Object} params - входные параметры
     * @param {ChartOptions} params.options - входные настройки блока
     * @param {Array<Series>} params.series - группировка значений
     * @return {Function|undefined}
     * @private
     */
    _calcTickPositioner: function(params) {
        var group = u._.get(params, 'options.groupByDate'),
            points = u._.get(params, 'series[0].data') || [],
            pointsX = points.map(u._.property('x')),
            result;

        switch (group) {
            case 'quarter':
                result = function() { return pointsX; };
                break;
            default:
                result = undefined;
        }

        return result;
    },

    _calcYAxisLabelFormat: function(number, params) {
        var yAxisIndexes = params.series.map(u._.property('yAxis')),
            yAxisIndex = u._.findIndex(yAxisIndexes, function(yAxis) {
                return yAxis == number;
            });

        yAxisIndex = ~yAxisIndex ? yAxisIndex : 0; // если индекс не найден, значит только одна ось Y

        return '{value} ' + u._.get(params, 'series[' + yAxisIndex + '].unitText','');
    },

    /**
     * https://wiki.yandex-team.ru/users/chizh/Palitra/
     * @returns {Array}
     * @private
     */
    _getColors: function() {
        return ['#97cc64', '#ffd963', '#fd5a3e', '#77b6e7', '#a955b8', '#dc9d6b', '#ea527f', '#4569bc', '#4ba062'];
    }

}, {});
