/**
 * @typedef {Object} Point
 * @property {Number} x - количество миллисекунд
 * @property {Number} y - значение
 * @property {Array} [detailedData]
 */

/**
 * @typedef {Object} Row
 * @property {String} name - заголовок для легенды
 * @property {String} [column] - тип столбца
 * @property {String} [groupBy] - тип стреза
 * @property {Array<Point>} data - список точек
 */

(function() {
    u.register({
        'b-chart': {

            MAX_SERIES_COUNT: 6,

            /**
             * Заменяет ключи столбцов/срезов на текст для клиента
             * @param {Array<Row>} series
             * @param {Object} params
             * @returns {Array<Row>}
             */
            translate: function(series, params) {
                var units = {
                    percent: '%',
                    quantitative: '',
                    currency: params.currency
                };

                return series.map(function(row) {
                    if (row.groupBy) {
                        row.name = u['text-store'].get(params, 'chart', row.groupBy, row.name);

                        // для заголовка тултипа
                        row.columnText = u['text-store'].get(params, 'chart', row.column);
                        row.groupByText = u['text-store'].get(params, 'chart', row.groupBy);
                    } else if (row.column) {
                        row.name = u['text-store'].get(params, 'chart', row.column);

                        // для заголовка тултипа
                        row.columnText = row.name;
                    }

                    row.unitText = units[row.unit] || '';

                    return row;
                });
            },

            /**
             * Формирует топ, оставшиеся обьединяются в «Остальные»
             * @param {Array<Row>} series
             * @param {Number} [limit]
             * @param {String} [otherText]
             * @returns {Array<Row>}
             */
            simplify: function(series, limit, otherText) {
                var newSeries = [],
                    otherSeries = [],
                    otherData = [],
                    points,
                    index,
                    sumY = function(series) {
                        return series.reduce(function(sum, row) {
                            return sum + row.data[index].y;
                        }, 0);
                    },
                    getDataByIndex = function(series, index) {
                        return series.map(function(row) {
                            return {
                                name: row.name,
                                data: row.data[index]
                            };
                        });
                    };

                if (typeof limit === 'string') {
                    otherText = limit;
                    limit = undefined;
                }

                if (limit === undefined) {
                    limit = this.MAX_SERIES_COUNT;
                }

                if (series.length > limit) {
                    newSeries = series.slice(0, limit - 1);
                    otherSeries = series.slice(limit - 1);

                    // т.к. количество точек во всех строках одинаковое
                    points = series[0].data.length;

                    for (index = 0; index < points; index++) {
                        otherData[index] = {
                            // X - дата, у всех столбцов должна совпадать
                            x: otherSeries[0].data[index].x,

                            // суммируем все значения Y
                            y: u.numberFormatter.round(sumY(otherSeries)),

                            // сохраняем значения для тултипа
                            detailedData: getDataByIndex(otherSeries, index)
                        };
                    }

                    newSeries.push({
                        name: otherText || u['text-store'].get('chart', 'others'),
                        data: otherData
                    });

                    return newSeries;
                } else {
                    return series;
                }
            },

            /**
             * Добавляет каждому объекту из series поле yAxis
             * @param {Array<Row>} series - данные для графика
             * @param {Object} groups - хэш, где ключ — едина измерения, значение ключа — список столбцов
             * @returns {Array<Row>}
             */
            setYAxis: function(series, groups) {
                var usedColumns,
                    usedGroup = [],
                    columnToGroup = {};

                // находим все используемые столбцы
                usedColumns = u._.compact(u._.uniq(series.map(function(row) {
                    return row.column;
                })));

                usedColumns.forEach(function(column) {
                    var group;

                    u._.keys(groups).some(function(groupName) {
                        if (groups[groupName].indexOf(column) !== -1) {
                            group = groupName;

                            return true;
                        }

                        return false;
                    });

                    columnToGroup[column] = group;
                    usedGroup.push(group);
                });

                usedGroup = u._.unique(usedGroup);

                if (usedGroup.length > 1) {
                    return series.map(function(row) {
                        row.yAxis = usedGroup.indexOf(columnToGroup[row.column]);

                        return row;
                    });
                }

                return series;
            }

        }
    });
}());
