(function() {
    var CHART_LINE_POINTS_LIMIT = 90;

    BEM.DOM.decl('b-charts-manager', {

        onSetMod: {
            js: function() {
                this._chartStatData = BEM.blocks['i-chart-stat-data'];
                this._filters = this.findBlockInside('filters', 'b-chart-filters');
                this._switcher = this.findBlockInside('switcher', 'button');
                this._export = this.findBlockInside('export', 'b-dropdown-trigger');

                this._filters &&
                    this._filters.on('change', this._onFilterChange, this);

                this._switcher &&
                    this._switcher.on('click', this._onSwitcherClick, this);

                this._export &&
                    this._export.on('export', function(e, data) {
                        this._exportChart({ print: data.name === 'print', type: data.MIME });
                    }, this);

                this._requestChartDataDebounced = $.debounce(this._requestChartData, 500, this);
            },
            show: {
                yes: function() {
                    this.buildChart();
                    this._requestChartData(this._getFiltersData());
                }
            }
        },

        _filters: null,

        /**
         * Вызывается прии изменении значения одного из фильтров
         * @param {Event} e
         * @param {Object} data
         * @param {String} data.filter - название фильтра
         * @param {Object|Array} data.data - значение фильтра
         * @private
         */
        _onFilterChange: function(e, data) {
            var view;

            if (data.filter === 'view') {
                view = data.data.currentVal;

                if (view === 'line' && this._chart.getMod('stacking') === 'yes') {
                    view = 'area';
                }

                this._chart.setMod('view', view);
            } else {
                this._requestChartDataDebounced(this._getFiltersData());
            }
        },

        /**
         * Обрабатывает клик по переключателю
         * @private
         */
        _onSwitcherClick: function() {
            this.toggleMod('show', 'yes');
        },

        /**
         * Дергает ручку для получения данных с сервера
         * @param {Object} filters - значения фильтров (название_фильтра: значение)
         * @private
         */
        _requestChartData: function(filters) {
            // Если запрос за данными уже выполняется, тогда складываем фильтры в очередь, чтобы
            // запросить их, когда будет возможность. В любой момент времени должно быть не больше одного
            // запроса за данными, т.к. на сервере есть ограничение на количество одновременных запросов
            if (this.chartStatDeferred && this.chartStatDeferred.state() === 'pending') {
                this.queuedChartFilters = filters;

                return;
            }

            // Очищаем очередь из запросов, чтобы был выход из рекурсии
            delete this.queuedChartFilters;

            var reportData = this.params.reportData;

            u.consts('rights').enableStatsMultiClientsMode && (reportData.multi_clients_mode = 1);
            !u.consts('rights').enableStatsMultiClientsMode && (reportData.ulogin = u.consts('ulogin'));

            filters.columns && (reportData.columns = filters.columns);
            filters.groupBy && (reportData.groupBy = filters.groupBy);
            filters.view && (reportData.view = filters.view);

            if (filters.columns.length === 0) {
                this.buildChart(reportData);
                this._chart.showMessage(iget2('b-charts-manager', 'vyberite-odin-ili-neskolko', 'Выберите один или несколько столбцов для отображения'));
                return;
            }

            this._beforeRequestData();

            this.chartStatDeferred = this._chartStatData.get(reportData, this);
            this.chartStatDeferred
                .then(function(data) {
                    // Проверяем, есть ли фильтры в очереди, чтобы обновить данные графика.
                    // Если есть - делаем запрос с этими фильтрами
                    if (this.queuedChartFilters) {
                        return this._requestChartData(this.queuedChartFilters);
                    }

                    this._afterRequestData();
                    this._onGotChartData(reportData, data);
                }, function(event, eventName) {
                    // Проверяем, есть ли фильтры в очереди, чтобы обновить данные графика.
                    // Если есть - делаем запрос с этими фильтрами
                    if (this.queuedChartFilters) {
                        return this._requestChartData(this.queuedChartFilters);
                    }

                    this._afterRequestData();

                    if (eventName === 'timeout' || event === 'too_much') {
                        this._showTimeoutError();
                    } else {
                        this._showRequestError();
                    }
                });
        },

        /**
         * Вызывается перед запросом
         * @private
         */
        _beforeRequestData: function() {
            this._filters.disableFilter('view');
            this._export && this._export.disable();
            this._chart.showLoading();
        },

        /**
         * Вызывается после завершения запроса (удачно или нет)
         * @private
         */
        _afterRequestData: function() {
            this._filters.enableFilter('view');
            this._export && this._export.enable();
            this._chart.hideLoading();
        },

        /**
         * Вызывается после получения данных с сервера
         * @param {Object} params - параметры
         * @param {String} params.view - тип графика
         * @param {Object} data - данные с сервера
         * @param {String} data.type - тип данных (column|slices)
         * @param {Array} data.series
         * @private
         */
        _onGotChartData: function(params, data) {
            var hasChartData = data.series.every(function(series) {
                return !u._.isEmpty(series.data);
            });

            if (hasChartData) {
                this.buildChart(params, data);
                this._controlChartView(params, data);
            } else {
                this._chart.showErrorMessage(iget2('b-charts-manager', 'net-dannyh-dlya-otrisovki', 'Нет данных для отрисовки графика.'));
            }
        },

        /**
         * Выводит сообщение об ошибке во время запроса
         * @private
         */
        _showRequestError: function() {
            BEM.blocks['b-confirm'].open({
                type: 'alert',
                message: [
                    iget2(
                        'b-charts-manager',
                        'proizoshla-neizvestnaya-oshibka-poprobuyte',
                        'Произошла неизвестная ошибка. Попробуйте перегрузить страницу или обратитесь к сервису позже.'
                    ),
                    iget2('b-charts-manager', 'izvinite-za-dostavlennoe-neudobstvo', 'Извините за доставленное неудобство.')
                ].join('<br/>')
            });

            this._chart.showErrorMessage(iget2(
                'b-charts-manager',
                'poprobuyte-peregruzit-stranicu-ili',
                'Попробуйте перегрузить страницу или обратитесь к сервису позже.'
            ));
        },

        /**
         * Выводит сообщение об ошибке во время запроса
         * @private
         */
        _showTimeoutError: function() {
            BEM.blocks['b-confirm'].open({
                type: 'alert',
                message: [
                    iget2('b-charts-manager', 'my-ne-smogli-postroit', 'Мы не смогли построить график из-за большого объёма данных.'),
                    '1. ' + iget2('b-charts-manager', 'poprobuyte-umenshit-period-otcheta', 'Попробуйте уменьшить период отчета.'),
                    '2. ' + iget2('b-charts-manager', 'poprobuyte-vybrat-drugoy-srez', 'Попробуйте выбрать другой срез.')
                ].join('<br/>')
            });

            this._chart.showErrorMessage(iget2('b-charts-manager', 'poprobuyte-vybrat-drugoy-stolbec', 'Попробуйте выбрать другой столбец или срез, изменить период.'));
        },

        /**
         * Переключает или выключает выбор типа графика, при необходимости
         * @param {Object} params - параметры
         * @param {String} params.view - тип графика
         * @param {Object} data - данные с сервера
         * @param {Array} data.series
         * @private
         */
        _controlChartView: function(params, data) {
            var ratio = 1;

            data.type == 'columns' && (ratio = data.series.length);
            if (data.series[0].data.length * ratio > CHART_LINE_POINTS_LIMIT) {
                this._filters.setValue('view', 'line');
                this._filters.getFilter('view').disableOptions(['column']);
            } else {
                this._filters.getFilter('view').enableOptions(['column']);
            }
        },

        /**
         * Возвращает блок графика
         * @return {null|*|BEM}
         * @private
         */
        _getChart: function() {
            return this._chart;
        },

        /**
         * Получает данные с фильтров
         * @return {Object} - преобразованные данные (название фильтра:значение)
         * @private
         */
        _getFiltersData: function() {
            return this._filters ?
                this._provideFiltersData(this._filters.getValue()) :
                {};
        },

        /**
         * @typedef {Object} FilterData
         * @property {String} filter - название фильтра
         * @property {Array|Object} data - значения фильтра
         */

        /**
         * Преобразуюет данные из фильтров в нужный формат
         * @param {Array<FilterData>} filtersRawData - данные из блока с фильтрами
         * @return {Object} - преобразованные данные (название фильтра:значение)
         * @private
         */
        _provideFiltersData: function(filtersRawData) {
            var filters = {};

            filtersRawData.forEach(function(item) {
                filters[item.filter] = item.type === 'select' ?
                    item.data.map(u._.property('value')) :
                    item.data;
            });

            return filters;
        },

        /**
         * Экспорт графика
         * @param {Object} params
         * @param {Boolean} params.print - напечатать ( type игнорируется)
         * @param {String} params.type - MIME тип
         * @private
         */
        _exportChart: function(params) {
            var chartBlock = this._getChart(),
                reportData = this.params.reportData;

            chartBlock && chartBlock.exportChart({
                print: params.print,
                type: params.type,
                filename: u._.compact([
                    reportData.date.from,
                    reportData.date.to,
                    reportData.statType
                ]).join('-')
            });
        },

        /**
         * Переводит и группирует данные для графика
         * @param {Array<Series>} series - данные графика
         * @param {'column'|'slices'} type - тип данных
         * @returns {Array<Series>}
         * @private
         */
        _formatSeries: function(series, type) {
            series = u['b-charts-manager'].applyGroup(series, { takeFrom: 'column', writeTo: 'unit' });
            series = u['b-chart'].translate(series, { currency: u.currencies.getName(this.params.clientCurrency) });

            if (type === 'slices') {
                series = u['b-chart'].simplify(series);
            } else if (type === 'columns') {
                series = u['b-chart'].setYAxis(series, u['b-charts-manager'].groups);
            }

            return series;
        },

        /**
         * Строит график с помощью b-chart
         * @param {ChartOptions} [params] - параметры
         * @param {Object} [data] - данные с сервера
         * @param {'column'|'slices'} [data.type] - тип данных
         * @param {Array<Series>} [data.series] - данные графика
         */
        buildChart: function(params, data) {
            params || (params = {});
            data || (data = {});

            params.title = iget2('b-charts-manager', 'grafiki-statistiki', 'Графики статистики');

            var type = data.type || 'columns',
                stacking = type === 'slices' ? 'yes' : '',
                view = params.view || 'column';

            if (view === 'line' && stacking === 'yes') {
                view = 'area';
            }

            var chartElem = BEM.DOM.replace(this.findElem('chart'), BEMHTML.apply({
                block: 'b-chart',
                mods: {
                    view: view,
                    stacking: stacking
                },
                mix: { block: 'b-charts-manager', elem: 'chart' },
                js: {
                    series: this._formatSeries(data.series || [], type),
                    options: params
                }
            }));

            this._chart = this.findBlockOn(chartElem, 'b-chart');

            return this;
        },

        destruct: function() {
            this._filters &&
                this._filters.un('change', this._onFilterChange);

            this._switcher &&
                this._switcher.un('click', this._onSwitcherClick);

            return this.__base.apply(this, arguments);
        }

    }, {

        live: function() {

            this.liveInitOnBlockInsideEvent('init', 'button');

        }

    });
})();
