/**
 * @param {Object} params
 * @param {String} params.ulogin - логин пользователя
 * @param {Number} [params.retCondId] - id условия (для редактирования)
 * @param {Number} [params.retId] - id свзки группа + условие (для редактирования)
 * @implements IErrorViewInterface
 */
BEM.DOM.decl({ block: 'b-conditions-display' }, {

    /**
     * Модель dm-retargeting
     */
    _model: null,
    /**
     * Флаг о готовности
     */
    _isReady: null,

    onSetMod: {

        js: function() {
            this._subscriptionManager = BEM.create('i-subscription-manager');
            this._errors = this.elem('errors');

            this._initPromise = this._getData(this.params.ulogin, this.params.retCondId)
                .then(this._onDataLoad.bind(this))
                .then(this._createModel.bind(this))
                .then(this._createForm.bind(this))
                .then(function() {
                    this._isReady = true;
                    this.trigger('ready');
                }.bind(this))
                .catch(function(e) {
                    this.setMod('loading', 'no');
                    this.showRequestError(iget2(
                        'b-conditions-display',
                        'nevozmozhno-poluchit-spisok-dostupnyh',
                        'Невозможно получить список доступных сегментов. Повторите попытку через несколько минут.'
                    ));
                }.bind(this));
        }

    },

    /**
     * Отобразить ошибку запроса внутри блока
     * @param {String} text - текст ошибки
     */
    showRequestError: function(text) {
        this.setMod('error', 'yes');

        BEM.DOM.update(this.elem('error-wrapper'), BEMHTML.apply({
            block: 'b-audience-selection',
            elem: 'error',
            text: text
        }));
    },

    /**
     * Спрятать ошибку запроса внутри блока
     */
    hideRequestError: function() {
        this.delMod('error');
    },

    /**
     * Возвращает значение флага
     * @returns {boolean}
     */
    isReady: function() {
        return !!this._isReady;
    },

    /**
     * Получение сегментов из web-api
     * @param {String} ulogin - логин пользователя
     * @return {*|Promise}
     */
    _getCryptaSegments: function(ulogin) {

        return BEM.blocks['i-crypta-segments-data'].get(ulogin);
    },

    /**
     * Запрашивает данные редактируемого "условия подбора аудитории"
     * @param {Number} [conditionId] - id условия подбора
     * @returns {RetCondition}
     * @private
     */
    _getCondition: function(conditionId) {

        return conditionId ?
            u['retargeting-dataprovider'].getCondition(conditionId) :
            { id: null, name: null };
    },

    /**
     * Получение данных для работы с формой
     * @param {String} ulogin - логин пользователя
     * @param {Number} [conditionId] - id условия подбора
     * @returns {Promise}
     * @private
     */
    _getData: function(ulogin, conditionId) {
        this.setMod('loading', 'yes');

        return Promise.all([
            this._getCryptaSegments(ulogin), // в этом блоке не нужны, но загружаем и кешируем сразу
            this._getCondition(conditionId)
        ]);
    },

    /**
     * Достает из серверного ответа сегменты и условие
     *
     * @param {[Array<CryptaSegmentServer>, RetCondition]} responses
     *
     * @returns {{
     *     segments: {Array<CryptaSegmentServer>}
     *     condition: {RetCondition}
     * }}
     */
    _onDataLoad: function(responses) {

        this.setMod('loading', 'no');

        return {
            condition: responses[1]
        };
    },

    /**
     * Создает и возвращает модель dm-retargeting
     *
     * @param {Object} data
     * @param {Array<CryptaSegmentServer>} data.segments - список сегментов
     * @param {RetCondition} data.condition - данные условия
     * @return {'BEM.MODEL<dm-retargeting>'}
     */
    _createModel: function(data) {
        var condition = data.condition,
            groups = {};

        if (condition.conditions) {
            groups = this._splitConditionGroups(condition.conditions);
        }

        this._model = BEM.MODEL.create(
            { name: 'dm-retargeting', id: condition.id },
            {
                ret_cond_id: condition.id,
                ret_id: this.params.retId,
                type: 'interests',  // пока только один тип
                condition_name: condition.name,
                'social-demo': groups['social_demo'],
                family: groups.family,
                interests: groups.interests,
                metrika: groups.metrika,
                behaviors: groups.behaviors,
                'audio-genres': groups['audio_genres']
            }
        );

        // фиксировать только сохраненные
        this._model.fix();

        this._subscriptionManager.on(this._model, 'change', function(e, data) {
            this.trigger('conditions:change');
        }, this);

        this.trigger('conditions:add', { model: this._model });

        return this._model;
    },

    /**
     * Разделяет группы по типам
     *
     * @param {Array<RetConditionGroup>} groups
     * @returns {{
     *     crypta: Array<RetConditionGroup>,
     *     metrika: Array<RetConditionGroup>
     *     }}
     * @private
     */
    _splitConditionGroups: function(groups) {
        var result = {},
            METRIKA_GOAL_TYPES = u['retargeting-goal-dto'].getMetrikaServerTypes();

        groups.forEach(function(group) {
            var type = group.goals[0].type;

            if (METRIKA_GOAL_TYPES.indexOf(type) !== -1) {
                result.metrika || (result.metrika = []);
                result.metrika.push(group);
            } else {
                result[type] || (result[type] = []);
                result[type].push(group);
            }
        });

        return result;
    },

    /**
     * Создает форму создания/редактирования аудитории
     *
     * @param {'BEM.MODEL<dm-retargeting>'} model
     */
    _createForm: function(model) {
        BEM.DOM.update(this.findElem('form'), BEMHTML.apply({
            block: 'b-audience-selection',
            js: {
                modelName: model.name,
                modelId: model.id
            }
        }));
        this._form = this.blockInside('b-audience-selection');
    },

    destruct: function() {
        this._subscriptionManager.dispose();
        this._model && this._model.destruct();
        this.__base.apply(this, arguments);
    }
});
